/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "i2c_state.h"

void i2c_init()
{
#if (GS32_PART_NUM != 0x35)
	GPIO_setPinConfig(I2C_SDA_PINMUX);
    GPIO_setPadConfig(I2C_SDA_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2C_SDA_PIN, GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(I2C_SCL_PINMUX);
    GPIO_setPadConfig(I2C_SCL_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2C_SCL_PIN, GPIO_QUAL_ASYNC);

#else

    I2C_SDA_PIN_MUX;
    I2C_SCL_PIN_MUX;

#endif

    I2C_disableModule(I2C_BASE);

    I2C_initController(I2C_BASE, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
    I2C_setAddressMode(I2C_BASE, I2C_ADDR_MODE_7BITS);
    I2C_setBitCount(I2C_BASE, I2C_BITCOUNT_8);
    I2C_setEmulationMode(I2C_BASE, I2C_EMULATION_FREE_RUN);

    I2C_enableModule(I2C_BASE);
}

uint8_t i2c_read_data(uint8_t dev_addr,uint16_t reg_addr,uint8_t *pdata,uint16_t datalen)
{
	I2C_Stage state = I2C_READY;
	uint16_t directionFlag = 0;
	uint16_t status = 0;
	uint16_t stopFlag = 0;
	uint16_t cnt = 0;

	while(!(stopFlag)){
		switch(state){
			case I2C_READY:

				if(directionFlag == 0){
					if(I2C_isBusBusy(I2C_BASE))
						return 1;
					I2C_setConfig(I2C_BASE, I2C_CONTROLLER_SEND_MODE);
					I2C_setDataCount(I2C_BASE, 1);
				}else if(directionFlag == 1){
					I2C_setConfig(I2C_BASE, I2C_CONTROLLER_RECEIVE_MODE);
					I2C_setDataCount(I2C_BASE, datalen);
				}
				I2C_setTargetAddress(I2C_BASE, dev_addr);

				state = I2C_START;
				break;

			case I2C_START:

				I2C_sendStartCondition(I2C_BASE);
				if(directionFlag == 0){
					I2C_putData(I2C_BASE,reg_addr);
					directionFlag = 1;
					state = I2C_READY;
					do{
						status = I2C_getStatus(I2C_BASE);
					}while(!(status & I2C_STS_TX_DATA_RDY));
				}else if(directionFlag == 1){
					state = I2C_TRANSFER;
				}

				break;

			case I2C_TRANSFER:

				for(cnt = 0;cnt<datalen;cnt++){
					do{
						status = I2C_getStatus(I2C_BASE);
					}while(!(status & I2C_STS_RX_DATA_RDY));
					pdata[cnt] = I2C_getData(I2C_BASE);
				}

				state = I2C_STOP;

				break;

			case I2C_STOP:

				I2C_sendStopCondition(I2C_BASE);
				stopFlag = 1;

				break;
		}
	}

	return 0;
}

uint8_t i2c_write_data(uint8_t dev_addr,uint16_t reg_addr,uint8_t *pdata,uint16_t datalen)
{
	volatile I2C_Stage state = I2C_READY;
	uint16_t status = 0;
	volatile uint16_t stopFlag = 0;
	volatile uint16_t cnt = 0;

	while(!(stopFlag)){
		switch(state){
			case I2C_READY:

				if(I2C_isBusBusy(I2C_BASE)){
					return 1;
				}

				I2C_setConfig(I2C_BASE, I2C_CONTROLLER_SEND_MODE);
				I2C_setTargetAddress(I2C_BASE, dev_addr);
				I2C_setDataCount(I2C_BASE, datalen+1);

				state = I2C_START;
				break;

			case I2C_START:

				I2C_sendStartCondition(I2C_BASE);
				I2C_putData(I2C_BASE,reg_addr);

				state = I2C_TRANSFER;

				break;

			case I2C_TRANSFER:

				for(cnt = 0;cnt<datalen;cnt++){
					do{
						status = I2C_getStatus(I2C_BASE);
					}while(!(status & I2C_STS_TX_DATA_RDY));
					I2C_putData(I2C_BASE,pdata[cnt]);
				}

				state = I2C_STOP;

				break;

			case I2C_STOP:

				I2C_sendStopCondition(I2C_BASE);
				stopFlag = 1;

				break;
		}
	}

	return 0;
}
