/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "device.h"
#include "lin_transfer.h"

static void lin_module_uart_mode_init(void)
{
	LIN_initParam_t LINinitParam = {0};
	LIN_LinModeInitParam_t LINModeInitParam = {0};

	LINinitParam.bit_order = LIN_TRANSFER_LSB;
	LINinitParam.char_length = LIN_CHAR_LENGTH_8BITS;
	LINinitParam.opmode = LIN_UART_MODE;
	LINinitParam.chl_mode = LIN_CHAL_LOCAL_LOOPBACK_MODE;
	LINinitParam.stopbit = LIN_STOP_BIT_1;
	LINinitParam.parity = LIN_PARITY_NONE;

	/* Initialize LIN module. */
	if (LIN_initModule(LIN_BASE, &LINinitParam))
		while(1);

	/* Set baud rate. */
#if (IS_GS32F00xx(0x30))

	uint32_t lin_clk_div = SysCtl_getLinClkDiv();
	lin_clk_div *= 2;

	if (LIN_setBaudRate(LIN_BASE,(DEVICE_PLLCLK_FREQ / lin_clk_div), 9600,
						LIN_OP_ASYNC_MODE, LIN_OVERSAMP_16))
		while (1);
#else
	if (LIN_setBaudRate(LIN_BASE, DEVICE_APBCLK_FREQ, 9600,
						LIN_OP_ASYNC_MODE, LIN_OVERSAMP_16))
		while (1);
#endif

	/* Enable receiver. */
	if (LIN_enableReceiver(LIN_BASE, true))
		while (1);

	/* Enable transmitter. */
	if (LIN_enableTransmitter(LIN_BASE, true))
		while (1);
}

static void lin_clk_init(void)
{
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_LIN);

#if (IS_GS32F00xx(0x30))

	int Lin_defaultClkDiv = 2;

	SysCtl_setLinClkDiv(Lin_defaultClkDiv);

#endif

	SysCtl_resetLin();
}

void lin_init(void)
{
	/* Set PIN mux */
#if(GS32_PART_NUM == 0x35)

	LIN_TX_PIN_MUX;
	LIN_RX_PIN_MUX;

#else

	GPIO_setPinConfig(LIN_TX_PIN);
	GPIO_setPinConfig(LIN_RX_PIN);

#endif

	/* Initialize LIN clock */
	lin_clk_init();

	/* Initialize LIN to UART mode. */
	lin_module_uart_mode_init();

	if (LIN_InterruptDisable(LIN_BASE, 0xffffffffU))
		while (1);

}

void lin_run(void)
{
	static uint8_t uart_txdata = 0;

	uint16_t uart_rxdata = 0;
	LIN_UartPutCharData(LIN_BASE, uart_txdata);
	DELAY_US(5000);
	LIN_UartGetCharData(LIN_BASE, &uart_rxdata);
	if(uart_rxdata != uart_txdata)
		while(1);
	uart_txdata++;
}

