/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "mcan_transfer.h"
#include "mcan.h"
#include "device.h"

#define M_CAN_MSG_MAX_LENGTH			MCAN_ELEM_SIZE_8BYTES
#define M_CAN_MSG_BUFFER_SIZE			(MCAN_getMsgObjSize(M_CAN_MSG_MAX_LENGTH) + 8U)

#define M_CAN_RAM_BASE_ADDR				0U
#define M_CAN_STANDARD_FILTER_BASE_ADDR	M_CAN_RAM_BASE_ADDR
#define M_CAN_STANDARD_FILTER_NUM		(32U)
#define M_CAN_STANDARD_FILTER_SIZE		(4U)

/* Note: The size of all memory regions cannot exceed 4096 bytes. */
#define M_CAN_EXTENDED_FILTER_BASE_ADDR	(M_CAN_STANDARD_FILTER_BASE_ADDR + \
										(M_CAN_STANDARD_FILTER_NUM * \
										 M_CAN_STANDARD_FILTER_SIZE))
#define M_CAN_EXTENDED_FILTER_NUM		(32U)
#define M_CAN_EXTENDED_FILTER_SIZE		(8U)

#define M_CAN_RXFIFO0_BASE_ADDR			(M_CAN_EXTENDED_FILTER_BASE_ADDR + \
										(M_CAN_EXTENDED_FILTER_NUM * \
										 M_CAN_EXTENDED_FILTER_SIZE))
#define M_CAN_RXFIFO0_NUM				(0U)
#define M_CAN_RXFIFO0_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXFIFO1_BASE_ADDR			(M_CAN_RXFIFO0_BASE_ADDR + \
										(M_CAN_RXFIFO0_NUM * \
										 M_CAN_RXFIFO0_SIZE))
#define M_CAN_RXFIFO1_NUM				(0U)
#define M_CAN_RXFIFO1_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXBUFFER_BASE_ADDR		(M_CAN_RXFIFO1_BASE_ADDR + \
										(M_CAN_RXFIFO1_NUM * \
										 M_CAN_RXFIFO1_SIZE))
#define M_CAN_RXBUFFER_NUM				(32U)
#define M_CAN_RXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXEVENTFIFO_BASE_ADDR		(M_CAN_RXBUFFER_BASE_ADDR + \
										(M_CAN_RXBUFFER_NUM * \
										 M_CAN_RXBUFFER_SIZE))
#define M_CAN_TXEVENTFIFO_NUM			(0U)
#define M_CAN_TXEVENTFIFO_SIZE			(M_CAN_TX_EVENT_FIFO_SIZE)

#define M_CAN_TXBUFFER_BASE_ADDR		(M_CAN_TXEVENTFIFO_BASE_ADDR + \
										(M_CAN_TXEVENTFIFO_NUM * \
										 M_CAN_TXEVENTFIFO_SIZE))
#define M_CAN_TXBUFFER_NUM				(32U)
#define M_CAN_TXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXFIFOQUEUE_BASE_ADDR		(M_CAN_TXBUFFER_BASE_ADDR + \
										(M_CAN_TXBUFFER_NUM * \
										 M_CAN_TXBUFFER_SIZE))
#define M_CAN_TXFIFOQUEUE_NUM			(0U)
#define M_CAN_TXFIFOQUEUE_SIZE			(M_CAN_MSG_BUFFER_SIZE)

MCAN_RxMessage_t RxMailbox0;
MCAN_RxMessage_t RxMailbox1;
MCAN_RxMessage_t RxMailbox2;

MCAN_TxMessage_t TxMailbox3 = {0};
MCAN_TxMessage_t TxMailbox4 = {0};

void mcan_init_mailbox(void)
{
	MCAN_StdMsgIDFilterElement_t std_filter = {0};
	MCAN_ExtMsgIDFilterElement_t ext_filter = {0};

	/* Add a new standard ID filter(Use Rx buffer 0). */
	/* Configuration the mailbox 0 ID to standard frame ID */
	//ECanaMboxes.MBOX0.MSGID.bit.IDE = 0;
	std_filter.sfid1 = 0x123;
	/* RX buffer number 0 */
	std_filter.sfid2 = 0;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	/* Enable filter 0(enable mailbox 0). */
	//ECanaRegs.CANME.bit.ME0 = 1;
	/* Enable mailbox 0 filter. */
	//ECanaMboxes.MBOX0.MSGID.bit.AME = 1;
	if (!MCAN_addStdMsgIDFilter(CAN_BASE, 0, &std_filter))
		while (1);

	/**
	 * TI receive ID filter 0(mailbox 0).
	 *
	 * LAM0 = 1: Accept a 0 or a 1 (don't care) for the corresponding bit of the received identifier.
	 * LAM0 = 0: Received identifier bit value must match the corresponding identifier bit of the MSGID register.
	 * ECanaLAMRegs.LAM0.bit.LAM_L = 0x0;
	 * ECanaLAMRegs.LAM0.bit.LAM_H = 0x0;
	 *
	 * LAMI == 1: Standard and extended frames can be received.
	 * LAMI == 0: The identifier extension bit stored in the mailbox determines which messages shall be received.
	 * ECanaLAMRegs.LAM0.bit.LAMI = 0x0;
	 */

	/* Add a new standard ID filter(Use Rx buffer 1). */
	/* Configuration the mailbox 1 ID to standard frame ID */
	//ECanaMboxes.MBOX1.MSGID.bit.IDE = 0;
	std_filter.sfid1 = 0x456;
	/* RX buffer number 1 */
	std_filter.sfid2 = 1;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	/* Enable filter 1(enable mailbox 1). */
	// ECanaRegs.CANME.bit.ME1 = 1;
	if (!MCAN_addStdMsgIDFilter(CAN_BASE, 1, &std_filter))
		while (1);

	/* Add a new extended ID filter(Use Rx buffer 2). */
	/* Configuration the mailbox 2 ID to Extended frame ID */
	//ECanaMboxes.MBOX2.MSGID.bit.IDE = 1;
	ext_filter.efid1 = 0x789;
	/* RX buffer number 2 */
	ext_filter.efid2 = 2;
	ext_filter.esync = MCAN_EXTIDF_SYNC_MSG_DISABLE;
	ext_filter.efec = MCAN_EXTIDF_ELE_STORE_IN_RXB_OR_DMSG;
	ext_filter.eft = MCAN_EXTIDF_RANGE_FROM_EFID1_TO_EFID2;
	/* Enable filter 2(enable mailbox 2). */
	// ECanaRegs.CANME.bit.ME2 = 1;
	if (!MCAN_addExtMsgIDFilter(CAN_BASE, 2, &ext_filter))
		while (1);

	/* Initialize Tx mailbox 0. */
	//ECanaMboxes.MBOX3.MSGCTRL.bit.DLC = 8;
	TxMailbox3.dlc = MCAN_DATA_LENGTH_8;
	/* Standard frame ID */
	//ECanaMboxes.MBOX3.MSGID.bit.IDE = 0;
	TxMailbox3.xtd = 0;
	/* Disable remote frame. */
	//ECanaMboxes.MBOX3.MSGCTRL.bit.RTR = 0;
	TxMailbox3.rtr = 0;
	/* Set frame ID. */
	//ECanaMboxes.MBOX3.MSGID.bit.STDMSGID = 0x1;
	TxMailbox3.id_std = 0x3ff;


	/* Initialize Tx mailbox 1. */
	//ECanaMboxes.MBOX4.MSGCTRL.bit.DLC = 8;
	TxMailbox4.dlc = MCAN_DATA_LENGTH_8;
	/* Extended frame ID */
	//ECanaMboxes.MBOX4.MSGID.bit.IDE = 0;
	TxMailbox4.xtd = 1;
	/* Disable remote frame. */
	//ECanaMboxes.MBOX3.MSGCTRL.bit.RTR = 0;
	TxMailbox4.rtr = 0;
	/* Set frame ID. */
	//ECanaMboxes.MBOX3.MSGID.bit.EXTMSGID_L = 0x155555 & 0xFFFF; //(0-15 bit)
	//ECanaMboxes.MBOX3.MSGID.bit.EXTMSGID_H = (0x155555 & 0x30000) >> 16; //(16-17 bit)
	//ECanaMboxes.MBOX3.MSGID.bit.STDMSGID = (0x155555 & 0x1FFC0000) >> 18; //(18 - 28)
	TxMailbox4.id_ext = 0x155555;
}

void mcan_config(void)
{
	MCAN_InitParams_t init_param = {0};
	MCAN_ConfigParams_t configParams = {0};
	MCAN_MsgRAMConfigParams_t RAMConfig = {0};
	MCAN_BitTimingParams_t configBitrate;

	/* CAN mode configuration. */
	init_param.fdMode = false;
	init_param.fdFormat = MCAN_FD_ISO_11898_1;
	init_param.brsEnable = false;
	init_param.txpEnable = true;
	init_param.efbi = false;
	init_param.pxhddisable = false;
	/* Enable the auto retransmission. */
	init_param.darEnable = true;
	init_param.wkupReqEnable = false;
	init_param.autoWkupEnable = false;
	init_param.emulationEnable = true;
	init_param.wdcPreload = false;
	init_param.wmMarker = MCAN_WMM_8BIT_MODE;
	/* Configure the TDC in CANFD mode. */
	init_param.tdcEnable = true;
	init_param.tdcConfig.tdcf = 0U;
	init_param.tdcConfig.tdco = 0U;

	/* Set CAN extended feature. */
	configParams.asmEnable = false;
	configParams.monEnable = false;
	configParams.timeoutCntEnable = false;
	configParams.timeoutPreload = 1;
	configParams.timeoutSelect = MCAN_TIMEOUT_SELECT_CONT;

	configParams.tsClock = MCAN_INTERNAL_TIMESTAMP;
	configParams.tsSelect = MCAN_TSCNTVAL_ALWAYS0;
	configParams.tsPrescalar = 1;

	/* Set global configuration. */
	configParams.filterConfig.rrfs = true;
	configParams.filterConfig.rrfe = true;
	configParams.filterConfig.anfs = MCAN_NON_MATCH_ACCEPT_IN_FIFO1;
	configParams.filterConfig.anfe = MCAN_NON_MATCH_ACCEPT_IN_FIFO1;

	/* Set standard ID filter. */
	RAMConfig.lss = M_CAN_STANDARD_FILTER_NUM;
	RAMConfig.flssa = M_CAN_STANDARD_FILTER_BASE_ADDR;

	/* Set extended ID filter. */
	RAMConfig.lse = M_CAN_EXTENDED_FILTER_NUM;
	RAMConfig.flesa = M_CAN_EXTENDED_FILTER_BASE_ADDR;

	/* Set Rx fifo 0. */
	RAMConfig.rxFIFO0OpMode = MCAN_RXFIFO_OP_MODE_BLOCKING;
	RAMConfig.rxFIFO0startAddr = M_CAN_RXFIFO0_BASE_ADDR;
	RAMConfig.rxFIFO0size = M_CAN_RXFIFO0_NUM;
	RAMConfig.rxFIFO0waterMark = 0;
	RAMConfig.rxFIFO0ElemSize  = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx fifo 1. */
	RAMConfig.rxFIFO1OpMode = MCAN_RXFIFO_OP_MODE_BLOCKING;
	RAMConfig.rxFIFO1startAddr = M_CAN_RXFIFO1_BASE_ADDR;
	RAMConfig.rxFIFO1size = M_CAN_RXFIFO1_NUM;
	RAMConfig.rxFIFO1waterMark = 0;
	RAMConfig.rxFIFO1ElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx buffer. */
	RAMConfig.rxBufStartAddr = M_CAN_RXBUFFER_BASE_ADDR;
	RAMConfig.rxBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx buffer */
	RAMConfig.txBufMode = MCAN_TXBUF_OP_IN_FIFO_MODE;
	RAMConfig.txStartAddr = M_CAN_TXBUFFER_BASE_ADDR;
	RAMConfig.txFIFOSize = M_CAN_TXFIFOQUEUE_NUM;
	RAMConfig.txBufNum = M_CAN_TXBUFFER_NUM;
	RAMConfig.txBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx event fifo. */
	RAMConfig.txEventFIFOStartAddr = M_CAN_TXEVENTFIFO_BASE_ADDR;
	RAMConfig.txEventFIFOSize = M_CAN_TXEVENTFIFO_NUM;
	RAMConfig.txEventFIFOWaterMark = 1;

	/* Input clock source 40Mhz, normal phase bitrate 1Mbps, 80% sample point. */
	configBitrate.nomRatePrescalar = 0;
	configBitrate.nomTimeSeg1 = 30;
	configBitrate.nomTimeSeg2 = 7;
	configBitrate.nomSynchJumpWidth = 2;

	/* Reset the MCAN module. */
	while (MCAN_isInReset(CAN_BASE));

	/* Check if the MCAN RAM is ready. */
	while (!MCAN_isMemInitDone(CAN_BASE));

	/* Set the MCAN mode to init mode. */
	if (!MCAN_setOpMode(CAN_BASE, MCAN_OPERATION_SW_INIT_MODE))
		while (1);

	/* Initialize the MCAN. */
	if (!MCAN_init(CAN_BASE, &init_param))
		while (1);

	if (!MCAN_config(CAN_BASE, &configParams))
		while (1);

	if (!MCAN_setBitTime(CAN_BASE, &configBitrate))
		while (1);

	if (!MCAN_msgRAMConfig(CAN_BASE, &RAMConfig))
		while (1);

	/* The received extension frame will be subjected to AND operation with Ext_mask. */
	if (!MCAN_setExtIDAndMask(CAN_BASE, 0x1fffffff))
		while (1);

	/* Enable/Disable Loopback mode. */
	if (!MCAN_lpbkModeEnable(CAN_BASE, MCAN_LPBK_MODE_EXTERNAL, false))
		while (1);

	/* Enable MCAN. */
	if (!MCAN_setOpMode(CAN_BASE, MCAN_OPERATION_NORMAL_MODE))
		while (1);

	 /* Configuration the external timestamp clock source. */
	 if (!MCAN_extTSCounterConfig(CAN_BASE, 0xffff))
	 	while (1);

	 /* Enable/Disable external timestamp clock source. */
	 if (!MCAN_extTSCounterEnable(CAN_BASE, true))
	 	while (1);

	 /* Enable/Disable external timestamp overflow interrupt. */
	 if (!MCAN_extTSEnableIntr(CAN_BASE, false))
	 	while (1);

	 /* Enable Tx Buffer 3 interrupt. */
	 if (!MCAN_txBufTransIntrEnable(CAN_BASE, 3, true))
		 while (1);

	 /* Enable Tx Buffer 4 interrupt. */
	 if (!MCAN_txBufTransIntrEnable(CAN_BASE, 4, true))
		 while (1);

	/* Select MCAN interrupt route to interrupt line 0 or 1. */
	if (!MCAN_selectIntrLine(CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF,
									   MCAN_INTERRUPT_LINE_0))
		while (1);

	/* Select MCAN interrupt route to interrupt line 0 or 1. */
	if (!MCAN_selectIntrLine(CAN_BASE, MCAN_INT_SRC_TRANSMISSION_COMPLETE,
									   MCAN_INTERRUPT_LINE_1))
		while (1);

	/* Enable MCAN interrupt. */
	if (!MCAN_enableIntr(CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF |
								   MCAN_INT_SRC_TRANSMISSION_COMPLETE,
								   true))
		while (1);

	if (!MCAN_enableIntrLine(CAN_BASE, MCAN_INTERRUPT_LINE_0, true))
		while (1);

	if (!MCAN_enableIntrLine(CAN_BASE, MCAN_INTERRUPT_LINE_1, true))
		while (1);
}

void mcan_irq_rx_handler(void)
{
	uint32_t interrupt_status;
	MCAN_RxNewDataStatus_t new_dat;

	interrupt_status = MCAN_getIntrStatus(CAN_BASE);

	if (interrupt_status & MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF) {
		MCAN_getNewDataStatus(CAN_BASE, &new_dat);

		/* Received a new message from buffer 0 */
		// if (ECanaRegs.CANRMP.bit.RMP0 == 1)
		if (new_dat.statusLow & (1U << 0)) {
			//ECanaMboxes.MOTO0
			MCAN_readMsgRam(CAN_BASE, MCAN_MEM_TYPE_BUF, 0, MCAN_RX_FIFO_NUM_0, &RxMailbox0);
		}

		/* Received a new message from buffer 1 */
		// if (ECanaRegs.CANRMP.bit.RMP1 == 1)
		if (new_dat.statusLow & (1U << 1)) {
			//ECanaMboxes.MOTO1
			MCAN_readMsgRam(CAN_BASE, MCAN_MEM_TYPE_BUF, 1, MCAN_RX_FIFO_NUM_0, &RxMailbox1);
		}

		/* Received a new message from buffer 2 */
		// if (ECanaRegs.CANRMP.bit.RMP2 == 1)
		if (new_dat.statusLow & (1U << 2)) {
			//ECanaMboxes.MOTO2
			MCAN_readMsgRam(CAN_BASE, MCAN_MEM_TYPE_BUF, 2, MCAN_RX_FIFO_NUM_0, &RxMailbox2);
		}

		MCAN_clearNewDataStatus(CAN_BASE, &new_dat);

		MCAN_getNewDataStatus(CAN_BASE, &new_dat);
		if (new_dat.statusHigh == 0 && new_dat.statusLow == 0)
			MCAN_clearIntrStatus(CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF);
	}
}

volatile uint32_t Tx3_req_count = 0;
volatile uint32_t Tx4_req_count = 0;

volatile uint32_t Tx3_irq_count = 0;
volatile uint32_t Tx4_irq_count = 0;

void mcan_irq_tx_handler(void)
{
	uint32_t interrupt_status;
	uint32_t tx_buffer_completed_stat;
	uint32_t tx_buffer_req_flag;

	interrupt_status = MCAN_getIntrStatus(CAN_BASE);

	if (interrupt_status & MCAN_INT_SRC_TRANSMISSION_COMPLETE) {
		tx_buffer_completed_stat = MCAN_getTxBufTransmissionStatus(CAN_BASE);
		tx_buffer_req_flag = MCAN_getTxBufTransmissionFlagStatus(CAN_BASE);

		/**
		 * Tx buffer 3 transmission completed,
		 * MCAN does not need to clear the send completion interrupt.
		 */
		//if (ECanaRegs.CANTA.bit.TA3 == 0x1)
		//	ECanaRegs.CANTA.bit.TA3 == 0x1;
		if ((tx_buffer_completed_stat & (1U << 3)) && (tx_buffer_req_flag & (1U << 3))) {
			Tx3_irq_count++;
			MCAN_clearTxBufTransmissionFlag(CAN_BASE, 3);
		}

		/**
		 * Tx buffer 4 transmission completed,
		 * MCAN does not need to clear the send completion interrupt.
		 */
		//if (ECanaRegs.CANTA.bit.TA3 == 0x1)
		//	ECanaRegs.CANTA.bit.TA3 == 0x1;
		if ((tx_buffer_completed_stat & (1U << 4)) && (tx_buffer_req_flag & (1U << 4))) {
			Tx4_irq_count++;
			MCAN_clearTxBufTransmissionFlag(CAN_BASE, 4);
		}

		MCAN_clearIntrStatus(CAN_BASE, MCAN_INT_SRC_TRANSMISSION_COMPLETE);
	}
}

void mcan_init(void)
{
	/* Select the MCAN clock source. */
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CAN);

	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);
	SysCtl_setCanClkDiv(DEVICE_PLLCLK_FREQ / 40000000U);
	SysCtl_resetCana();
	SysCtl_resetCanb();

#if IS_GS32F00xx(0x30)

#if (GS32_PART_NUM == 0x0039)

	GPIO_setPinConfig(CAN_TX_PIN);
	GPIO_setPinConfig(CAN_RX_PIN);

#elif (GS32_PART_NUM == 0x35)

#if (CAN_BASE == CANB_BASE)

	GpioCtrlRegs.GPAMUX1.bit.GPIO8 = GPIO_8_CANB_TX & GPIO_MUX_MASK;
	GpioCtrlRegs.GPAMUX1.bit.GPIO10 = GPIO_10_CANB_RX & GPIO_MUX_MASK;
	GpioCtrlRegs.GPAPUD.bit.GPIO10 = 0;
	GpioCtrlRegs.GPAPUD.bit.GPIO8 = 0;

#elif (CAN_BASE == CANA_BASE)



#endif

#endif

#endif

	mcan_config();

	/* Initialize mailbox. */
	mcan_init_mailbox();

	Interrupt_register(CAN_IRQ_LINE0, &mcan_irq_rx_handler);
	Interrupt_enable(CAN_IRQ_LINE0);

	Interrupt_register(CAN_IRQ_LINE1, &mcan_irq_tx_handler);
	Interrupt_enable(CAN_IRQ_LINE1);
}

void mcan_process(void)
{
	uint32_t tx_status;

	while (1) {
		tx_status = MCAN_getTxBufReqPend(CAN_BASE);

		/* Waiting for the transmission to end. */
		//if (ECanaRegs.CANTRS.bit.TRS3 == 0x0)
		if ((tx_status & M_CAN_TXBRP_TRP3_M) == 0) {
			//ECanaMboxes.MBOX3.MDL.byte.BYTE0++
			TxMailbox3.data[0]++;
			Tx3_req_count++;
			MCAN_writeMsgRam(CAN_BASE, MCAN_MEM_TYPE_BUF, 3, &TxMailbox3);

			MCAN_setTxBufTransmissionFlag(CAN_BASE, 3);
			//			ECanaRegs.CANTRS.bit.TRS3
			MCAN_txBufAddReq(CAN_BASE, 3U);

			while (MCAN_getTxBufReqPend(CAN_BASE) & M_CAN_TXBRP_TRP3_M);
		}

		//if (ECanaRegs.CANTRS.bit.TRS4 == 0x0)
		if ((tx_status & M_CAN_TXBRP_TRP4_M) == 0) {
			//ECanaMboxes.MBOX4.MDL.byte.BYTE0++
			TxMailbox4.data[0]++;
			Tx4_req_count++;
			MCAN_writeMsgRam(CAN_BASE, MCAN_MEM_TYPE_BUF, 4, &TxMailbox4);

			MCAN_setTxBufTransmissionFlag(CAN_BASE, 4);
			//			ECanaRegs.CANTRS.bit.TRS4
			MCAN_txBufAddReq(CAN_BASE, 4U);

			while (MCAN_getTxBufReqPend(CAN_BASE) & M_CAN_TXBRP_TRP4_M);
		}
	}
}
