/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "sdfm_sync_fiforead.h"
#include "sdfm.h"
#include "epg.h"

#define CLK_GEN_PERIOD   11
#define CLK_GEN_OFFSET   0

int16_t  filter1Result[MAX_SAMPLES];
int16_t  filter2Result[MAX_SAMPLES];
int16_t  filter3Result[MAX_SAMPLES];
int16_t  filter4Result[MAX_SAMPLES];

void EPG_Init(void)
{
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPG);

	/* OUTPUTXBAR Pin Mux */
    GPIO_setPinConfig(GPIO_2_OUTPUTXBAR1);
    GPIO_setDirectionMode(2, GPIO_DIR_MODE_OUT);
    GPIO_setPadConfig(2, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(2, GPIO_QUAL_SYNC);


    /* Connect EPG Output to GPIO through xbar */

	XBAR_setOutputMuxConfig(OUTPUTXBAR_BASE, XBAR_OUTPUT1,
							XBAR_OUT_MUX30_EPGOUT0);

	XBAR_enableOutputMux(OUTPUTXBAR_BASE, XBAR_OUTPUT1, XBAR_MUX30);


    /* Select Clock Generator output for EPG output of 0,1 */

	EPG_selectEPGOutput(EPG_BASE, EPG_OUT0, EPG_OUT_SEL_CLK);


    /* Select Clock Generator Data output for EPG output of 0,1 */

    EPG_selectClkOutput(EPG_BASE, EPG_OUT0, EPG_CLKGEN0_CLKOUT0_DCLK);


    /* Set Clock Generator 0 period to CLK_GEN_PERIOD */

    EPG_setClkGenPeriod(EPG_BASE, EPG_CLKGEN0, CLK_GEN_PERIOD);


    /* Set offset of clock generator output 0 to 0 */

    EPG_setClkGenOffset(EPG_BASE, EPG_CLKGEN0, 0, CLK_GEN_OFFSET);


   /* Enable EPG */

    EPG_enableGlobal(EPG_BASE);
}

void pinmux_init(void)
{
#if SDFM_BASE == SDFM1_BASE
    /* SDFM1 Pin Mux */
    /* SDFM1 SD1_D1_GP16 */
    GPIO_setPinConfig(SD1_D1_PIN_MUX);
    GPIO_setPadConfig(SD1_D1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C1 _GP17 */
    GPIO_setPinConfig(SD1_C1_PIN_MUX);
    GPIO_setPadConfig(SD1_C1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D2 _GP32 */
    GPIO_setPinConfig(SD1_D2_PIN_MUX);
    GPIO_setPadConfig(SD1_D2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C2 _GP33 */
    GPIO_setPinConfig(SD1_C2_PIN_MUX);
    GPIO_setPadConfig(SD1_C2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D3 _GP20 */
    GPIO_setPinConfig(SD1_D3_PIN_MUX);
    GPIO_setPadConfig(SD1_D3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C3 _GP21 */
    GPIO_setPinConfig(SD1_C3_PIN_MUX);
    GPIO_setPadConfig(SD1_C3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D4 _GP22 */
    GPIO_setPinConfig(SD1_D4_PIN_MUX);
    GPIO_setPadConfig(SD1_D4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D4_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C4 _GP23 */
    GPIO_setPinConfig(SD1_C4_PIN_MUX);
    GPIO_setPadConfig(SD1_C4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C4_PIN, GPIO_QUAL_ASYNC);

#elif SDFM_BASE == SDFM2_BASE
    /* SDFM2 Pin Mux */
    /* SDFM2 SD2_D1_GP24 */
    GPIO_setPinConfig(SD2_D1_PIN_MUX);
    GPIO_setPadConfig(SD2_D1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C1 _GP25 */
    GPIO_setPinConfig(SD2_C1_PIN_MUX);
    GPIO_setPadConfig(SD2_C1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D2 _GP26 */
    GPIO_setPinConfig(SD2_D2_PIN_MUX);
    GPIO_setPadConfig(SD2_D2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C2 _GP27 */
    GPIO_setPinConfig(SD2_C2_PIN_MUX);
    GPIO_setPadConfig(SD2_C2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D3 _GP28 */
    GPIO_setPinConfig(SD2_D3_PIN_MUX);
    GPIO_setPadConfig(SD2_D3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C3 _GP29 */
    GPIO_setPinConfig(SD2_C3_PIN_MUX);
    GPIO_setPadConfig(SD2_C3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D4 _GP30 */
    GPIO_setPinConfig(SD2_D4_PIN_MUX);
    GPIO_setPadConfig(SD2_D4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D4_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C4 _GP31 */
    GPIO_setPinConfig(SD2_C4_PIN_MUX);
    GPIO_setPadConfig(SD2_C4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C4_PIN, GPIO_QUAL_ASYNC);
#endif
}

void sdfm_init(uint32_t base)
{
    uint16_t  hlt, llt;

    /* Configure SDFM type to 0 and see if data ack generated SDINT. */
    SysCtl_configureType(0);

    /* Input Control Unit */

    /* Set modulator clock mode: Modulator Clock rate = Modulator data rate */
    SDFM_setupModulatorClock(base, SDFM_FILTER_1,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(base, SDFM_FILTER_2,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(base, SDFM_FILTER_3,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(base, SDFM_FILTER_4,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);

    /* Selects clock source for SDFM channels. */
    SDFM_selectClockSource(base, SDFM_FILTER_1, SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(base, SDFM_FILTER_2, SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(base, SDFM_FILTER_3, SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(base, SDFM_FILTER_4, SDFM_CLK_SOURCE_SD1_CLK);

    /* Comparator Unit */

    /* over and under value threshold settings */
    hlt = 0x7FFF;
    llt = 0x0000;

    /* Configure Comparator Unit's comparator filter type and comparator's
     * OSR value, higher threshold, lower threshold
     */
    SDFM_configComparator(base,
                         (SDFM_FILTER_1 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);
    SDFM_configComparator(base,
                         (SDFM_FILTER_2 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);
    SDFM_configComparator(base,
                         (SDFM_FILTER_3 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);
    SDFM_configComparator(base,
                         (SDFM_FILTER_4 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);

    /* Data Filter Unit */

    /* filter type, OSR value, data shift bit values, FIFO waterline and enable
     * or disable data filter
     */
    SDFM_configDataFilter(base,
                         (SDFM_FILTER_1 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(7)));
    SDFM_configDataFilter(base,
                         (SDFM_FILTER_2 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(7)));
    SDFM_configDataFilter(base,
                         (SDFM_FILTER_3 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(7)));
    SDFM_configDataFilter(base,
                         (SDFM_FILTER_4 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(7)));

    /* FIFO Control Unit*/

    /* Set data ready interrupt source as fifo interrupt */
    SDFM_setDataReadyInterruptSource(base, SDFM_FILTER_1,
                                    SDFM_DATA_READY_SOURCE_FIFO);
    SDFM_setDataReadyInterruptSource(base, SDFM_FILTER_2,
                                    SDFM_DATA_READY_SOURCE_FIFO);
    SDFM_setDataReadyInterruptSource(base, SDFM_FILTER_3,
                                    SDFM_DATA_READY_SOURCE_FIFO);
    SDFM_setDataReadyInterruptSource(base, SDFM_FILTER_4,
                                    SDFM_DATA_READY_SOURCE_FIFO);

    /* set the FIFO interrupt level */
    SDFM_setFIFOInterruptLevel(base, SDFM_FILTER_1, FIFO_INT_NUMBER);
    SDFM_setFIFOInterruptLevel(base, SDFM_FILTER_2, FIFO_INT_NUMBER);
    SDFM_setFIFOInterruptLevel(base, SDFM_FILTER_3, FIFO_INT_NUMBER);
    SDFM_setFIFOInterruptLevel(base, SDFM_FILTER_4, FIFO_INT_NUMBER);

    /* Enable FIFO */
    SDFM_enableFIFOBuffer(base, SDFM_FILTER_1);
    SDFM_enableFIFOBuffer(base, SDFM_FILTER_2);
    SDFM_enableFIFOBuffer(base, SDFM_FILTER_3);
    SDFM_enableFIFOBuffer(base, SDFM_FILTER_4);

    /* Enable FIFO Interrupt */
    SDFM_enableInterrupt(base, SDFM_FILTER_1,
                        (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT));
    SDFM_enableInterrupt(base, SDFM_FILTER_2,
                        (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT));
    SDFM_enableInterrupt(base, SDFM_FILTER_3,
                        (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT));
    SDFM_enableInterrupt(base, SDFM_FILTER_4,
                        (SDFM_FIFO_INTERRUPT | SDFM_FIFO_OVERFLOW_INTERRUPT));

    /* Enable Master filter bit:
     * Unless this bit is set none of the filter modules can be enabled.
     * All the filter modules are synchronized when master filter
     * bit is enabled after individual filter modules are enabled.
     */
    SDFM_enableMainFilter(base);

    /* Disable external reset */
    SDFM_disableExternalReset(base, SDFM_FILTER_1);
    SDFM_disableExternalReset(base, SDFM_FILTER_2);
    SDFM_disableExternalReset(base, SDFM_FILTER_3);
    SDFM_disableExternalReset(base, SDFM_FILTER_4);

    /* Enable interrupts */

    /* Following SDFM interrupts can be enabled / disabled using this function.
     * Enable / disable comparator high threshold
     * Enable / disable comparator low threshold
     * Enable / disable modulator clock failure
     * Enable / disable data filter acknowledge
     */
    SDFM_disableInterrupt(base, SDFM_FILTER_1, SDFM_MODULATOR_FAILURE_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_2, SDFM_MODULATOR_FAILURE_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_3, SDFM_MODULATOR_FAILURE_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_4, SDFM_MODULATOR_FAILURE_INTERRUPT);

    SDFM_disableInterrupt(base, SDFM_FILTER_1,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(base, SDFM_FILTER_2,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(base, SDFM_FILTER_3,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(base, SDFM_FILTER_4,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));

    /* Enable master interrupt so that any of the filter interrupts can trigger
     * by SDFM interrupt to CPU.
     */
    SDFM_enableMainInterrupt(base);
}

/*  - SDFM FIFO ISR - */

void sdfm1_D1_FIFO_ISR(void)
{
    uint16_t i = 0;
    static uint16_t loopCounter1 = 0;

    if (loopCounter1 >= MAX_SAMPLES)
    {
        asm ("NOP");
    }
    else if (SDFM_getFIFOISRStatus(SDFM1_BASE, SDFM_FILTER_1) == 0x01U)
    {
        for (i = 0; i < FIFO_INT_NUMBER; i++)
            filter1Result[loopCounter1++] = (int16_t)(SDFM_getFIFOData(
                SDFM1_BASE, SDFM_FILTER_1) >> 16U);
    }

    /* Clear SDFM flag register (SDIFLG) */
    SDFM_clearInterruptFlag(SDFM1_BASE,
                           (SDFM_MAIN_INTERRUPT_FLAG |
                            SDFM_FILTER_1_FIFO_INTERRUPT_FLAG |
                            SDFM_FILTER_1_FIFO_OVERFLOW_FLAG));
}

void sdfm1_D2_FIFO_ISR(void)
{
    uint16_t i = 0;
    static uint16_t loopCounter2 = 0;

    if (loopCounter2 >= MAX_SAMPLES)
    {
        asm ("NOP");
    }
    else if (SDFM_getFIFOISRStatus(SDFM1_BASE, SDFM_FILTER_2) == 0x01U)
    {
        for (i = 0; i < FIFO_INT_NUMBER; i++)
            filter2Result[loopCounter2++] = (int16_t)(SDFM_getFIFOData(
                SDFM1_BASE, SDFM_FILTER_2) >> 16U);
    }

    /* Clear SDFM flag register (SDIFLG) */
    SDFM_clearInterruptFlag(SDFM1_BASE,
                           (SDFM_MAIN_INTERRUPT_FLAG |
                            SDFM_FILTER_2_FIFO_INTERRUPT_FLAG |
                            SDFM_FILTER_2_FIFO_OVERFLOW_FLAG));
}

void sdfm1_D3_FIFO_ISR(void)
{
    uint16_t i = 0;
    static uint16_t loopCounter3 = 0;

    if (loopCounter3 >= MAX_SAMPLES)
    {
        asm ("NOP");
    }
    else if (SDFM_getFIFOISRStatus(SDFM1_BASE, SDFM_FILTER_3) == 0x01U)
    {
        for (i = 0; i < FIFO_INT_NUMBER; i++)
            filter3Result[loopCounter3++] = (int16_t)(SDFM_getFIFOData(
                SDFM1_BASE, SDFM_FILTER_3) >> 16U);
    }

    /* Clear SDFM flag register (SDIFLG) */
    SDFM_clearInterruptFlag(SDFM1_BASE,
                           (SDFM_MAIN_INTERRUPT_FLAG |
                            SDFM_FILTER_3_FIFO_INTERRUPT_FLAG |
                            SDFM_FILTER_3_FIFO_OVERFLOW_FLAG));
}

void sdfm1_D4_FIFO_ISR(void)
{
    uint16_t i = 0;
    static uint16_t loopCounter4 = 0;

    if (loopCounter4 >= MAX_SAMPLES)
    {
        asm ("NOP");
    }
    else if (SDFM_getFIFOISRStatus(SDFM1_BASE, SDFM_FILTER_4) == 0x01U)
    {
        for (i = 0; i < FIFO_INT_NUMBER; i++)
            filter4Result[loopCounter4++] = (int16_t)(SDFM_getFIFOData(
                SDFM1_BASE, SDFM_FILTER_4) >> 16U);
    }

    /* Clear SDFM flag register (SDIFLG) */
    SDFM_clearInterruptFlag(SDFM1_BASE,
                           (SDFM_MAIN_INTERRUPT_FLAG |
                            SDFM_FILTER_4_FIFO_INTERRUPT_FLAG |
                            SDFM_FILTER_4_FIFO_OVERFLOW_FLAG));
}

void sdfm2_D1_FIFO_ISR(void)
{
    uint16_t i = 0;
    static uint16_t loopCounter5 = 0;

    if (loopCounter5 >= MAX_SAMPLES)
    {
        asm ("NOP");
    }
    else if (SDFM_getFIFOISRStatus(SDFM2_BASE, SDFM_FILTER_1) == 0x01U)
    {
        for (i = 0; i < FIFO_INT_NUMBER; i++)
            filter1Result[loopCounter5++] = (int16_t)(SDFM_getFIFOData(
                SDFM2_BASE, SDFM_FILTER_1) >> 16U);
    }

    /* Clear SDFM flag register (SDIFLG) */
    SDFM_clearInterruptFlag(SDFM2_BASE,
                           (SDFM_MAIN_INTERRUPT_FLAG |
                            SDFM_FILTER_1_FIFO_INTERRUPT_FLAG |
                            SDFM_FILTER_1_FIFO_OVERFLOW_FLAG));
}

void sdfm2_D2_FIFO_ISR(void)
{
    uint16_t i = 0;
    static uint16_t loopCounter6 = 0;

    if (loopCounter6 >= MAX_SAMPLES)
    {
        asm ("NOP");
    }
    else if (SDFM_getFIFOISRStatus(SDFM2_BASE, SDFM_FILTER_2) == 0x01U)
    {
        for (i = 0; i < FIFO_INT_NUMBER; i++)
            filter2Result[loopCounter6++] = (int16_t)(SDFM_getFIFOData(
                SDFM2_BASE, SDFM_FILTER_2) >> 16U);
    }

    /* Clear SDFM flag register (SDIFLG) */
    SDFM_clearInterruptFlag(SDFM2_BASE,
                           (SDFM_MAIN_INTERRUPT_FLAG |
                            SDFM_FILTER_2_FIFO_INTERRUPT_FLAG |
                            SDFM_FILTER_2_FIFO_OVERFLOW_FLAG));
}

void sdfm2_D3_FIFO_ISR(void)
{
    uint16_t i = 0;
    static uint16_t loopCounter7 = 0;

    if (loopCounter7 >= MAX_SAMPLES)
    {
        asm ("NOP");
    }
    else if (SDFM_getFIFOISRStatus(SDFM2_BASE, SDFM_FILTER_3) == 0x01U)
    {
        for (i = 0; i < FIFO_INT_NUMBER; i++)
            filter3Result[loopCounter7++] = (int16_t)(SDFM_getFIFOData(
                SDFM2_BASE, SDFM_FILTER_3) >> 16U);
    }

    /* Clear SDFM flag register (SDIFLG) */
    SDFM_clearInterruptFlag(SDFM2_BASE,
                           (SDFM_MAIN_INTERRUPT_FLAG |
                            SDFM_FILTER_3_FIFO_INTERRUPT_FLAG |
                            SDFM_FILTER_3_FIFO_OVERFLOW_FLAG));
}

void sdfm2_D4_FIFO_ISR(void)
{
    uint16_t i = 0;
    static uint16_t loopCounter8 = 0;

    if (loopCounter8 >= MAX_SAMPLES)
    {
        asm ("NOP");
    }
    else if (SDFM_getFIFOISRStatus(SDFM2_BASE, SDFM_FILTER_4) == 0x01U)
    {
        for (i = 0; i < FIFO_INT_NUMBER; i++)
            filter4Result[loopCounter8++] = (int16_t)(SDFM_getFIFOData(
                SDFM2_BASE, SDFM_FILTER_4) >> 16U);
    }

    /* Clear SDFM flag register (SDIFLG) */
    SDFM_clearInterruptFlag(SDFM2_BASE,
                           (SDFM_MAIN_INTERRUPT_FLAG |
                            SDFM_FILTER_4_FIFO_INTERRUPT_FLAG |
                            SDFM_FILTER_4_FIFO_OVERFLOW_FLAG));
}
