/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File:	spi_ex3_external_loopback.c
 *
 * @Tile:	SPI Digital Loopback without using FIFOs and Interrupts
 *
 * @brief:
 * This program uses the external loopback between two SPI modules.
 * Both the SPI FIFOs and interrupts are not used in this example.
 * SPIA is configured as a peripheral and SPI B is configured as controller.
 * This example demonstrates full duplex communication where both controller
 * and peripheral transmits and receives data simultaneously.
 *
 * @External Connections:
 * Refer to SysConfig for external connections (GPIO pin numbers) specific to each device
 *
 * @Watch Variables:
 * - TxData_SPIA - Data send from SPIA (peripheral).
 * - TxData_SPIA - Data send from SPIA (peripheral).
 * - RxData_SPIA - Data received by SPIA (peripheral).
 * - RxData_SPIA - Data received by SPIA (peripheral)
 */

#include <spi_ex03_board.h>
#include "driverlib.h"
#include "device.h"

/* Main */
int main(void)
{
    uint16_t i;

    volatile uint16_t TxData_SPIA[] = {0xFF, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F};
    volatile uint16_t RxData_SPIA[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    volatile uint16_t TxData_SPIB[] = {0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F};
    volatile uint16_t RxData_SPIB[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    /* Initialize device clock and peripherals */
    Device_init();

    __enable_irq();

    Board_init();

    DEVICE_DELAY_US(23);

    /* Loop forever. Suspend or place breakpoints to observe the buffers. */
    for(i = 0; i < 16; i++)
    {
    	/* Set the TX buffer of peripheral SPI. */
        SPI_writeDataNonBlocking(SPIA_BASE, TxData_SPIA[i]);

        /* Set the controller TX buffer. This triggers the data transmission */
        SPI_writeDataNonBlocking(SPIB_BASE, TxData_SPIB[i]);

        /* Read the received data */
        RxData_SPIA[i] = SPI_readDataBlockingFIFO(SPIA_BASE);
        RxData_SPIB[i] = SPI_readDataBlockingFIFO(SPIB_BASE);

        /* Check the received data */
        if(RxData_SPIA[i] != TxData_SPIB[i])
        {
            ESTOP0;
        }
        if(RxData_SPIB[i] != TxData_SPIA[i])
        {
            ESTOP0;
        }
    }

    while(1);

    return 0;
}

