/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File:	spi_ex05_polling_flash_interrupt.c
 *
 * @Title:	SPI Flash
 *
 * @brief:
 * This program will write 8 bytes to FLASH and read them back.
 * The device communicates with the FLASH via SPI and specific opcodes.
 * This example is written to work with the SPI Serial FLASH AT25128/256.
 *
 * @External Connections:
 * - Connect external SPI FLASH
 * - Connect GPIO63 (PICO) to external FLASH SI pin
 * - Connect GPIO64 (POCI) to external FLASH SO pin
 * - Connect GPIO65 (CLK) to external FLASH SCK pin
 * - Connect GPIO66 or GPIO08 (CS) to external FLASH CS pin
 * - Connect the external FLASH VCC and GND pins
 *
 * @Watch Variables:
 * - writeBuffer - Data that is written to external FLASH
 * - readBuffer  - Data that is read back from FLASH
 * - error       - Error count
 *
 */

#include <spi_ex05_board.h>
#include "driverlib.h"
#include "device.h"
#include "w25q64.h"

/* The number of data to be sent */
#define TEST_DATA_COUNT		256

/* read operation : read data array */
uint8_t readBuffer[TEST_DATA_COUNT] = {0};

/* write operation : write data array
 * The first four fields are commands
 */
uint8_t writeBuffer[TEST_DATA_COUNT] = {0};

uint32_t jedec_id = 0;

void spiFIFOISR(void);

int main(void)
{
    uint32_t i;

    Device_init();

    /*
     * Disable global interrupt
     */
    __disable_irq();

    Board_init();

    /*
     * Set the CS to inactive
     * when GPIO08 as CS
     */
    CS_HIGH;

    /*
     * Initialize the send data buffers
     */
    for (i = 0; i < TEST_DATA_COUNT; i++)
    {
    	writeBuffer[i] = i;
    }

    /*
     * Enable global interrupt
     */
    __enable_irq();

    /* Read the flash id and confirm that the connection is correct */
    jedec_id = w25q64_read_jedec_id(mySPI_BASE);

    /* Erase 64k space */
    w25q64_erase(mySPI_BASE, 0, 0x10000);

    /* Write 256 bytes */
    w25q64_write(mySPI_BASE,0, TEST_DATA_COUNT,writeBuffer);

    /* Read a page of data */
    w25q64_read(mySPI_BASE, 0, TEST_DATA_COUNT, readBuffer);

    /* Compare whether the data write and read are consistent */
    for (i = 0; i < TEST_DATA_COUNT; i++)
    {
        if (readBuffer[i] !=  writeBuffer[i])
        {
        	ESTOP0;
        }
    }

    for(;;);

    return 0;

}

void spiFIFOISR(void)
{

    /* Gets the interrupt flag bit */
    uint32_t status = SPI_getInterruptStatus(mySPI_BASE);
    /* If FIFO is error */
    if((status & SPI_ISR_TXOIS) \
    		||(status & SPI_ISR_RXUIS)\
    		||(status & SPI_ISR_RXOIS))
    {
    	SPI_disableInterrupt(mySPI_BASE,status);
    	ESTOP0;
    }

    /* Clear the Current interrupt flag bit */
    SPI_clearAllInterruptStatus(mySPI_BASE);

}

