/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File:	spi_ex07_loopback_dma.c
 *
 * @Title:	SPI loop back use DMA
 *
 * @brief:
 * This program uses the internal loopback test mode of the SPI module.
 * This is a very basic loopback that does  use the DMA.
 * A stream of data is sent and then compared to the received stream.
 * The pinmux and SPI modules are configure through the sysconfig file.
 *
 * @Watch Variables:
 * - send 	 - Data to send
 * - recive  - DReceived data
 *
 */

#include <spi_ex07_board.h>
#include "device.h"
#include "driverlib.h"

/*
 * Globals
 */

volatile uint16_t done = 0;         // Flag to set when all data transferred
volatile uint32_t num = 0;
volatile uint32_t num2 = 0;


uint16_t send[128];
uint16_t recive[128];

/*
 * Function Prototypes
 */
void INT_mySPI0_RX_DMA_ISR(void);
void INT_mySPI0_TX_DMA_ISR(void);
void spiRxFIFOISR(void);

int main(void)
{
    uint16_t i;

    /*
     * Initialize device clock and peripherals
     */
    Device_init();

    /*
     * Board initialization from SysConfig
     */
    Board_init();

    /*
     * Initialize the data buffers
     */
    for(i = 0; i < 128; i++)
    {
    	send[i] = i;
    	recive[i]= 0;
    }

    /*
     * Enable Global Interrupt (INTM) and realtime interrupt (DBGM)
     */
    EINT;

#if IS_GS32F00xx(0x12)
    DMA_startChannel(mySPI0_TX_DMA_BASE);
    DMA_startChannel(mySPI0_RX_DMA_BASE);
#elif IS_GS32F3xx(0x22)
    XDMA_startChannel(mySPI0_TX_DMA_BASE);
    XDMA_startChannel(mySPI0_RX_DMA_BASE);
#endif

    /*
     * Enable Global Interrupt (INTM) and realtime interrupt (DBGM)
     */
    while(1);

}

void spiRxFIFOISR(void)
{
	/* Gets the interrupt flag bit */
    uint32_t status = SPI_getInterruptStatus(mySPI_BASE);

    /* If FIFO is error */
    if((status & SPI_ISR_TXOIS) \
    		||(status & SPI_ISR_RXUIS)\
    		||(status & SPI_ISR_RXOIS))
    {
    	SPI_disableInterrupt(mySPI_BASE,status);
    	ESTOP0;
    }

    SPI_clearAllInterruptStatus(mySPI_BASE);
}

//
// DMA Channel 0 ISR
//
void INT_mySPI0_TX_DMA_ISR(void)
{
#if IS_GS32F00xx(0x12)
    DMA_stopChannel(mySPI0_TX_DMA_BASE);
	DMA_clearInterrupt(mySPI0_TX_DMA_BASE, DMA_INT_TFR);
#elif IS_GS32F3xx(0x22)
    XDMA_clearInterrupt(mySPI0_TX_DMA_BASE, XDMA_INT_BLOCK);
#endif
}

//
// DMA Channel 1 ISR
//
void INT_mySPI0_RX_DMA_ISR(void)
{
    uint16_t i;

#if IS_GS32F00xx(0x12)
    DMA_stopChannel(mySPI0_RX_DMA_BASE);
	DMA_clearInterrupt(mySPI0_RX_DMA_BASE, DMA_INT_TFR);
#elif IS_GS32F3xx(0x22)
    XDMA_clearInterrupt(mySPI0_RX_DMA_BASE, XDMA_INT_BLOCK);
#endif
    /*
     * Check for data integrity
     */
    for(i = 0; i < 128; i++)
    {
        if (recive[i] != i)
        {
        	while(1);
            /* Something went wrong. rData doesn't contain expected data. */
            ESTOP0;
        }
    }
}
