/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "spi_loopback_fifo_interrupt.h"
#include "device.h"
#include "spi.h"
#include "log.h"

static volatile uint16_t sdata[CONFIG_SPI_FIFO_LEVE];
static volatile uint16_t rdata[CONFIG_SPI_FIFO_LEVE];

static
void __spi_tx_isr_handle(void)
{
    uint16_t status, i;
    /** <\ @brief get interrupt status. */
    status = SPI_getInterruptStatus(CONFIG_SPI_BASE);

    if(status & SPI_INT_TXFF)
    {
        /**
         * @brief Check whether the remaining FIFO meets
         * the requirement for a one-time transmission.
         */
        if((SPI_FIFO_RXFULL - SPI_getRxFIFOStatus(CONFIG_SPI_BASE)) > CONFIG_SPI_FIFO_LEVE)
        {
            for(i = 0; i < CONFIG_SPI_FIFO_LEVE; i++)
            {
                SPI_writeDataNonBlocking(CONFIG_SPI_BASE, sdata[i]);
            }
        }
        SPI_clearInterruptStatus(CONFIG_SPI_BASE, SPI_INT_TXFF);
    }
}

static
void __spi_rx_isr_handle(void)
{
    uint16_t status, i;
    /** <\ @brief get interrupt status. */
    status = SPI_getInterruptStatus(CONFIG_SPI_BASE);
    /** <\ @check if fifo overrun. */
    if(status & SPI_INT_RXFF_OVERFLOW)
    {
        log_error("received overflow!\n");
        ESTOP0;
        SPI_clearInterruptStatus(CONFIG_SPI_BASE, SPI_INT_RXFF_OVERFLOW);
    }
    /** <\ @brief rxfifo event. */
    if(status & SPI_INT_RXFF)
    {
        for(i = 0; i < CONFIG_SPI_FIFO_LEVE; i++)
        {
            rdata[i] = SPI_readDataNonBlocking(CONFIG_SPI_BASE);
            if(rdata[i] != sdata[i])
            {
                log_error("transmission failed!\n");
                ESTOP0;
            }
        }
        log_info("transmission ok!\n");
        SPI_clearInterruptStatus(CONFIG_SPI_BASE, SPI_INT_RXFF);
    }
}

static
void __spi_pin_init(void)
{

}

void spi_init(void)
{
    uint16_t i;

    log_info("03_spi_loopback_fifo_interrupt start!\n");

    /** <\ @brief enable SPIA and SPIB clk, same as APB bus. */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPI);
    /** <\ @brief reset SPIA and SPIB. */
    SysCtl_resetSpi();
    /** <\ @brief SPI pins initial. */
    __spi_pin_init();
    /** <\ @brief holed SPI in reset state before configure SPI. */
    SPI_disableModule(CONFIG_SPI_BASE);

    SPI_setConfig(CONFIG_SPI_BASE, DEVICE_APBCLK_FREQ,
                  SPI_PROT_POL0PHA0, SPI_MODE_CONTROLLER,
                  CONFIG_SPI_BITRATE, CONFIG_SPI_DATAWIDTH);
    /** <\ @brief enable fifo mode. */
    SPI_enableFIFO(CONFIG_SPI_BASE);

    SPI_disableInterrupt(CONFIG_SPI_BASE,
                        SPI_INT_RXFF |
                        SPI_INT_RXFF_OVERFLOW |
                        SPI_INT_TXFF);
    /** <\ @brief reset rx/tx fifo pointer to zero. */
    SPI_resetTxFIFO(CONFIG_SPI_BASE);
    SPI_resetRxFIFO(CONFIG_SPI_BASE);
    SPI_reset(CONFIG_SPI_BASE);
    /** <\ @brief set fifo interrupt level. */
    SPI_setFIFOInterruptLevel(CONFIG_SPI_BASE,
                              CONFIG_SPI_FIFO_LEVE,
                              CONFIG_SPI_FIFO_LEVE);
    /** <\ @brief enable fifo interrupt. */
    SPI_enableInterrupt(CONFIG_SPI_BASE,
                        SPI_INT_TXFF|
                        SPI_INT_RXFF_OVERFLOW |
                        SPI_INT_RXFF);

    SPI_clearInterruptStatus(CONFIG_SPI_BASE,
                         SPI_INT_RX_OVERRUN |
                         SPI_INT_RX_DATA_TX_EMPTY |
                         SPI_INT_RXFF |
                         SPI_INT_RXFF_OVERFLOW |
                         SPI_INT_TXFF);
    SPI_enableLoopback(CONFIG_SPI_BASE);
    /** <\ @brief release SPI. */
    SPI_enableModule(CONFIG_SPI_BASE);
    /** <\ @brief register system interrupt. */
    Interrupt_register(CONFIG_SPI_TX_IRQn, __spi_tx_isr_handle);
    Interrupt_register(CONFIG_SPI_RX_IRQn, __spi_rx_isr_handle);

    Interrupt_enable(CONFIG_SPI_TX_IRQn);
    Interrupt_enable(CONFIG_SPI_RX_IRQn);
    /** <\ @brief initial transmission data. */
    for(i = 0; i < CONFIG_SPI_FIFO_LEVE; i++)
    {
        sdata[i] = i;
        rdata[i] = 0;
    }
}

void spi_process(void)
{
    uint16_t i;
    while(1)
    {
        /**
         * @brief send and received by interrupt,
         * there is nothing to do.
         */
        __NOP();
    }
}
