/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "spi_loopback_dma.h"
#include "device.h"
#include "spi.h"
#include "dma.h"
#include "regs/regs_dma.h"
#include "log.h"

#define __SPI_TRANSMISSION_DATA_SIZE (512U)
#define __SPI_TRANSMISSION_BLOCK_SIZE (128U)

static volatile uint16_t sdata[__SPI_TRANSMISSION_DATA_SIZE];

static volatile uint16_t rdata[__SPI_TRANSMISSION_DATA_SIZE];

/** <\ @brief block transmission timing. */
static uint16_t __g_dma_tfs_cnt = __SPI_TRANSMISSION_DATA_SIZE / __SPI_TRANSMISSION_BLOCK_SIZE;

static
void __dma_tx_isr_handle(void)
{
    DMA_clearInterrupt(CONFIG_DMA_TXCH_BASE, DMA_INT_BLOCK);
}

static
void __dma_rx_isr_handle(void)
{
    DMA_clearInterrupt(CONFIG_DMA_RXCH_BASE, DMA_INT_BLOCK);

    __g_dma_tfs_cnt--;

    if(__g_dma_tfs_cnt != 0)
    {
        DMA_startChannel(CONFIG_DMA_TXCH_BASE);
        DMA_startChannel(CONFIG_DMA_RXCH_BASE);
    }

    return;
}

static
void __dma_tx_init(void)
{
    DMA_ConfigParams dmaTxInitParam = {0};

    DMA_stopChannel(CONFIG_DMA_TXCH_BASE);
    /** <\ @brief configure DMA TX channel. */
    dmaTxInitParam.enableInterrupt = 1;
    dmaTxInitParam.srcAddr = (uint32_t)(sdata);
    dmaTxInitParam.destAddr = CONFIG_SPI_TXBUF;

    dmaTxInitParam.blockTS = __SPI_TRANSMISSION_BLOCK_SIZE;
    dmaTxInitParam.ttfc = DMA_TT_FC_1_M2P_DMAC;
    /** <\ @brief burst count per transmission. */
    dmaTxInitParam.srcBtl = DMA_BTL_8;
    dmaTxInitParam.destBtl = DMA_BTL_8;
    dmaTxInitParam.srcAddrDirect = DMA_ADDR_INCRE;
    dmaTxInitParam.destAddrDirect = DMA_ADDR_NO_CHANGE;

    dmaTxInitParam.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dmaTxInitParam.destTrWidthBytes = DMA_TR_WIDTH_BYTE_2;

    dmaTxInitParam.reloadSrc = 0;
    dmaTxInitParam.reloadDst = 0;

    /** <\ @brief translate start form SPI request. */
    dmaTxInitParam.dmaDstReqId = CONFIG_SPI_DMA_TX_REQ_ID;
    dmaTxInitParam.chPriority = DMA_CH_PRIORITY_7;

    DMA_configChannel(CONFIG_DMA_TXCH_BASE, &dmaTxInitParam);
    DMA_clearInterrupt(CONFIG_DMA_TXCH_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(CONFIG_DMA_TXCH_BASE, DMA_INT_BLOCK);

    Interrupt_register(CONFIG_DMA_TXCH_IRQn, __dma_tx_isr_handle);
    Interrupt_enable(CONFIG_DMA_TXCH_IRQn);
}

static
void __dma_rx_init(void)
{
    DMA_ConfigParams dmaRxInitParam = {0};

    DMA_stopChannel(CONFIG_DMA_RXCH_BASE);
    /** <\ @brief configure DMA RX channel. */
    dmaRxInitParam.enableInterrupt = 1;
    dmaRxInitParam.srcAddr = CONFIG_SPI_RXBUF;
    dmaRxInitParam.destAddr = (uint32_t)(rdata);

    dmaRxInitParam.blockTS = __SPI_TRANSMISSION_BLOCK_SIZE;
    dmaRxInitParam.ttfc = DMA_TT_FC_2_P2M_DMAC;
    /** <\ @brief burst count per transmission. */
    dmaRxInitParam.srcBtl = DMA_BTL_8;
    dmaRxInitParam.destBtl = DMA_BTL_8;
    dmaRxInitParam.srcAddrDirect = DMA_ADDR_NO_CHANGE;
    dmaRxInitParam.destAddrDirect = DMA_ADDR_INCRE;

    dmaRxInitParam.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dmaRxInitParam.destTrWidthBytes = DMA_TR_WIDTH_BYTE_2;

    dmaRxInitParam.reloadSrc = 0;
    dmaRxInitParam.reloadDst = 0;

    /** <\ @brief translate start form spi request. */
    dmaRxInitParam.dmaSrcReqId = CONFIG_SPI_DMA_RX_REQ_ID;
    dmaRxInitParam.chPriority = DMA_CH_PRIORITY_7;

    DMA_configChannel(CONFIG_DMA_RXCH_BASE, &dmaRxInitParam);
    DMA_clearInterrupt(CONFIG_DMA_RXCH_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(CONFIG_DMA_RXCH_BASE, DMA_INT_BLOCK);

    Interrupt_register(CONFIG_DMA_RXCH_IRQn, &__dma_rx_isr_handle);
    Interrupt_enable(CONFIG_DMA_RXCH_IRQn);
}

static
void __dma_init(void)
{
    DMA_initController(CONFIG_DMA_BASE);
    __dma_rx_init();
    __dma_tx_init();
}

void spi_init(void)
{
    uint16_t i;

    log_info("05_spi_loopback_dma demo start!\n");

    /** <\ @brief enable SPIA and SPIB clk, same as APB bus. */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPI);
    /** <\ @brief reset SPIA and SPIB. */
    SysCtl_resetSpi();
    /** <\ @brief holed SPI in reset state before configure SPI. */
    SPI_disableModule(CONFIG_SPI_BASE);

    SPI_setConfig(CONFIG_SPI_BASE, DEVICE_APBCLK_FREQ,
                  SPI_PROT_POL0PHA0, SPI_MODE_CONTROLLER,
                  CONFIG_SPI_BITRATE, CONFIG_SPI_DATAWIDTH);

	SPI_setPTESignalPolarity(CONFIG_SPI_BASE, SPI_PTE_ACTIVE_LOW);
    /** <\ @brief enable fifo mode. */
    SPI_enableFIFO(CONFIG_SPI_BASE);

    /** <\ @brief reset rx/tx fifo pointer to zero. */
    SPI_resetTxFIFO(CONFIG_SPI_BASE);
    SPI_resetRxFIFO(CONFIG_SPI_BASE);
    SPI_reset(CONFIG_SPI_BASE);

    /** <\ @brief set fifo interrupt level. */
    SPI_setFIFOInterruptLevel(CONFIG_SPI_BASE,
                              CONFIG_SPI_FIFO_LEVE,
                              CONFIG_SPI_FIFO_LEVE);

    SPI_clearInterruptStatus(CONFIG_SPI_BASE,
                             SPI_INT_TXFF|
                             SPI_INT_RXFF_OVERFLOW |
                             SPI_INT_RXFF);
    /** <\ @brief enable fifo interrupt. */
    SPI_disableInterrupt(CONFIG_SPI_BASE,
                         SPI_INT_TXFF|
                         SPI_INT_RXFF_OVERFLOW |
                         SPI_INT_RXFF);

    SPI_enableLoopback(CONFIG_SPI_BASE);

    SPI_setEmulationMode(CONFIG_SPI_BASE, SPI_EMULATION_FREE_RUN);

    /** <\ @brief release SPI. */
    SPI_enableModule(CONFIG_SPI_BASE);

    /** <\ @brief initial SPI dma. */
    __dma_init();

    /** <\ @brief initial transmission data. */
    for(i = 0; i < __SPI_TRANSMISSION_DATA_SIZE; i++)
    {
        sdata[i] = i;
        rdata[i] = 0;
    }
    DisableDCache();
}

void spi_process(void)
{
    uint16_t i;

    DMA_startChannel(CONFIG_DMA_RXCH_BASE);
    DMA_startChannel(CONFIG_DMA_TXCH_BASE);

    /** <\ @brief wait for dma transmission completely. */
    while(__g_dma_tfs_cnt);
    /** <\ @brief compare data. */
    for(i = 0; i < __SPI_TRANSMISSION_DATA_SIZE; i++)
    {
        if(sdata[i] != rdata[i])
        {
            log_error("transmission failed!\n");
            ESTOP0;
        }
    }

    log_info("spi/dma test ok!\n");

    while(1)
    {
        __NOP();
    }

}
