/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * @example: Receive interrupt by TimeOut(in FIFO mode)
 * @Language:English
 * @encode:
 * @brief:
 * This routine demonstrates how the SCI module uses FIFO and TimeOut Interrupt to receive indefinite length data within 16 bytes.
 * After enabling the FIFO and initiating the receive interrupt,
 * two interrupts will trigger the enable, TimeOut and FIFO Level,
 * Therefore, attention should be paid when judging the interrupt state.
 * The triggering of TimeOut is based on the FIFO waterline implementation.
 * When the FIFO does not reach the waterline and no data is received within 4 bytes,
 * the TimeOut interrupt is triggered.
 * Therefore, the sent bytes should be less than the waterline height
 *
 * Program flow:
 * In initializing the SCI program,the program enables FIFO and sets the watermark of RxFIFO to 8,
 * so the bytes sent by the sender should be less than 8.
 * In the interrupt service function, the receive data valid flag is only used to receive data,
 * and the completion of data transmission is determined by TimeOut.
 * TimeOut will set the relevant flag after the data transmission is completed,
 * After detecting the flag position, the main program will print out the received data.
 *
 *	Note:
 *	1. When using TimeOut and FIFO level, attention should be paid to interrupt clearing.
 *		TimeOut and FIFO level interrupt clearing both read RBR, and FIFO level has a higher priority than TimeOut.
 *		Therefore, when receiving data, both flags should be checked simultaneously, otherwise it will loop into interrupts.
 *	2. When using, special attention should be paid that TimeOut will not be triggered when the number of bytes sent is equal to the waterline of RxFIFO!!!
 *	3. When setting the SCI baud rate, it should be ensured that SCI is in an idle state. SCI can be made idle through the SCI_deset() function before setting the baud rate.
 * */

/*
 * @example: Receive interrupt by TimeOut(in FIFO mode)
 * @Language:Chinese
 * @encode:GBK
 * @brief:
 *	ʾSCIģʹFIFOóʱж(TimeOut Interrupt)ʵֽ16ֽڵĲݽա
 *	ʹFIFOжϺ󣬻жϴʹܣʱж(TimeOut)FIFOˮ(FIFO Level)
 *	жж״̬ʱӦע⡣
 *	TimeOutĴǻFIFOˮʵ֣FIFOδﵽˮ4ֽʱδյݣTimeOutжϡ
 *	ֽӦСˮ߸߶
 *
 *	̣
 *	ʼSCIУʹFIFORxFIFOˮΪ8˷Ͷ͵ֽӦС8
 *	жϷУݺϷ־λݣݵĴTimeOutжϡ
 *	TimeOutж϶εݴɺ󣬻λر־λ
 *	⵽־λΪ󣬻Ὣյݴӡ
 *
 *	ע⣺
 *		1ʹTimeOutFIFO levelʱעжTimeOutFIFO levelжΪȡRBR
 *			FIFO levelűTimeOutߵȼڽʱӦͬʱж־λѭжϡ
 *		2ʹʱӦرע⣬͵ֽRxFIFOˮʱᴥTimeOut
 *		3SCIʱӦ֤SCIڿ״̬ͨSCI_reset()ʹSCIУٽвá
 * */

#include "device.h"
#include "sci_ex04_rx_irq_TimeOut.h"

int main(void)
{

	/* initialize the Flash and System Clock, and enable the clocks for all peripherals. */
    Device_init();

    /* Enable global interrupt */
    __enable_irq();

    UART_Init(MY_SCI_BASE, 115200);

    UART_writeCharArray(MY_SCI_BASE, "SCI_ex02_rx_irq start \r\n", 24);

    sci_ex04_rx_irq_TimeOut_Example(MY_SCI_BASE);

	UART_writeCharArray(MY_SCI_BASE, "SCI_ex02_rx_irq end \r\n", 22);

	for(;;);
    return 0;
}




