/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * @example: printf and log
 * @Language:English
 * @encode:
 * @brief:
 * 	This routine demonstrates how to use formatted printing (printf) and log printing (log).
 * 	Before use, make sure that the SCI module defined by the 'OG_SCI_SASE' macro has been initialized.
 * 	'OG_SCI_SASE' is defined in 'inc/board.h'.
 *
 * 	Program flow:
 * 	By directly calling UartPrint_init (uint32_t SCIBase, uint32_t baud) for initialization,
 * 		printf and log can be used.
 * 	UartPrint_init (uint322-t SCIBase, uint322-t baud) initializes the default SCI and
 * 		pins on the development board.
 * 	If you want to modify, you can implement SCI initialization by yourself and
 * 		modify the macro definition of LOG_SCI_SASE.
 *
 * 	By default, ANSI color escape sequences are used for log printing,
 * 		which can print content in different colors.
 * 		However, some serial port tools may not be able to parse it,
 * 		which can be disabled by commenting on LOG-USE_COLOR.
 *
 *	Note:
 *		1. When using printf to print line breaks, simply use '\ n'.
 *		2. Before using log printing, it is necessary to set the log level (log_det_level()).
 * */

/*
 * @example: printf and log
 * @Language:Chinese
 * @encode:GBK
 * @brief:
 *	ʾʹøʽӡ(printf)־ӡ(log)
 *	ʹǰȷ "LOG_SCI_BASE" SCIģѳʼ
 *	"LOG_SCI_BASE"  "GS32Studio/packages/GS32-DSPWare/gs32_driver/init/device.c"
 *
 *	̣
 *	ֱͨӵUartPrint_init(uint32_t SCIBase, uint32_t baud)ֱӳʼʹprintflog
 *	UartPrint_init(uint32_t SCIBase, uint32_t baud)ʼǿĬϵSCIš
 *	Ҫ޸ĿʵSCIĳʼ޸LOG_SCI_BASEĺ궨弴ɡ
 *
 *	־ӡlogĬʹANSIɫתУɴӡͬɫݣ
 *		ִڹߴ޷ͨעLOG_USE_COLORرա
 *
 *
 *	ע⣺
 *		1ʹprintfӡеʱʹ'\n'ɡ
 *		2ʹ־logӡǰ־ȼ(log_set_level())
 *
 * */

#include "device.h"
#include "log.h"
#include "sci_ex09_printf.h"

int main(void)
{
	/* initialize the Flash and System Clock, and enable the clocks for all peripherals. */
    Device_init();

    /* The printf initializes default SCI */
    UartPrint_init(LOG_SCI_BASE, 115200);

    /* Set log level,Optional: LOG_TRACE, LOG_DEBUG, LOG_INFO, LOG_WARN, LOG_ERROR, LOG_FATAL */
    log_set_level(LOG_DEBUG);

    UART_writeCharArray(LOG_SCI_BASE, "SCI_ex09_printf start \r\n", 26);

    /* Run loopback test */
    sci_ex09_loopback_Example(LOG_SCI_BASE);

    UART_writeCharArray(LOG_SCI_BASE, "SCI_ex09_printf end \r\n", 24);

    for(;;);
    return 0;
}




