/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define LED_GPIO_PIN    GPIO_PIN_LED1
#define LED_GPIO_CFG    GPIO_CFG_LED1

#define WWD_BASE 		WWD1_BASE

/* The time setting is approximately 1 second */
#define WWD_MAX_COUNT 	19531
/* The time setting is approximately 0.5 second */
#define WWD_WINDOW_SIZE 9765
#define Delay_time 		520


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

void WWD_init(uint32_t base, uint32_t timeout_cnt, uint32_t win_cnt)
{
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
	/* WindowWatchdog can trigger reset */ 
	SysCtl_setDspWwdgResetEn(1);
#elif IS_GS32F3xx(0x22)
	if(WWD1_BASE == base){
		/* CPU1 WindowWatchdog can trigger reset */ 
		SysCtl_setDspCpu1WwdgResetEn(1);
	}if(WWD2_BASE == base){
		/* Whether system reset follows CPU2's watchdog reset */
		SysCtl_setCpu2ResetOnSys(1);
		/* CPU2 WindowWatchdog can trigger reset */ 
		SysCtl_setDspCpu2WwdgResetEn(1);
	}
#endif
	/* disables the WindowWatchdog timer */
	SysCtl_disableWindowWatchdog(base);

	/* configure the Windowwatchdog timer reload  win_cnt value */
	SysCtl_initWindowWatchdog(base, timeout_cnt, win_cnt);

	/* Set pre allocation coefficient(Default 512 division frequency) */
	SysCtl_setWatchdogPredivider(base,SYSCTL_WD_PREDIV_512);
	/* Set the division coefficient(Default 1 division frequency) */
	SysCtl_setWatchdogPrescaler(base,SYSCTL_WD_PRESCALE_1);

	/* enable watchdog reset mode */
	SysCtl_setWindowWatchdogMode(base, WD_MODE_RESET_FIRST);

	/* Debug Windowwatchdog Hold */
	SysCtl_enableWatchdogDebugMode(base);
	
	/* turn on Windowwatchdog timer */
	SysCtl_enableWindowWatchdog(base);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief  main function.
 */
int main(void)
{
	uint16_t WindowWatchdogCounterValue = 0;
	__disable_irq();

    Device_init();

	/* Set the pin to gpio function */
	GPIO_setPinConfig(LED_GPIO_CFG);
	/* Output level is low */
	GPIO_writePin(LED_GPIO_PIN, 0);
	/* Disable simulation function */
	GPIO_setAnalogMode(LED_GPIO_PIN, GPIO_ANALOG_DISABLED);
	/* Set pins to output mode */
	GPIO_setDirectionMode(LED_GPIO_PIN, GPIO_DIR_MODE_OUT);

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("DSP Window Watchdog Reset!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    log_info("Code @ 0x%08X, Data @ 0x%08X.\r\n", (uint32_t)main, (uint32_t)&WindowWatchdogCounterValue);

	/* Initialize the Window watchdog */
    WWD_init(WWD_BASE, WWD_MAX_COUNT ,WWD_WINDOW_SIZE);

	/* Delay 500ms */
	DELAY_US(500 * 1000);

	/* Disable WindowWatchdog */
	SysCtl_disableWindowWatchdog(WWD_BASE);

	/* Get the current count value */
	WindowWatchdogCounterValue = SysCtl_getWindowWatchdogCounterValue(WWD_BASE);

	log_info(" WindowsWatchdogCounterValue = %d\r\n\r\n",WindowWatchdogCounterValue);

	/* Output level is high */
	GPIO_writePin(LED_GPIO_PIN, 1);

	/* turn on WindowWatchdog timer (This will cause the counting register to reload)*/
	SysCtl_enableWindowWatchdog(WWD_BASE);

    while (1) {

        /* Increasing or decreasing the delay will trigger a reset */
        DELAY_US(Delay_time * 1000);

        /* Feed a dog */
		SysCtl_serviceWindowWatchdog(WWD_BASE);

    }

    return 0;
}

