/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    low_level_init.c
*   @brief   low level initialization. MPOST / wdg enable / double reset ...
*   @details
*
*/
/*
 * commit history
 * 20241108, zhaolei, add CCM remap.
 * 2025/1/1,  hezhiyua,Modify the base and offset address names of WD, add Pin reset judgment
 * */

#include "device.h"

extern const char __StackTop;
extern const char __StackBottom;

extern void Flash_initConfig(void);
extern void Clock_init(void);

#if defined(STARTUP_SYSCLK_FREQ) && BOOT_HARTID == 0
__INIT_FUNC_T __STATIC_FORCEINLINE void low_level_delay(register volatile uint32_t cnt)
{
    while(cnt--);
}

__INIT_FUNC_T  __STATIC_FORCEINLINE void low_level_setup_pll(void)
{
    register uint32_t tacc = (STARTUP_SYSCLK_FREQ/40/1000/1000 + 1);

    SysCtl_setSysclkS2nEn(0);
    low_level_delay(10);

    SysCtl_setRefClkSel(REF_CLK_TYPE_INTRENAL_OSC1_10MHZ);

    SysCtl_setApbClkDiv(CRG_DIV_1);
#if IS_GS32F00xx(0x12)
    SysCtl_setSlvSysClkDiv(CRG_DIV_1);
#endif
    SysCtl_setDspSysClkDiv(CRG_DIV_1);
#if IS_GS32F00xx(0x30) || IS_GS32F3xx(0x22)
    SysCtl_setPeriClkDiv(CRG_DIV_1);
#endif
    SysCtl_setSysDivLoad();

    low_level_delay(10);

#ifdef FLASH_INTERFACE_EFC_BASE
    Flash_unlockKey0(FLASH_INTERFACE_EFC_BASE, FLASH_UNLOCK_CODE1);
    Flash_unlockKey1(FLASH_INTERFACE_EFC_BASE, FLASH_UNLOCK_CODE2);
    Flash_unlockKey0(FLASH_INTERFACE_EFC_BASE, FLASH_UNLOCK_CODE3);
    Flash_disableRegisterWriteProtect(FLASH_INTERFACE_EFC_BASE);
    Flash_setWaitstates(FLASH_INTERFACE_EFC_BASE, 1);
#endif

#ifdef FLASH_INTERFACE_EFC2_BASE
    Flash_unlockKey0(FLASH_INTERFACE_EFC2_BASE, FLASH_UNLOCK_CODE1);
    Flash_unlockKey1(FLASH_INTERFACE_EFC2_BASE, FLASH_UNLOCK_CODE2);
    Flash_unlockKey0(FLASH_INTERFACE_EFC2_BASE, FLASH_UNLOCK_CODE3);
    Flash_disableRegisterWriteProtect(FLASH_INTERFACE_EFC2_BASE);
    Flash_setWaitstates(FLASH_INTERFACE_EFC2_BASE, 1);
#endif

    //--- Step 2 : Config the PLL, Read PLL_LCK until PLL Locked
    SysCtl_setPllDisableSscg(0x01);
    SysCtl_setPllResetptr(0x01);
    SysCtl_setPllPd(0x01);
    low_level_delay(100);

    SysCtl_setPllFbdivCfg(4 * STARTUP_SYSCLK_FREQ / (10*1000*1000));
    SysCtl_setPllFracCfg(0);
    SysCtl_setPllDivCfg(0x02 | (0x01 << 3) | (0x01 << 6));
    SysCtl_setPllDivval(0x00000001UL);
    SysCtl_setPllSpread(0);
    SysCtl_setPllDownspread(0);
    SysCtl_setPllOthpdCfg(2);                               /* integer mode */

    low_level_delay(10);

    SysCtl_setPllDisableSscg(0);
    low_level_delay(10);

    SysCtl_setPllPd(0);
    low_level_delay(100);

    SysCtl_setPllResetptr(0);

    /* Wait until PLL lock status is 1 */
    while(SysCtl_getPllLck() == 0);         /* TODO: timeout */

    SysCtl_clearPllUnlckHis();

#ifdef FLASH_INTERFACE_EFC_BASE
    Flash_setWaitstates(FLASH_INTERFACE_EFC_BASE, tacc);
#endif

#ifdef FLASH_INTERFACE_EFC2_BASE
    Flash_setWaitstates(FLASH_INTERFACE_EFC2_BASE, tacc);
#endif

    //--- Step 3 : Config APBCLK/CANCLK/ANACLK Divider
#if IS_GS32F00xx(0x12)
    SysCtl_setSlvSysClkDiv(CRG_DIV_1);
#endif
    SysCtl_setDspSysClkDiv(CRG_DIV_1);
#if IS_GS32F3xx(0x22)
    SysCtl_setPeriClkDiv(CRG_DIV_4);
#endif

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
    SysCtl_setApbClkDiv(CRG_DIV_4);
#elif IS_GS32F00xx(0x30)
    SysCtl_setPeriClkDiv(CRG_DIV_2);
    SysCtl_setApbClkDiv(CRG_DIV_4);
#endif
    SysCtl_setSysDivLoad();

    //--- Step 4 : SYSCLK Slow2Normal, Config SYSCLK Divider to 2div and then to 1div
    SysCtl_setSysclkS2nEn(0x01);
    low_level_delay(100);
}
#endif

__WEAK __INIT_FUNC_T __attribute__((naked)) void low_level_init_start(void)
{
    /* call after reset & global interrupt is disabled
       register read & write only, no stack support */

#if defined(STARTUP_SYSCLK_FREQ) && BOOT_HARTID == 0
	low_level_setup_pll();
#endif

	/* enable ccm remap */
#if IS_GS32F00xx(0x12)
    HWREG(SYSCTL_BASE + SYSCTL_O_BUS_REMAP_CFG) = 1<<3;     //remap ccm to 0x20000000
#endif

#ifdef  FLASH_TARGET
    /* clear ram on POR reset */
#if IS_GS32F00xx(0x12)
    register uint32_t isPOR = HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) & 0x30000000;
    register uint32_t clearByHw = HWREG(FLASH_INTERFACE_EFC_BASE + SET_USER_OPT1) & (1<<(40-32));

    if (isPOR && !clearByHw) {
    	/* trigger RAM clear */

//    	HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) = 0x1F;
//    	HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_REQ) = 0x1F;
//
//    	while ((HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) & 0x1F) != 0x1F);

    	/* clear POR flag*/
    	HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) &= ~0x30000000;
    }

#elif IS_GS32F00xx(0x30)
    register uint32_t isPOR = HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) & 0x30000000;
	register uint32_t clearByHw = HWREG(FLASH_INTERFACE_EFC_BASE + SET_USER_OPT1) & (1<<(40-32));

	if (isPOR) {
		if(!clearByHw) {
			/* trigger SRAM clear */
			HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) = 0x07;
			HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_REQ) = 0x07;

			while ((HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) & 0x07) != 0x07);
		}

		/* trigger LSRAM clear by software only*/
		HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_DONE_RPT) = 0x08;
		HWREG(SYSCTL_BASE + SYSCTL_O_MEM_INI_REQ) = 0x08;

		register uint32_t cnt = (((64+16)*1024/4*2 + (128+32)*1024/4*3)/2 + 300)/2;	/* 4byte/2cycle shadow, 4byte/3cycle */
		while (cnt--);

		/* clear POR flag*/
		HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) &= ~0x30000000;
	}
#elif IS_GS32F3xx(0x22)
	register uint32_t clearByHw = HWREG(FLASH_INTERFACE_EFC_BASE + SET_USER_OPT1) & (1<<(42-32));
	register uint32_t hartid = __get_hart_id();
	register uint32_t addr;

    if (!clearByHw) {
        if (hartid == 0 && (HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) & 0x30000000) != 0) {       /* CPU1 */
            /* clear lsram */
            for (addr = 0x10800000; addr< 0x10800000 + (256+64)*1024; addr += 4*8) {
                HWREG(addr + 0) = 0;
                HWREG(addr + 4) = 0;
                HWREG(addr + 8) = 0;
                HWREG(addr + 12) = 0;
                HWREG(addr + 16) = 0;
                HWREG(addr + 20) = 0;
                HWREG(addr + 24) = 0;
                HWREG(addr + 28) = 0;
            }

            /* clear axisram */
            for (addr = 0x10200000; addr< 0x10200000 + 128*1024; addr += 4*8) {
                HWREG(addr + 0) = 0;
                HWREG(addr + 4) = 0;
                HWREG(addr + 8) = 0;
                HWREG(addr + 12) = 0;
                HWREG(addr + 16) = 0;
                HWREG(addr + 20) = 0;
                HWREG(addr + 24) = 0;
                HWREG(addr + 28) = 0;
            }

            /* clear flex ram */
            if (HWREG(FLASH_INTERFACE_EFC_BASE + SET_USER_OPT1) & (1<<(40-32))) {   /* single core has flex ram */
                for (addr = 0x10220000; addr< 0x10220000 + 128*1024; addr += 4*8) {
                    HWREG(addr + 0) = 0;
                    HWREG(addr + 4) = 0;
                    HWREG(addr + 8) = 0;
                    HWREG(addr + 12) = 0;
                    HWREG(addr + 16) = 0;
                    HWREG(addr + 20) = 0;
                    HWREG(addr + 24) = 0;
                    HWREG(addr + 28) = 0;
                }
            }

            /* clear POR flag*/
            HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) &= ~0x30000000;
        }
        else if (hartid == 1 && (HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) & 0x0C000000) != 0) {    /* CPU2 */
            /* clear lsram */
            for (addr = 0x10800000; addr< 0x10800000 + (256+64)*1024; addr += 4*8) {
                HWREG(addr + 0) = 0;
                HWREG(addr + 4) = 0;
                HWREG(addr + 8) = 0;
                HWREG(addr + 12) = 0;
                HWREG(addr + 16) = 0;
                HWREG(addr + 20) = 0;
                HWREG(addr + 24) = 0;
                HWREG(addr + 28) = 0;
            }

            /* clear POR flag*/
            HWREG(SYSCTL_BASE + SYSCTL_O_DIG_SYSCTL_POR_SPARE1) &= ~0x0C000000;
        }
    }
#endif
#endif  /* FLASH_TARGET */

#if IS_GS32F3xx(0x22) && defined(__POR_DOUBLE_RESET)
    if((__RV_CSR_READ(CSR_MHARTID) & 0xFF) == 0) {  //cpu1 only
       if ((HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_RST_RECORD) & 0x7F) == 0 ||
       		(HWREG(CRG_CFG_BASE + CRG_CFG_O_CPU1_RST_RECORD) & 0x7F) == 0x20) {   //por reset or Pin reset
            HWREG(WD1_BASE + WD_O_LOAD) = 10*1000*1000/512/100;           //20ms
            HWREG(WD1_BASE + WD_O_CONTROL) = 3;
            while(1);
        }
    }
#endif


#ifdef __INIT_WDG_PROTECT
    //init watchdog
    HWREG(WD1_BASE + WD_O_LOAD) = 10*1000*1000/512/2;           //1s
    HWREG(WD1_BASE + WD_O_CONTROL) = 3;
    HWREG(WD1_BASE + WD_O_INTCLR) = 0x07215AA1U;
    HWREG(WD1_BASE + WD_O_INTCLR) = 0x0609CA7FU;
#endif

    __ASM volatile("ret");
}

__WEAK __INIT_FUNC_T void low_level_init_end(void)
{
    //call before main()

#ifdef __INIT_WDG_PROTECT
    //disable watchdog
    HWREG(WD1_BASE + WD_O_INTCLR) = 0x07215AA1U;
    HWREG(WD1_BASE + WD_O_INTCLR) = 0x0609CA7FU;
    HWREG(WD1_BASE + WD_O_CONTROL) = 0;
#endif

#if defined(__ENABLE_STACK_CHECK) &&  __ENABLE_STACK_CHECK != 0
    __RV_CSR_WRITE(CSR_MSTACK_CTRL, 0);
    __RV_CSR_WRITE(CSR_MSTACK_BASE, &__StackTop);
    __RV_CSR_WRITE(CSR_MSTACK_BOUND, &__StackBottom);
    __RV_CSR_WRITE(CSR_MSTACK_CTRL, MSTACK_CTRL_UDF_EN | MSTACK_CTRL_OVF_TRACK_EN);     //enable overflow, underflow, check
#endif
}

