//#############################################################################
//
// FILE:   main.c
//
// TITLE:  ePWM Using DMA.
//
//! \addtogroup driver_example_list
//! <h1>ePWM DMA</h1>
//!
//! This example configures ePWM1 and DMA as follows:
//!  - ePWM1 is set up to generate PWM waveforms
//!  - DMA5 is set up to update the CMPAHR, CMPA, CMPBHR and CMPB every period
//!    with the next value in the configuration array. This allows the user to
//!    create a DMA enabled fifo for all the CMPx and CMPxHR registers to
//!    generate unconventional PWM waveforms.
//!  - DMA6 is set up to update the TBPHSHR, TBPHS, TBPRDHR and TBPRD every
//!    period with the next value in the configuration array.
//!  - Other registers such as AQCTL can be controlled through the DMA as well
//!    by following the same procedure. (Not used in this example)
//!
//! \b External \b Connections \n
//! - 
//! - 
//!
//! \b Watch \b Variables \n
//! - None.
//
//
//#############################################################################
/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    dma_ex03_epwm_puty_change.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdio.h>
#include "device.h"
#include "dma_ex03_epwm_puty_change_board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define TB_CLK    20000000U      
//#define PWM_CLK   500                      // We want to output at 5 kHz
//#define PRD_VAL   (TB_CLK / (PWM_CLK * 2))  // Calculate value period value
                                            // for up-down count mode
#define PRD_VAL        40000
#define PRD_VAL1   PRD_VAL/4  // Calculate value period value

#define EPWM_TIMER_TBPRD    40000
DMA_ConfigParams dmaCfg;
typedef uint16_t uint16;

#define CHANGE_PWM_DUTY_BY_SW   0
#define CHANGE_PWM_DUTY_BY_DMA    1
#define TRIG_DMA_BY_TIMER       0
#define TRIG_DMA_BY_EPWMSOCA    1

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */


/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
volatile uint32_t done0, done1;
volatile uint32_t idx0, idx1;

/* This is the source data for modified PWM duty, totaling 8 groups with 16 data each */
__attribute__((aligned(8))) uint16 pwm1_duty[16*8] = {
    1000, 2000,  3000,  4000,  5000,  6000,  7000,  8000,
    9000, 10000, 11000, 12000, 13000, 14000, 15000, 16000,
    1000, 2000,  3000,  4000,  5000,  6000,  7000,  8000,
    9000, 10000, 11000, 12000, 13000, 14000, 15000, 16000,
    1000, 2000,  3000,  4000,  5000,  6000,  7000,  8000,
    9000, 10000, 11000, 12000, 13000, 14000, 15000, 16000,
    1000, 2000,  3000,  4000,  5000,  6000,  7000,  8000,
    9000, 10000, 11000, 12000, 13000, 14000, 15000, 16000,
    1000, 2000,  3000,  4000,  5000,  6000,  7000,  8000,
    9000, 10000, 11000, 12000, 13000, 14000, 15000, 16000,
    1000, 2000,  3000,  4000,  5000,  6000,  7000,  8000,
    9000, 10000, 11000, 12000, 13000, 14000, 15000, 16000,
    1000, 2000,  3000,  4000,  5000,  6000,  7000,  8000,
    9000, 10000, 11000, 12000, 13000, 14000, 15000, 16000,
    1000, 2000,  3000,  4000,  5000,  6000,  7000,  8000,
    9000, 10000, 11000, 12000, 13000, 14000, 15000, 16000,
};

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

void initEPWM(uint32_t base);
//void initEPWM(void);
//void EPWM1XINT_IRQHandler(void);
void Plu0_Init(void);
void Plu0_OutputCfg(void);
void Test_DmaInit(void);
void CpuTimer_Init(uint32_t base, uint32_t period);

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/**
 * \brief   initEPWM - Initialize PWM output 5 KHz square wave signal and configure EPWM1SOCA signal
 *
 * \param   none
 *
 * \retval  None
 */
void initEPWM(uint32_t base)
{
    EPWM_setEmulationMode(base, EPWM_EMULATION_STOP_AFTER_FULL_CYCLE);

    //
    // Set-up TBCLK
    //
    EPWM_setTimeBasePeriod(base, EPWM_TIMER_TBPRD);
    EPWM_setPhaseShift(base, 0U);
    EPWM_setTimeBaseCounter(base, 0U);

    //
    // Set Compare values
    //
    EPWM_setCounterCompareValue(base,
                                EPWM_COUNTER_COMPARE_A,
                                EPWM_TIMER_TBPRD/2);
    EPWM_setCounterCompareValue(base,
                                EPWM_COUNTER_COMPARE_B,
                                EPWM_TIMER_TBPRD/2);

    //
    // Set up counter mode
    //
    EPWM_setTimeBaseCounterMode(base, EPWM_COUNTER_MODE_UP);
    EPWM_disablePhaseShiftLoad(base);
    EPWM_setClockPrescaler(base,
                           EPWM_CLOCK_DIVIDER_64,
                           EPWM_HSCLOCK_DIVIDER_1);

    //
    // Set up shadowing
    //
    EPWM_setCounterCompareShadowLoadMode(base,
                                         EPWM_COUNTER_COMPARE_A,
                                         EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setCounterCompareShadowLoadMode(base,
                                         EPWM_COUNTER_COMPARE_B,
                                         EPWM_COMP_LOAD_ON_CNTR_ZERO);

    //
    // Set actions
    //

    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);


    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_B,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);

    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_B,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);

    //
    // Interrupt where we will change the Compare Values
    // Select INT on Time base counter zero event,
    // Enable INT, generate INT on 1st event
    //
    EPWM_setInterruptSource(base, EPWM_INT_TBCTR_ZERO);
    EPWM_enableInterrupt(base);
    EPWM_setInterruptEventCount(base, 1U);

    EPWM_enableADCTrigger(base, EPWM_SOC_A);
    EPWM_setADCTriggerSource(base,
                             EPWM_SOC_A,
                             EPWM_SOC_TBCTR_ZERO);
    EPWM_setADCTriggerEventPrescale(base,
                                    EPWM_SOC_A,
                                       1);
    EPWM_clearADCTriggerFlag(base,
                             EPWM_SOC_A);
    EPWM_enableADCTrigger(base, EPWM_SOC_B);
    EPWM_setADCTriggerSource(base,
                             EPWM_SOC_B,
                             EPWM_SOC_TBCTR_ZERO);
    EPWM_setADCTriggerEventPrescale(base,
                                    EPWM_SOC_B,
                                       1);
    EPWM_clearADCTriggerFlag(base,
                             EPWM_SOC_B);
}


void DMA_CH1_IRQHandler()
{
    if (idx0 >= 40000 * 100)
    {
        done0 = 1;
        DMA_stopChannel(DMA_CH1_BASE);
    }
    idx0 ++;
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);
    __DSB();
}

void DMA_CH2_IRQHandler()
{
    if (idx1 >= 40000 * 100)
    {
        done1 = 1;
        DMA_stopChannel(DMA_CH2_BASE);
    }
    idx1 ++;

    DMA_clearInterrupt(DMA_CH2_BASE, DMA_INT_BLOCK);
    __DSB();
}


void Test_DmaInit(void){

    DMA_initController(DMA_BASE);

    dmaCfg.enableInterrupt = 1;
    dmaCfg.srcAddr = (uint32_t)(&pwm1_duty[0]);
    dmaCfg.destAddr = Test_EPWM + EPWM_O_CMPA + 2;
    dmaCfg.blockTS = 128;
    dmaCfg.ttfc    = DMA_TT_FC_1_M2P_DMAC;//DMA_TT_FC_1_M2P_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_1;
    dmaCfg.destBtl = DMA_BTL_1;
    dmaCfg.srcAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = DMA_ADDR_NO_CHANGE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dmaCfg.destTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dmaCfg.reloadSrc=1;
    dmaCfg.reloadDst=0;


    dmaCfg.dmaDstReqId = DMAMUX_ReqId_PWMA;
    DMA_configChannel(DMA_CH1_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);


    dmaCfg.destAddr = Test_EPWM + EPWM_O_CMPB + 2;
    dmaCfg.dmaDstReqId = DMAMUX_ReqId_PWMB;
    DMA_configChannel(DMA_CH2_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH2_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH2_BASE, DMA_INT_BLOCK);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
#if __arm__
#define DMA_INT_CH0_IRQn        (INT_DMA_CH0)
#define DMA_INT_CH1_IRQn        (INT_DMA_CH1)
#else
#define DMA_INT_CH1_IRQn        (INT_DMA1_CH1)
#define DMA_INT_CH2_IRQn        (INT_DMA1_CH2)
#endif
 int main(void)
{
    int i,j, dir=0;
#if ((GS32F3xx & 0xFF00) || (GS32F00xx & 0xFF00)) 
    //
    // Initialize device clock and peripherals
    //
    Device_init();
#endif

    //
    // Disable global interrupts.
    //
    __disable_irq();

#if ((GS32F3xx & 0xFF) || (GS32F00xx & 0xFF))
    //
    // config arm clock different from RISCV
    //
    HWREG(CRG_CFG_BASE + 0x008) = 1;        //Slow 2 normal
    HWREG(CRG_CFG_BASE + 0x010) = 0x0;      //arm clk div = 0
    HWREG(CRG_CFG_BASE + 0x01C) = 1;        //update
#endif

    idx0 = 0;
    idx1 = 0;
    done0 = 0;
    done1 = 0;
    UartPrint_init(LOG_SCI_BASE, 115200);

    // STOP_ON_RESET = 0 => start DSP core.
    //(*(uint32_t *)(0x40370048)) = 0x00;

    printf("NOTE::::: case go here\n");

#if ((GS32F3xx & 0xFF) || (GS32F00xx & 0xFF))
    //
    // config for pwm output
    //
    //HWREG(0x4003F000 + 0x120) = 0x10000;
    HWREG(0x4003F000 + 0x44) = 0x40000;
    HWREG(0x40008000 + 0x31C) = 0x3;
#endif

    Board_init();

    //
    // Enable specific PIE & CPU interrupts:
    // DMA interrupt - Group 7, interrupt 1
    //
    Interrupt_register(DMA_INT_CH1_IRQn, &DMA_CH1_IRQHandler);
    Interrupt_register(DMA_INT_CH2_IRQn, &DMA_CH2_IRQHandler);
    Interrupt_enable(DMA_INT_CH1_IRQn);
    Interrupt_enable(DMA_INT_CH2_IRQn);
    //
    // Setup ePWM1 to generate a 5 kHz signal to be an input to the eQEP
    //
    initEPWM(Test_EPWM);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
    //CpuTimer_Init(CPUTIMER0_BASE, 2000000);        //100ms @ 20MHz

    Test_DmaInit();

    //
    // Enable global intertupts
    //
    __enable_irq();

    DMA_startChannel(DMA_CH1_BASE);
    DMA_startChannel(DMA_CH2_BASE);

    while(done0 == 0 || done1 == 0);

    while(1) {}
}



#ifdef __cplusplus
}
#endif
/*****END OF FILE****/

