/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    dma_ex11_sci_autoback_with_timeout_burst_board.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "dma_ex11_sci_autoback_with_timeout_burst_board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */


__IO ComTxRxType ComTxRx = {0};
__IO Uint16 Sci_Rx_Flag = {0};
Uint16 Rx_count_Pr = 0U;
Uint16 Rx_Length = 0U;
DMA_ConfigParams ch4dmaCfg = {0};
DMA_ConfigParams ch5dmaCfg = {0};
#if (defined(LOAD_MONITOR) && LOAD_MONITOR)
__IO DLOAD_STRUCT    dload;
#endif

char *startinfo = "123456789abcdefghjklmnopqrst=============================================>>>>\n";
/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */


//*****************************************************************************
//
// Board Configurations
// Initializes the rest of the modules. 
// Call this function in your application if you wish to do all module 
// initialization.
// If you wish to not use some of the initializations, instead of the 
// Board_init use the individual Module_inits
//
//*****************************************************************************
void Board_init()
{
    EALLOW;    
    PinMux_init(); 
    EDIS;
}

// PINMUX Configurations
void PinMux_init()
{
#if (SCI_BASE == SCIA_BASE)
    GPIO_setPadConfig(SCIA_TX_GPIO, GPIO_PIN_TYPE_PULLUP);
    GPIO_setPadConfig(SCIA_RX_GPIO, GPIO_PIN_TYPE_PULLUP);
    GPIO_setPinConfig(SCIA_TX_PIN);
    GPIO_setPinConfig(SCIA_RX_PIN);
#else
    GPIO_setPinConfig(SCIB_TX_PIN);
    GPIO_setPinConfig(SCIB_RX_PIN);
#endif
}


//
// configureDMAChannels - Initialize DMA ch 0 to transfer ADC results
//                        
void DMA_CHN_Modeconfig(void)
{
    //
    // Define for SCI-TX-DMA
    //    
    ch4dmaCfg.enableInterrupt = 1;

    ch4dmaCfg.srcAddr = (Uint32)(ComTxRx.Buf);
    ch4dmaCfg.destAddr = (Uint32)(SCI_BASE + SCI_O_RBR_THR_DLL);
    ch4dmaCfg.blockTS =  MAX_COM_BUF_LEN;
    ch4dmaCfg.reloadSrc = FALSE;
    ch4dmaCfg.reloadDst = FALSE;
    ch4dmaCfg.ttfc    = DMA_TT_FC_1_M2P_DMAC;
    ch4dmaCfg.srcBtl  = DMA_BTL_1;
    ch4dmaCfg.destBtl = DMA_BTL_1;
    ch4dmaCfg.srcAddrDirect = DMA_ADDR_INCRE;
    ch4dmaCfg.destAddrDirect = DMA_ADDR_NO_CHANGE;
    ch4dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_1;
    ch4dmaCfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_1;
    ch4dmaCfg.dmaDstReqId = DMAMUX_ReqId_TX;
    
    DMA_configChannel(DMA_CH4_BASE, &ch4dmaCfg);
    DMA_clearInterrupt(DMA_CH4_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH4_BASE, DMA_INT_TFR);

    //
    // Define for SCI-RX-DMA
    //    
    ch5dmaCfg.enableInterrupt = 1;

    ch5dmaCfg.srcAddr = (Uint32)(SCI_BASE + SCI_O_RBR_THR_DLL);
    ch5dmaCfg.destAddr = (Uint32)ComTxRx.Buf;
    ch5dmaCfg.blockTS = MAX_COM_BUF_LEN;
    ch5dmaCfg.reloadSrc = FALSE;
    ch5dmaCfg.reloadDst = FALSE;
    ch5dmaCfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;
    ch5dmaCfg.srcBtl  = DMA_BTL_1;
    ch5dmaCfg.destBtl = DMA_BTL_1;
    ch5dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_1;
    ch5dmaCfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_1;
    ch5dmaCfg.srcAddrDirect = DMA_ADDR_NO_CHANGE;
    ch5dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    ch5dmaCfg.dmaSrcReqId = DMAMUX_ReqId_RX;
    
    DMA_configChannel(DMA_CH5_BASE, &ch5dmaCfg);
    DMA_clearInterrupt(DMA_CH5_BASE, DMA_INT_SRCTRAN);
    DMA_unMaskInterrupt(DMA_CH5_BASE, DMA_INT_SRCTRAN);
}

CCMFUNC_T void sDMA_configChannel(uint32_t base, const DMA_ConfigParams *transfParams)
{
    Uint32 high = 0;

    //
    // Set up SOURCE address.
    //
    HWREG(base + DMA_O_SAR) = (uint32_t)transfParams->srcAddr;

    //
    // Set up DESTINATION address.
    //
    HWREG(base + DMA_O_DAR) = (uint32_t)transfParams->destAddr;

    
    /* DMA channel key configuration - channel CTRL register */

#if (DMAC_VERSION == 0x30)
    //high = (1<<15); /* bit47, DONE */
    high += (transfParams->blockTS & 0x00007fff);   /* Bit32~Bit46, Block_Ts max is 32767  */
#else
    //high = (1<<12); /* bit44, DONE */
    high += (transfParams->blockTS & 0x00000fff);   /* Bit32~x, Block_Ts max is 4095  */
#endif


    DMAREGH(base, DMA_O_CTL) = high;
}


void InitCpuTimer(uint32_t base, uint32_t period)
{
    if(period > 0xFFFF){
        CPUTimer_setTimerSize(base, CPUTIMER_TIMERSIZE_32BIT);
    } else {
        CPUTimer_setTimerSize(base, CPUTIMER_TIMERSIZE_16BIT);
    }

    CPUTimer_setTimerMode(base, CPUTIMER_TIMERMODE_PERIODIC);
    //
    // Set the timer period
    //
    CPUTimer_setPeriod(base, period-1U);
    //
    // The enabling timer is interrupted
    //
    CPUTimer_enableInterrupt(base);
}

CCMFUNC_T void sTimeBaseIsr(void)
{
    Uint16 i = 0;
    Uint32 curBlockTs = 0U;
    
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)
    volatile uint32_t val1 = 0U;
    volatile uint32_t val2 = 0U;
    cFlash_Cache_clearStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    cFlash_Cache_clearStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);

    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);
#endif    
#if (defined(LOAD_MONITOR) && LOAD_MONITOR)
    dload.timer0_isr.dlog_load_start = CPU_getCycleCnt();
#endif
    //
    // Clear over flow flag.
    //
    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    //HWREG(CPUTIMER0_BASE + CPUTIMER_O_TIMERXINTCLR) |= CPUTIMER_TIMERXINTCLR;

    if(Sci_Rx_Flag == 1)
    {
        
        if (ComTxRx.RxDelay > 0)
        {
            ComTxRx.RxDelay--;
        }        

        if (ComTxRx.RxDelay < 1 || ComTxRx.Error == 1)
        {
            DMA_stopChannel(DMA_CH5_BASE);
            //
            // Beyond the receiving time, reset the receiving timeout.
            //
            ComTxRx.pRx = 0;  
            ComTxRx.Error = 0;
            ComTxRx.RxDelay = COM_TIMEOUT;
            Sci_Rx_Flag = 0;

            Rx_count_Pr = MAX_COM_BUF_LEN;
            //
            // Prepare the UART RX_DMA before start channel.
            // Please don't start channel here before uart RX interrupt.
            //
            sDMA_configChannel(DMA_CH5_BASE, &ch5dmaCfg);
			DMA_startChannel(DMA_CH5_BASE);
            ComTxRx.State = UART_RX;
        }
        else
        {
            curBlockTs = DMAREGH(DMA_CH5_BASE, DMA_O_CTL);
                
            if (Rx_count_Pr != curBlockTs)
                Rx_count_Pr = curBlockTs;
            else
            {
                Sci_Rx_Flag = 0;
                Rx_Length = curBlockTs;
                DMA_stopChannel(DMA_CH5_BASE);

                //
                // To do: check the Msg 
                //
                {
                    // Send back the mas 
                    ch4dmaCfg.blockTS = Rx_Length;
                    sDMA_configChannel(DMA_CH4_BASE, &ch4dmaCfg);
                    DMA_startChannel(DMA_CH4_BASE);
                }                
            }
        }
    }

#if (defined(LOAD_MONITOR) && LOAD_MONITOR)      
    dload.timer0_isr.dlog_load_stop = CPU_getCycleCnt();
    dload.timer0_isr.dlog_loads = dload.timer0_isr.dlog_load_stop - dload.timer0_isr.dlog_load_start;
#if LOAD_MONITOR_AVR         
    if (dload.timer0_isr.dlog_load_max < dload.timer0_isr.dlog_loads)
        dload.timer0_isr.dlog_load_max = dload.timer0_isr.dlog_loads;       
    dload.timer0_isr.dlog_load_aver = 
         ((dload.timer0_isr.dlog_load_aver * dload.timer0_isr.dlog_load_cnt) + dload.timer0_isr.dlog_loads) /
         (dload.timer0_isr.dlog_load_cnt++ + 1);
#endif 
#endif      
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)    
    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);    
#endif  
}


void DMA_CH5_IRQHandler(void)
{
    DMA_clearInterrupt(DMA_CH5_BASE, DMA_INT_TFR | DMA_INT_SRCTRAN);
    //
    // Stop dma interruption.
    //
    DMA_maskInterrupt(DMA_CH5_BASE, DMA_INT_SRCTRAN);
    // Record the flag and timeout for SCI-RX DMA procesxs.
    Sci_Rx_Flag = 1U;
    ComTxRx.RxDelay = 3000U;
}

void DMA_CH4_IRQHandler(void)
{
	DMA_clearInterrupt(DMA_CH4_BASE, DMA_INT_TFR);

	DMA_stopChannel(DMA_CH4_BASE);

    ComTxRx.pRx = 0;
    ComTxRx.State = UART_RX;

    Rx_count_Pr = MAX_COM_BUF_LEN;

    DMA_stopChannel(DMA_CH5_BASE);
    sDMA_configChannel(DMA_CH5_BASE, &ch5dmaCfg);
    DMA_unMaskInterrupt(DMA_CH5_BASE, DMA_INT_SRCTRAN);
    DMA_startChannel(DMA_CH5_BASE);
}

void Uart_Init(void)
{

    //
    // Init the SCIB controler
    //
    SCI_setBaud(SCI_BASE, DEVICE_APBCLK_FREQ, 115200);
    SCI_setInitConfig(SCI_BASE, SCI_NO_PARITY, SCI_STOP_1BIT, SCI_CHAR_8BITS);

    // Enable FIFO and set water-mark
    SCI_enableFIFO(SCI_BASE);
    SCI_setFIFOInterruptLevel(SCI_BASE, SCI_FIFO_TX0, SCI_FIFO_RX1);
}


#ifdef __cplusplus
}
#endif

