/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    main.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdio.h>
#include "device.h"
#include "driverlib.h"

#include "eqep_ex3_calculation.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define eqep_ex3_ans_size  5

//#define TB_CLK    DEVICE_SYSCLK_FREQ         // ePWM Time base clock is SYSCLK/2
#if IS_GS32F00xx(0x12)
#define TB_CLK    DEVICE_SYSCLK_FREQ
#elif IS_GS32F3xx(0x22)
#define TB_CLK    DEVICE_SYSCLK_FREQ/2
#endif
#define PWM_CLK   5000                       // We want to output at 5 kHz
#define PRD_VAL   (TB_CLK / (PWM_CLK * 2))  // Calculate value period value
                                            // for up-down count mode
#define BASE_FREQ       10000  // Base/max frequency is 10 kHz
// See Equation 5 in eqep_ex1_calculation.c
#define FREQ_SCALER_PR  (((TB_CLK / 128) * 256) / (2 * BASE_FREQ))
// See Equation 2 in eqep_ex1_calculation.c
#define FREQ_SCALER_FR  ((BASE_FREQ * 2) / 100)


//
// EQEP1 -> myEQEP0 Pinmux
//
//
// EQEP1_A - GPIO Settings
//
// EQEP1_A - GPIO Settings
//
#define GPIO_PIN_EQEP1_A 10
#define myEQEP0_EQEPA_GPIO 10
#define myEQEP0_EQEPA_PIN_CONFIG GPIO_10_EQEP1_A
//
// EQEP1_B - GPIO Settings
//
#define GPIO_PIN_EQEP1_B 11
#define myEQEP0_EQEPB_GPIO 11
#define myEQEP0_EQEPB_PIN_CONFIG GPIO_11_EQEP1_B

//
// EQEP1_INDEX - GPIO Settings
//
#define GPIO_PIN_EQEP1_INDEX 23
#define myEQEP0_EQEPINDEX_GPIO 23
#define myEQEP0_EQEPINDEX_PIN_CONFIG GPIO_23_EQEP1_INDEX

//*****************************************************************************
//
// EQEP Configurations
//
//*****************************************************************************
#define myEQEP0_BASE EQEP1_BASE

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
FreqCal_Object eqep_ex3_freq =
{
    FREQ_SCALER_PR,  // freqScalerPR
    FREQ_SCALER_FR,  // freqScalerFR
    BASE_FREQ,       // baseFreq
    0, 0, 0, 0, 0    // Initialize outputs to zero
};

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

uint32_t eqep_ex3_count =0;  // counter to check measurement gets saturated
uint32_t eqep_ex3_pass=0, eqep_ex3_fail =0; // Pass or fail indicator

//Temprory

int eqep_ex3_ans[eqep_ex3_ans_size];
int eqep_ex3_fre[200]={0};

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
void eqep_ex3_myEQEP0_init();
void eqep_ex3_initEPWM(void);
void EPWM1XINT_IRQHandler(void);
void eqep_ex3_GPIO_Init();
/**
  * \brief  Main program
  * \param  None
  * \retval None
  */
int main(void)
{
     //
     // Initialize device clock and peripherals
     //
	Device_init();

    //
    // Disable pin locks and enable internal pullups.
    //
	eqep_ex3_GPIO_Init();


    //
    // Board Initialization
    // Setup eQEP1, configuring the unit timer and quadrature capture units
    //
    eqep_ex3_myEQEP0_init();


    //
    // Setup ePWM1 to generate a 5 kHz signal to be an input to the eQEP
    //

    eqep_ex3_initEPWM();

    //epeq1a
    XBAR_enableEPWMMux(XBAR_TRIP4, XBAR_MUX01);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT1, 0);
    XBAR_setEPWMMuxConfig(XBAR_TRIP4, XBAR_EPWM_MUX01_INPUTXBAR1);


    Interrupt_register(INT_EPWM1,EPWM1XINT_IRQHandler);
	Interrupt_SetPriority(INT_EPWM1,2,2);
	Interrupt_enable(INT_EPWM1);
	__enable_irq();

    //
    // Enable Global Interrupt (INTM) and realtime interrupt (DBGM)
    //
		#if 0
    EINT;
    ERTM;
		#endif

    //
    // Setup eQEP1, configuring the unit timer and quadrature capture units
    //

    //
    // Loop indefinitely
    //

    while(1)
    {
        ;
    }

}

void eqep_ex3_GPIO_Init()
{
	//
	// EPWM1 -> myEPWM1 Pinmux
	//
	GPIO_setPinConfig(GPIO_0_EPWM1_A);
	GPIO_setPadConfig(0, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(0, GPIO_QUAL_SYNC);
	GPIO_setLoopBack(0,true);

	//eqep1A
	GPIO_setPinConfig(myEQEP0_EQEPA_PIN_CONFIG);
	GPIO_setPadConfig(myEQEP0_EQEPA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEQEP0_EQEPA_GPIO, GPIO_QUAL_SYNC);
    //eqep1B
	GPIO_setPinConfig(myEQEP0_EQEPB_PIN_CONFIG);
	GPIO_setPadConfig(myEQEP0_EQEPB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEQEP0_EQEPB_GPIO, GPIO_QUAL_SYNC);
	//eqep1I
	GPIO_setPinConfig(myEQEP0_EQEPINDEX_PIN_CONFIG);
	GPIO_setPadConfig(myEQEP0_EQEPINDEX_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEQEP0_EQEPINDEX_GPIO, GPIO_QUAL_SYNC);
}


void eqep_ex3_myEQEP0_init()
{
	EQEP_SourceSelect source_myEQEP0 =
	{
		EQEP_SOURCE_PWMXBAR1, 		// eQEPA source
		EQEP_SOURCE_DEVICE_PIN,		// eQEPB source
		EQEP_SOURCE_DEVICE_PIN,  	// eQEP Index source
	};

	//
	// Selects the source for eQEPA/B/I signals
	//
	EQEP_selectSource(myEQEP0_BASE, source_myEQEP0);
	//
	// Set the strobe input source of the eQEP module.
	//
	EQEP_setStrobeSource(myEQEP0_BASE,EQEP_STROBE_FROM_GPIO);
	//
	// Sets the polarity of the eQEP module's input signals.
	//
	EQEP_setInputPolarity(myEQEP0_BASE,false,false,false,false);
	//
	// Configures eQEP module's quadrature decoder unit.
	//
	EQEP_setDecoderConfig(myEQEP0_BASE, (EQEP_CONFIG_UP_COUNT | EQEP_CONFIG_2X_RESOLUTION | EQEP_CONFIG_NO_SWAP | EQEP_CONFIG_IGATE_DISABLE));
	//
	// Set the emulation mode of the eQEP module.
	//
	EQEP_setEmulationMode(myEQEP0_BASE,EQEP_EMULATIONMODE_STOPIMMEDIATELY);
	//
	// Configures eQEP module position counter unit.
	//
	EQEP_setPositionCounterConfig(myEQEP0_BASE,EQEP_POSITION_RESET_IDX,4294967295U);
	//
	// Sets the current encoder position.
	//
	EQEP_setPosition(myEQEP0_BASE,0U);
	//
	// Enables the eQEP module unit timer.
	//
	EQEP_enableUnitTimer(myEQEP0_BASE,TB_CLK/100);
	//
	// Disables the eQEP module watchdog timer.
	//
	EQEP_disableWatchdog(myEQEP0_BASE);
	//
	// Configures the quadrature modes in which the position count can be latched.
	//
	EQEP_setLatchMode(myEQEP0_BASE,(EQEP_LATCH_UNIT_TIME_OUT|EQEP_LATCH_RISING_STROBE|EQEP_LATCH_RISING_INDEX));
	//
	// Set the quadrature mode adapter (QMA) module mode.
	//
	EQEP_setQMAModuleMode(myEQEP0_BASE,EQEP_QMA_MODE_BYPASS);
	//
	// Disable Direction Change During Index
	//
	EQEP_disableDirectionChangeDuringIndex(myEQEP0_BASE);
	//
	// Configures the mode in which the position counter is initialized.
	//
	EQEP_setPositionInitMode(myEQEP0_BASE,(EQEP_INIT_DO_NOTHING));
	//
	// Sets the software initialization of the encoder position counter.
	//
	EQEP_setSWPositionInit(myEQEP0_BASE,true);
	//
	// Sets the init value for the encoder position counter.
	//
	EQEP_setInitialPosition(myEQEP0_BASE,0U);
	//
	// Enables the eQEP module.
	//
	EQEP_enableModule(myEQEP0_BASE);
	//
	// Configures eQEP module edge-capture unit.
	//
	EQEP_setCaptureConfig(myEQEP0_BASE,EQEP_CAPTURE_CLK_DIV_128,EQEP_UNIT_POS_EVNT_DIV_256);
	//
	// Enables the eQEP module edge-capture unit.
	//
	EQEP_enableCapture(myEQEP0_BASE);
}

void eqep_ex3_initEPWM(void)
{

    //
    // Set phase shift to 0 and clear the time base counter
    //
    EPWM_setPhaseShift(EPWM1_BASE, 0);
    EPWM_setTimeBaseCounter(EPWM1_BASE, 0);

    //
    // Disable the shadow load; the load will be immediate instead
    //
    EPWM_disableCounterCompareShadowLoadMode(EPWM1_BASE,
                                             EPWM_COUNTER_COMPARE_A);
    EPWM_disableCounterCompareShadowLoadMode(EPWM1_BASE,
                                             EPWM_COUNTER_COMPARE_B);

    //
    // Set the compare A value to half the period value, compare B to 0
    //
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, PRD_VAL/2);
  //  EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_B, 0);

    //
    // Set action qualifier behavior on compare A events
    // - EPWM1A --> 1 when CTR = CMPA and increasing
    // - EPWM1A --> 0 when CTR = CMPA and decreasing
    //
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);

    //
    // Configure EPWM1B to be complementary to EPWM1A
    //
    EPWM_setDeadBandDelayPolarity(EPWM1_BASE, EPWM_DB_FED,
                                  EPWM_DB_POLARITY_ACTIVE_LOW);
    EPWM_setDeadBandDelayMode(EPWM1_BASE, EPWM_DB_FED, true);
    EPWM_setDeadBandDelayMode(EPWM1_BASE, EPWM_DB_RED, true);

    //
    // Enable interrupt when the counter is equal to 0
    //
    EPWM_setInterruptSource(EPWM1_BASE, EPWM_INT_TBCTR_ZERO);
    EPWM_enableInterrupt(EPWM1_BASE);

    //
    // Interrupt on first event
    //
    EPWM_setInterruptEventCount(EPWM1_BASE, 1);

    //
    // Set the time base clock prescaler to /1
    //
    EPWM_setClockPrescaler(EPWM1_BASE, EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);

    //
    // Set the period value; don't shadow the register
    //
    EPWM_setPeriodLoadMode(EPWM1_BASE, EPWM_PERIOD_DIRECT_LOAD);
    EPWM_setTimeBasePeriod(EPWM1_BASE, PRD_VAL);

    //
    // Put the time base counter into up-down count mode
    //
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP_DOWN);

    //
    // Sync the ePWM time base clock
    //
	SysCtl_enablePeripheral_TBCLKSYNC();

}


/**
 * \brief   ePWM1 ISR- interrupts once per ePWM period
 *
 * \param   none
 *
 * \retval  None
 */
void EPWM1XINT_IRQHandler(void)
{
	uint32_t i=0;
    //
    // Checks for events and calculates frequency.
    //
    eqep_ex3_FreqCal_calculate(&eqep_ex3_freq, &eqep_ex3_count);

    //
    // Comparing the eQEP measured frequency with the ePWM frequency
    // After count becomes 3 , eQEP measurement gets saturated and
    // classifying eqep_ex3_pass = 1 if measured frequency is 50 more or
    // less than input
    //
    if (eqep_ex3_count >= 3){

        if (((eqep_ex3_freq.freqHzFR - PWM_CLK) < 50) && ((eqep_ex3_freq.freqHzFR - PWM_CLK) > -50)){
            eqep_ex3_pass = 1; eqep_ex3_fail = 0;
          }
        else {
            eqep_ex3_fail = 1; eqep_ex3_pass = 0;
          }
    }

    //
    // Clear interrupt flag and issue ACK
    //
    EPWM_clearEventTriggerInterruptFlag(EPWM1_BASE);
}


#ifdef __cplusplus
}
#endif

/*****END OF FILE****/
