/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    main.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdio.h>
#include "device.h"


/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#if IS_GS32F00xx(0x12)
#define TB_CLK    DEVICE_SYSCLK_FREQ
#elif IS_GS32F3xx(0x22)
#define TB_CLK    DEVICE_SYSCLK_FREQ/2
#endif
#define PWM_CLK   5000                     // We want to output at 5 kHz
#define PRD_VAL   (TB_CLK / (PWM_CLK * 2))  // Calculate value period value
                                           // for up-down count mode
#define UNIT_PERIOD  10000U // Specify the period in microseconds
//Uncomment if motor is available for interfacing
//#define MOTOR

//
// EQEP1 -> myEQEP0 Pinmux
//
//
// EQEP1_A - GPIO Settings
//
//
// EQEP1_A - GPIO Settings
//
#define GPIO_PIN_EQEP1_A 10
#define myEQEP0_EQEPA_GPIO 10
#define myEQEP0_EQEPA_PIN_CONFIG GPIO_10_EQEP1_A
//
// EQEP1_B - GPIO Settings
//
#define GPIO_PIN_EQEP1_B 11
#define myEQEP0_EQEPB_GPIO 11
#define myEQEP0_EQEPB_PIN_CONFIG GPIO_11_EQEP1_B

//
// EQEP1_INDEX - GPIO Settings
//
#define GPIO_PIN_EQEP1_INDEX 23
#define myEQEP0_EQEPINDEX_GPIO 23
#define myEQEP0_EQEPINDEX_PIN_CONFIG GPIO_23_EQEP1_INDEX

//*****************************************************************************
//
// EQEP Configurations
//
//*****************************************************************************
#define myEQEP0_BASE EQEP1_BASE

// Interrupt Settings for INT_myEQEP0
#define INT_myEQEP0 INT_EQEP1
#define INT_myEQEP0_INTERRUPT_ACK_GROUP INTERRUPT_ACK_GROUP5


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

uint32_t eqep_ex5_oldcount = 0;  // stores the previous position counter value
int32_t eqep_ex5_freq = 0;   // measured quadrature signal frequency of motor using eQEP
float32_t speed = 0.0f;       // measured speed of motor in rpm
int32_t dir = 0;       // direction of rotation of motor
uint32_t eqep_ex5_count = 0;     // just to make sure measured frequency gets saturated
uint32_t eqep_ex5_pass = 0, eqep_ex5_fail = 0; // test pass or fail indicator


//Temprory
#define eqep_ex5_ans_size  16
int32_t eqep_ex5_ans[eqep_ex5_ans_size];
int32_t eqep_ex5_ans_freq[eqep_ex5_ans_size];
uint32_t eqep_ex5_ans_tempPoscnt[eqep_ex5_ans_size];


/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
void eqep_ex5_myEQEP0_init();
void eqep_ex5_initEPWM(void);
void EQEP1_INT_IRQHandler(void);
void eqep_ex5_GPIO_Init();
/**
  * \brief  Main program
  * \param  None
  * \retval None
  */
int main(void)
{
  //
    // Initialize device clock and peripherals
    //
	Device_init();

    //
    // Disable pin locks and enable internal pullups.
    //
	eqep_ex5_GPIO_Init();

    //
    // Initialize PIE and clear PIE registers. Disables CPU interrupts.
    //
    //Interrupt_initModule();

    //
    // Initialize the PIE vector table with pointers to the shell Interrupt
    // Service Routines (ISR).
    //
    //Interrupt_initVectorTable();

    eqep_ex5_myEQEP0_init();

    //
    // Initialize GPIOs for use as EPWM1A,EPWM1B and EQEP1A,
    // EQEPB
    //
    #if 0
    GPIO_setPinConfig(GPIO_0_EPWM1_A);
    GPIO_setPadConfig(0, GPIO_PIN_TYPE_STD);
    GPIO_setPinConfig(GPIO_1_EPWM1_B);
    GPIO_setPadConfig(1, GPIO_PIN_TYPE_STD);
    #endif

#ifndef MOTOR
    //
    // Setup ePWM1 to generate a 5 kHz signal to be an input to the eQEP
    //
    eqep_ex5_initEPWM();

#endif
#if (GS32F00xx == 0x0012 || GS32F00xx == 0x0011)
	//epeq1a
    XBAR_enableEPWMMux(XBAR_TRIP4, XBAR_MUX01);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT1, 0);
    XBAR_setEPWMMuxConfig(XBAR_TRIP4, XBAR_EPWM_MUX01_INPUTXBAR1);

	//epeq1b
    XBAR_enableEPWMMux(XBAR_TRIP5, XBAR_MUX03);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT2, 1);
    XBAR_setEPWMMuxConfig(XBAR_TRIP5, XBAR_EPWM_MUX03_INPUTXBAR2);
	//epeq1I
    XBAR_enableEPWMMux(XBAR_TRIP7, XBAR_MUX05);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT3, 2);
    XBAR_setEPWMMuxConfig(XBAR_TRIP7, XBAR_EPWM_MUX05_INPUTXBAR3);

#endif



	Interrupt_register(INT_EQEP1,EQEP1_INT_IRQHandler);
	Interrupt_SetPriority(INT_EQEP1,0,0);
	Interrupt_enable(INT_EQEP1);


__enable_irq();



    #if 0
    EINT;
    ERTM;
    #endif

    //
    // Loop indefinitely
    //

    while(1)
    {
        ;
    }

}

void eqep_ex5_GPIO_Init()
{
	//
	// EPWM1 -> myEPWM1 Pinmux
	//
	GPIO_setPinConfig(GPIO_0_EPWM1_A);
	GPIO_setPadConfig(0, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(0, GPIO_QUAL_SYNC);
	GPIO_setLoopBack(0,true);

	GPIO_setPinConfig(GPIO_1_EPWM1_B);
	GPIO_setPadConfig(1, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(1, GPIO_QUAL_SYNC);
	GPIO_setLoopBack(1,true);

	//eqep1A
	GPIO_setPinConfig(myEQEP0_EQEPA_PIN_CONFIG);
	GPIO_setPadConfig(myEQEP0_EQEPA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEQEP0_EQEPA_GPIO, GPIO_QUAL_SYNC);
    //eqep1B
	GPIO_setPinConfig(myEQEP0_EQEPB_PIN_CONFIG);
	GPIO_setPadConfig(myEQEP0_EQEPB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEQEP0_EQEPB_GPIO, GPIO_QUAL_SYNC);
	//eqep1I
	GPIO_setPinConfig(myEQEP0_EQEPINDEX_PIN_CONFIG);
	GPIO_setPadConfig(myEQEP0_EQEPINDEX_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEQEP0_EQEPINDEX_GPIO, GPIO_QUAL_SYNC);
}

void eqep_ex5_myEQEP0_init()
{
	//
	// Disable, clear all flags and interrupts
	//
	EQEP_disableInterrupt(myEQEP0_BASE,
		(EQEP_INT_GLOBAL     		|
		EQEP_INT_POS_CNT_ERROR		|
		EQEP_INT_PHASE_ERROR    	|
		EQEP_INT_DIR_CHANGE    		|
		EQEP_INT_WATCHDOG          	|
		EQEP_INT_UNDERFLOW         	|
		EQEP_INT_OVERFLOW        	|
		EQEP_INT_POS_COMP_READY    	|
		EQEP_INT_POS_COMP_MATCH   	|
		EQEP_INT_STROBE_EVNT_LATCH	|
		EQEP_INT_INDEX_EVNT_LATCH 	|
		EQEP_INT_UNIT_TIME_OUT   	|
		EQEP_INT_QMA_ERROR));
	EQEP_clearInterruptStatus(myEQEP0_BASE,
		(EQEP_INT_GLOBAL     		|
		EQEP_INT_POS_CNT_ERROR		|
		EQEP_INT_PHASE_ERROR    	|
		EQEP_INT_DIR_CHANGE    		|
		EQEP_INT_WATCHDOG          	|
		EQEP_INT_UNDERFLOW         	|
		EQEP_INT_OVERFLOW        	|
		EQEP_INT_POS_COMP_READY    	|
		EQEP_INT_POS_COMP_MATCH   	|
		EQEP_INT_STROBE_EVNT_LATCH	|
		EQEP_INT_INDEX_EVNT_LATCH 	|
		EQEP_INT_UNIT_TIME_OUT   	|
		EQEP_INT_QMA_ERROR));

	EQEP_SourceSelect source_myEQEP0 =
	{
		EQEP_SOURCE_DEVICE_PIN, 		// eQEPA source
		EQEP_SOURCE_DEVICE_PIN,		// eQEPB source
		EQEP_SOURCE_DEVICE_PIN,  	// eQEP Index source
	};
	//
	// Selects the source for eQEPA/B/I signals
	//
	EQEP_selectSource(myEQEP0_BASE, source_myEQEP0);
	//
	// Set the strobe input source of the eQEP module.
	//
	EQEP_setStrobeSource(myEQEP0_BASE,EQEP_STROBE_FROM_GPIO);
	//
	// Sets the polarity of the eQEP module's input signals.
	//
	EQEP_setInputPolarity(myEQEP0_BASE,false,false,false,false);
	//
	// Configures eQEP module's quadrature decoder unit.
	//
	EQEP_setDecoderConfig(myEQEP0_BASE, (EQEP_CONFIG_QUADRATURE | EQEP_CONFIG_1X_RESOLUTION | EQEP_CONFIG_NO_SWAP | EQEP_CONFIG_IGATE_DISABLE));
	//
	// Set the emulation mode of the eQEP module.
	//
	EQEP_setEmulationMode(myEQEP0_BASE,EQEP_EMULATIONMODE_RUNFREE);
	//
	// Configures eQEP module position counter unit.
	//
	EQEP_setPositionCounterConfig(myEQEP0_BASE,EQEP_POSITION_RESET_IDX,4294967295U);
	//
	// Sets the current encoder position.
	//
	EQEP_setPosition(myEQEP0_BASE,0U);
	//
	// Enables the eQEP module unit timer.
	//
	//EQEP_enableUnitTimer(myEQEP0_BASE,1200000U);
	EQEP_enableUnitTimer(myEQEP0_BASE,TB_CLK/100);
	//
	// Disables the eQEP module watchdog timer.
	//
	EQEP_disableWatchdog(myEQEP0_BASE);
	//
	// Configures the quadrature modes in which the position count can be latched.
	//
	//EQEP_setLatchMode(myEQEP0_BASE,(EQEP_LATCH_UNIT_TIME_OUT|EQEP_LATCH_RISING_STROBE|EQEP_LATCH_RISING_INDEX));
	EQEP_setLatchMode(myEQEP0_BASE,(EQEP_LATCH_UNIT_TIME_OUT));
	//
	// Set the quadrature mode adapter (QMA) module mode.
	//
	EQEP_setQMAModuleMode(myEQEP0_BASE,EQEP_QMA_MODE_BYPASS);
	//
	// Disable Direction Change During Index
	//
	EQEP_disableDirectionChangeDuringIndex(myEQEP0_BASE);
	//
	// Enables individual eQEP module interrupt sources.
	//
	EQEP_enableInterrupt(myEQEP0_BASE,(EQEP_INT_UNIT_TIME_OUT));
	//
	// Configures the mode in which the position counter is initialized.
	//
	EQEP_setPositionInitMode(myEQEP0_BASE,(EQEP_INIT_DO_NOTHING));
	//
	// Sets the software initialization of the encoder position counter.
	//
	EQEP_setSWPositionInit(myEQEP0_BASE,true);
	//
	// Sets the init value for the encoder position counter.
	//
	EQEP_setInitialPosition(myEQEP0_BASE,0U);
	//
	// Enables the eQEP module.
	//
	EQEP_enableModule(myEQEP0_BASE);
}

/**
 * \brief   eqep_ex5_initEPWM - Function to configure ePWM1 to generate a 5 kHz signal.
 *
 * \param   none
 *
 * \retval  None
 */
void eqep_ex5_initEPWM(void)
{
    //
    // Set phase shift to 0 and clear the time base counter
    //
    EPWM_setPhaseShift(EPWM1_BASE, 0);
    EPWM_setTimeBaseCounter(EPWM1_BASE, 0);

    //
    // Disable the shadow load; the load will be immediate instead
    //
    EPWM_disableCounterCompareShadowLoadMode(EPWM1_BASE,
                                                EPWM_COUNTER_COMPARE_A);
    EPWM_disableCounterCompareShadowLoadMode(EPWM1_BASE,
                                                EPWM_COUNTER_COMPARE_B);

    //
    // Set the compare A value to half the period value, compare B to 0
    //
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, PRD_VAL/2);
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_B, 0);

    //
    // Set action qualifier behavior on compare A events
    // - EPWM1A --> 1 when CTR = CMPA and increasing
    // - EPWM1A --> 0 when CTR = CMPA and decreasing
    //
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_A,
                                    EPWM_AQ_OUTPUT_HIGH,
                                    EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_A,
                                    EPWM_AQ_OUTPUT_LOW,
                                    EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);

    //
    // Set action qualifier behavior on compare B events
    // - EPWM1B --> 1 when CTR = PRD and increasing
    // - EPWM1B --> 0 when CTR = 0 and decreasing
    //
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_B,
                                    EPWM_AQ_OUTPUT_HIGH,
                                    EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(EPWM1_BASE, EPWM_AQ_OUTPUT_B,
                                    EPWM_AQ_OUTPUT_LOW,
                                    EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);

    //
    // Set the time base clock prescaler to /1
    //
    EPWM_setClockPrescaler(EPWM1_BASE, EPWM_CLOCK_DIVIDER_1,
                            EPWM_HSCLOCK_DIVIDER_1);

    //
    // Set the period value; don't shadow the register
    //
    EPWM_setPeriodLoadMode(EPWM1_BASE, EPWM_PERIOD_DIRECT_LOAD);
    EPWM_setTimeBasePeriod(EPWM1_BASE, PRD_VAL);

    //
    // Put the time base counter into up-down count mode
    //
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP_DOWN);



	SysCtl_enablePeripheral_TBCLKSYNC();


}


/**
 * \brief   eqep1 ISR- interrupts once per ePWM period
 *
 * \param   none
 *
 * \retval  None
 */
void EQEP1_INT_IRQHandler(void)
{
    uint32_t temp = 0, newcount = 0 ;
	  uint32_t tempPoscnt;

		//GPIO_writePin(1,1);

    eqep_ex5_count++; // increment count value so to wait for frequency saturation

    newcount = EQEP_getPositionLatch(myEQEP0_BASE); // new position counter value

		tempPoscnt = EQEP_getPosition(myEQEP0_BASE);

    temp = newcount;

		if (eqep_ex5_count >= 100 && eqep_ex5_count <116){
				eqep_ex5_ans[eqep_ex5_count-100] = temp;
				eqep_ex5_ans_freq[eqep_ex5_count-100] = eqep_ex5_freq;
				eqep_ex5_ans_tempPoscnt[eqep_ex5_count-100] = tempPoscnt;
		}else if(eqep_ex5_count >= 116){
			//for(; eqep_ex5_count <200; eqep_ex5_count++);
		}

   //
   // Gets direction of rotation of motor
   //
   dir = EQEP_getDirection(myEQEP0_BASE);

   //
   // Calculates the number of position in unit time based on
   // motor's direction of rotation
   //
   if (dir > 0 ){
       if (newcount >= eqep_ex5_oldcount)
            newcount = newcount - eqep_ex5_oldcount;
        else
            newcount = (0xFFFFFFFF - eqep_ex5_oldcount) + newcount;
        }
   else {
        if (newcount <= eqep_ex5_oldcount)
            newcount = eqep_ex5_oldcount - newcount;
        else
            newcount = (0xFFFFFFFF - newcount) + eqep_ex5_oldcount;
        }

   //
   // Stores the current position count value to eqep_ex5_oldcount variable
   //
   eqep_ex5_oldcount = temp;

   //
   // Simulated Frequency and speed calculation
   //
   eqep_ex5_freq = (newcount * (uint32_t)1000000U)/((uint32_t)UNIT_PERIOD);
   speed = (eqep_ex5_freq * 60)/4000.0f;

   //
   // Compares the measured quadrature simulated frequency with input quadrature
   // frequency and if difference is within the measurement resolution
   // (1/10ms = 100Hz) then eqep_ex5_pass = 1 else fail = 1
   //
#ifndef MOTOR
   if (eqep_ex5_count >= 2){
        if (((eqep_ex5_freq - PWM_CLK * 4) < 100) && ((eqep_ex5_freq - PWM_CLK * 4) > -100)){
            eqep_ex5_pass = 1; eqep_ex5_fail = 0;
        }
        else {
            eqep_ex5_fail = 1; eqep_ex5_pass = 0;
        }
    }
#endif


   //
   // Clear interrupt flag and issue ACK
   //
		EQEP_clearInterruptStatus(myEQEP0_BASE,EQEP_INT_UNIT_TIME_OUT|EQEP_INT_GLOBAL);
		#if 0
		Interrupt_clearACKGroup(INT_myEQEP0_INTERRUPT_ACK_GROUP);
		#endif

		//GPIO_writePin(1,0);
//		GPIO_disableWrite(0x0F0F);
 }




#ifdef __cplusplus
}
#endif

/*****END OF FILE****/
