/*
 * device_init.c
 *
 *  Created on: 2024
 *      Author: hpec
 */


#include <stdio.h>
#include <string.h>

#include "device.h"

#include "driverlib.h"


#include "parameter.h"
#include "device_init.h"

#include "epwm.h"

//struct ADC_RESULT_REGS *AdcaResult = (struct ADC_RESULT_REGS *)ADCARESULT_BASE;

//
// ADC constants
//
#define ADC_SAMPLE_PERIOD (DEVICE_SYSCLK_FREQ * SAMPLING_PERIOD)
#define PWM_PERIOD        (ADC_SAMPLE_PERIOD /(DEVICE_SYSCLK_FREQ/DEVICE_AHBCLK_FREQ) /2 /4)    /* sysclk to ahbclk; /2 -> updown mode; /4 -> EPWM_CLOCK_DIVIDER_4 */
#define ADC_ACQWIN 40

//
// PWM constants
//
//#pragma SET_DATA_SECTION("cla_shared")

uint32_t PWM_PERIOD_MAX = PWM_PERIOD;
uint32_t PWM_HALF_MAX = PWM_PERIOD/2;

void initADC()
{
	uint32_t value;

    //
    // Configure the ADC-A base registers
    //
    ADC_disableConverter(ADCA_BASE);                                    // Power down ADC for configuration
    //ADC_setVREF(ADCA_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_2_5V); // VREF internal 2.5V
    ADC_setVREF(ADCA_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_3_3V); // VREF internal 3.3V
    //ADC_setVREF(ADCA_BASE, ADC_REFERENCE_EXTERNAL, ADC_REFERENCE_3_3V); // VREF external 3.3V
    ADC_setINLTrim(ADCA_BASE);
    ADC_setPrescaler(ADCA_BASE, ADC_CLK_DIV_4_0);

    //
    // Configure INT pulse generation at end of acquisition (early interrupt)
    //
    ADC_setInterruptPulseMode(ADCA_BASE, ADC_PULSE_END_OF_ACQ_WIN);

    //
    // Power up ADC
    //
    ADC_enableConverter(ADCA_BASE);

    //
    // Wait 1 ms after power-up before using the ADC
    //
    DEVICE_DELAY_US(5000);

    //
    // SOC configuration - Trigger using ePWM1-ADCSOCA
    //
#if (GS32F00xx == 0x0011) //DAC->ADC loopback on FPGA
    ADC_setupSOC(ADCA_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_EPWM1_SOCA, ADC_CH_ADCIN10, ADC_ACQWIN); // config the channel from DAC1
    ADC_setupSOC(ADCA_BASE, ADC_SOC_NUMBER1, ADC_TRIGGER_EPWM1_SOCA, ADC_CH_ADCIN11, ADC_ACQWIN); // config the channel from DAC2
#else
    ADC_setupSOC(ADCA_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_EPWM1_SOCA, ADC_CH_ADCIN0, ADC_ACQWIN);
    ADC_setupSOC(ADCA_BASE, ADC_SOC_NUMBER1, ADC_TRIGGER_EPWM1_SOCA, ADC_CH_ADCIN1, ADC_ACQWIN);
#endif

    //
    // No ADC interrupt triggers SOC0 (TRIGSEL determined by SOC and not ADCINT1 or ADCINT2)
    //
    ADC_setInterruptSOCTrigger(ADCA_BASE, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);
    ADC_setInterruptSOCTrigger(ADCA_BASE, ADC_SOC_NUMBER1, ADC_INT_SOC_TRIGGER_NONE);

    //
    // All SOCs handled in round-robin mode
    //
    ADC_setSOCPriority(ADCA_BASE, ADC_PRI_ALL_ROUND_ROBIN);

    //
    // ADCA1 interrupt configuration
    //
    ADC_enableContinuousMode(ADCA_BASE, ADC_INT_NUMBER);                       // Interrupt pulses regardless of flag state
    ADC_setInterruptSource(ADCA_BASE, ADC_INT_NUMBER, ADC_SOC_NUMBER0);        // SOC0 triggers the interrupt
}

//*****************************************************************************
//
// Helper function to configure PWM
//
//*****************************************************************************
void initPWMx(uint32_t base)
{

    //
    // Configure PWM in up down mode
    //
//    EPWM_setSyncOutPulseMode(base,EPWM_SYNC_OUT_PULSE_ON_EPWMxSYNCIN);

	EPWM_setClockPrescaler(base, EPWM_CLOCK_DIVIDER_4, EPWM_HSCLOCK_DIVIDER_2);   // TBCLK = EPWMCLK
    //
    // Configure timer poeriod
    //
    EPWM_setTimeBasePeriod(base, PWM_PERIOD_MAX);
    EPWM_setTimeBaseCounter(base, 0);
    EPWM_setPhaseShift(base, 0);
    EPWM_setCounterCompareValue(base, EPWM_COUNTER_COMPARE_A, PWM_PERIOD_MAX);

    EPWM_enablePhaseShiftLoad(base);
    EPWM_setEmulationMode(base, EPWM_EMULATION_FREE_RUN);                         // Ignore emulation suspend
    EPWM_setPeriodLoadMode(base, EPWM_PERIOD_DIRECT_LOAD);
    EPWM_setTimeBaseCounterMode(base, EPWM_COUNTER_MODE_UP_DOWN);                 // Enable the timer in count up down mode

    EPWM_setCountModeAfterSync(base, EPWM_COUNT_MODE_UP_AFTER_SYNC);

    //
    // Set counter to reload on 0
    //
    EPWM_setCounterCompareShadowLoadMode(base, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setCounterCompareShadowLoadMode(base, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);

    //
    // Setup action qualifier
    //
    //EPWM_setActionQualifierActionComplete(base, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW_DOWN_CMPA | EPWM_AQ_OUTPUT_HIGH_UP_CMPA);

    EPWM_setActionQualifierAction(base, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(base, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
}



//*****************************************************************************
//
// Initialize the PWM modules (1, 2, 3)
//
//*****************************************************************************
void initPWM(void)
{
    //
    // Must disable the clock to the ePWM modules to have all ePWM modules synchronized
    //
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

	//SysCtl_Ahb_setSyncSelSyncOut(1);
	//SysCtl_Ahb_setSyncSelePWMSyncIn(2, 1);
	//SysCtl_Ahb_setSyncSelePWMSyncIn(3, 1);

    //
    // Initialize all PWMs
    //
    initPWMx(EPWM1_BASE);
    initPWMx(EPWM2_BASE);
    initPWMx(EPWM3_BASE);

    EPWM_enableSyncOutPulseSource(EPWM1_BASE, EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);
    EPWM_setSyncInPulseSource(EPWM1_BASE, EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM8);
    EPWM_setSyncInPulseSource(EPWM2_BASE, EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM1);
    EPWM_setSyncInPulseSource(EPWM3_BASE, EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM1);

//    EPWM_setInterruptSource(EPWM1_BASE, EPWM_INT_TBCTR_PERIOD);
//    EPWM_enableInterrupt(EPWM1_BASE);

    //
    // Enable PWM1 to trigger ADCA SOC
    //
    EPWM_enableADCTrigger(EPWM1_BASE, EPWM_SOC_A);                               // Enable ADC SOCA event
    EPWM_setADCTriggerSource(EPWM1_BASE, EPWM_SOC_A, EPWM_SOC_TBCTR_PERIOD);     // Set SOCA on PRD event
    EPWM_setADCTriggerEventPrescale(EPWM1_BASE, EPWM_SOC_A, 1);                  // Generate SOCA on first event
}

void disablePWM(void)
{
	EPWM_setActionQualifierActionComplete(EPWM1_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW_DOWN_CMPA | EPWM_AQ_OUTPUT_LOW_UP_CMPA);
	EPWM_setActionQualifierActionComplete(EPWM1_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_LOW_DOWN_CMPB | EPWM_AQ_OUTPUT_LOW_UP_CMPB);

	EPWM_setActionQualifierActionComplete(EPWM2_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW_DOWN_CMPA | EPWM_AQ_OUTPUT_LOW_UP_CMPA);
	EPWM_setActionQualifierActionComplete(EPWM2_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_LOW_DOWN_CMPB | EPWM_AQ_OUTPUT_LOW_UP_CMPB);

	EPWM_setActionQualifierActionComplete(EPWM3_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW_DOWN_CMPA | EPWM_AQ_OUTPUT_LOW_UP_CMPA);
	EPWM_setActionQualifierActionComplete(EPWM3_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_LOW_DOWN_CMPB | EPWM_AQ_OUTPUT_LOW_UP_CMPB);
}



//*****************************************************************************
//
// Initialize the DAC modules (A, B)
//
//*****************************************************************************
void initDAC(void)
{
#if defined(DACA_BASE) || defined(DACB_BASE)
    //
    // Set VREFHI as as the DAC reference voltage
    //
    DAC_setReferenceVoltage(DACB_BASE, DAC_REF_INTERNAL);
    DAC_setReferenceVoltage(DACA_BASE, DAC_REF_INTERNAL);

    // Set gain=2
    DAC_setGainMode(DACB_BASE, DAC_GAIN_TWO);
    DAC_setGainMode(DACA_BASE, DAC_GAIN_TWO);
    // Set gain=1
    //DAC_setGainMode(DACB_BASE, DAC_GAIN_ONE);
    //DAC_setGainMode(DACA_BASE, DAC_GAIN_ONE);

    //
    // Set load mode as load on next SYSCLK
    //
    DAC_setLoadMode(DACB_BASE, DAC_LOAD_SYSCLK);
    DAC_setLoadMode(DACA_BASE, DAC_LOAD_SYSCLK);

    //
    // Enable DAC output
    //
    DAC_enableOutput(DACB_BASE);                    // Enable DAC output
    DAC_enableOutput(DACA_BASE);                    // Enable DAC output
    DAC_enable(DACA_BASE);
    DAC_enable(DACB_BASE);

    //
    // Initialize DAC output to 0
    //
    DAC_setShadowValue(DACB_BASE, ADJUST_Ib_OUTPUT(0));
    DAC_setShadowValue(DACA_BASE, ADJUST_Ia_OUTPUT(0));

    //
    // Required delay after enabling the DAC (delay for DAC to power up)
    //
    DEVICE_DELAY_US(10);
#endif
}

//
// config the internal loop DAC to ADC
//
void initDAC2ADC()
{
#if (GS32F00xx == 0x0011) // enable the DAC->ADC loopback on FPGA
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE) |= (1<<4); // enable the DAC1 to ADC ch10, DAC2 to ADC ch11
	//ASysCtl_setAnaTopSpare(0, 4);
#endif
}

void Device_setup()
{
    // Initialize peripherals
    //
    initADC();
    initDAC2ADC();
    initDAC();
    initPWM();

    //GPIO0~GPIO5 are configured as EPWM outputs
    GPIO_setPinConfig(GPIO_0_EPWM1_A);
    GPIO_setPinConfig(GPIO_1_EPWM1_B);
    GPIO_setPinConfig(GPIO_2_EPWM2_A);
    GPIO_setPinConfig(GPIO_3_EPWM2_B);
    GPIO_setPinConfig(GPIO_4_EPWM3_A);
    GPIO_setPinConfig(GPIO_5_EPWM3_B);

    //GPIO32 is toggled for test purpose
    GPIO_setPinConfig(GPIO_32_GPIO32);
    GPIO_setDirectionMode(32, GPIO_DIR_MODE_OUT);

}
