/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    ecap_ex02_mointor_period.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "board_cfg.h"
#include "ecap_ex02_board.h"
#include "ecap_ex02_mointor_period.h"
#include "printf.h"
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define PWM1_TIMER_MIN     500U
#define PWM1_TIMER_MAX     8000U
#define EPWM_TIMER_UP      1U
#define EPWM_TIMER_DOWN    0U

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */


uint32_t ecap1IntCount;         // Interrupt counts
uint32_t ecap1PassCount;        // Successful captures
uint32_t ecap1ErrorCount;       // Error counts
uint32_t epwm1TimerDirection;
vuint16_t cap2Count;
vuint16_t cap3Count;
vuint16_t cap4Count;
vuint16_t epwm1PeriodCount;
uint16_t PeriodFine=0;


/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/**
 * \brief   "Error function"
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void error()
{
    ecap1ErrorCount++;
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/**
 * \brief   // initEPWM - Configure ePWM
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void initEPWM()
{
    // Disable sync(Freeze clock to PWM as well)
    //
    // SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    // Configure ePWM
    //       Counter runs in up-count mode.
    //       Action qualifier will toggle output on period match
    //
    EPWM_setClockPrescaler(EPWM1_BASE,
                           EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);
	EPWM_setTimeBasePeriod(EPWM1_BASE, PWM_PRD_VAL);
	EPWM_setTimeBaseCounter(EPWM1_BASE,0U);
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP);
    EPWM_disablePhaseShiftLoad(EPWM1_BASE);
    EPWM_setPhaseShift(EPWM1_BASE, 0U);
	EPWM_setCounterCompareValue(EPWM1_BASE,EPWM_COUNTER_COMPARE_A,PWM_CMP_VAL);
	EPWM_setCounterCompareShadowLoadMode(EPWM1_BASE,EPWM_COUNTER_COMPARE_A,EPWM_COMP_LOAD_ON_CNTR_ZERO);
	EPWM_enableSyncOutPulseSource(EPWM1_BASE,EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);
    EPWM_setActionQualifierAction(EPWM1_BASE,
                                  EPWM_AQ_OUTPUT_A,
								  EPWM_AQ_OUTPUT_TOGGLE,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
	EPWM_setActionQualifierAction(EPWM1_BASE,
                                  EPWM_AQ_OUTPUT_B,
								  EPWM_AQ_OUTPUT_TOGGLE,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
	EPWM_enableSyncOutPulseSource(EPWM1_BASE, EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);

    SysCtl_enablePeripheral_TBCLKSYNC();
    EPWM_setHrpwmDllCfg0(PREEPWM_BASE,0x1);

    // Enable sync and clock to PWM
    //
    // SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}


void main(void){

	uint8_t status = 0;
	uint8_t reload_cnt = 0;
	uint32_t tmp32;

	Device_init();
	__enable_irq();
	// Board initialization
    // Configure GPIO 16 as eCAP input
    // Enable interrupts required for this example
    //
	Board_init();



    // Configure ePWM
	initEPWM();


    //
    // Configure the signal monitoring min and max value
    //
    ECAP_configureMinValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, PWM_PRD_VAL-2);
    ECAP_configureMaxValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, PWM_PRD_VAL+2);
    ECAP_configureMinValue(myECAP1_BASE, ECAP_MONITORING_UNIT_2, 0);
    ECAP_configureMaxValue(myECAP1_BASE, ECAP_MONITORING_UNIT_2, 0);
    //
    // Initialize counters:
    //

    cap2Count = 0U;
    cap3Count = 0U;
    cap4Count = 0U;
    while(1)
    {

    }
}

//
// myECAP0 ISR
//
__interrupt void Ecap_ex02_IrqHandler(void)
{
    //
    // Capture the ECAP events.
    //
	uint32_t munt_state=ECAP_getInterruptSource(myECAP1_BASE);

        epwm1PeriodCount = EPWM_getTimeBasePeriod(EPWM1_BASE);

        if((munt_state>>9) !=0x8)
        {
            error();
            ECAP_clearInterrupt(myECAP1_BASE,(ECAP_ISR_SOURCE_MUNIT_1_ERROR_EVT2 ||
            			 	 	 	 	 	  ECAP_ISR_SOURCE_MUNIT_1_ERROR_EVT1 ||
											  ECAP_ISR_SOURCE_MUNIT_2_ERROR_EVT2));
        }


        ecap1IntCount++;

    // Keep track of the ePWM direction and adjust period accordingly to
    // generate a variable frequency PWM.

    if(epwm1TimerDirection == EPWM_TIMER_UP)
    {
        if(epwm1PeriodCount < PWM1_TIMER_MAX)
        {
            EPWM_setTimeBasePeriod(EPWM1_BASE, ++epwm1PeriodCount);
            //set muint value
            ECAP_configureMinValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, epwm1PeriodCount-1);
            ECAP_configureMaxValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, epwm1PeriodCount+1);
        }
        else
        {
            epwm1TimerDirection = EPWM_TIMER_DOWN;
            EPWM_setTimeBasePeriod(EPWM1_BASE, --epwm1PeriodCount);
            //set muint value
            ECAP_configureMinValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, epwm1PeriodCount-1);
            ECAP_configureMaxValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, epwm1PeriodCount+1);
        }
    }
    else
    {
        if(epwm1PeriodCount > PWM1_TIMER_MIN)
        {
            EPWM_setTimeBasePeriod(EPWM1_BASE, --epwm1PeriodCount);
            //set muint value
            ECAP_configureMinValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, epwm1PeriodCount-1);
            ECAP_configureMaxValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, epwm1PeriodCount+1);
        }
        else
        {
            epwm1TimerDirection = EPWM_TIMER_UP;
            EPWM_setTimeBasePeriod(EPWM1_BASE, ++epwm1PeriodCount);
            //set muint value
            ECAP_configureMinValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, epwm1PeriodCount-1);
            ECAP_configureMaxValue(myECAP1_BASE, ECAP_MONITORING_UNIT_1, epwm1PeriodCount+1);
        }
    }

    // Count correct captures

    ecap1PassCount++;

    // Clear interrupt flags for more interrupts.

    ECAP_clearInterrupt(myECAP1_BASE,ECAP_ISR_SOURCE_CAPTURE_EVENT_4);
    ECAP_clearGlobalInterrupt(myECAP1_BASE);
    // Start eCAP

     ECAP_reArm(myECAP1_BASE);
}

#ifdef __cplusplus
}
#endif

