/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "gpio_ex3_interrupt.h"
#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

volatile uint32_t xint1Count = 0; /* Counter for XINT1 interrupt occurrences */
volatile uint32_t loopCount  = 0; /* Counter for loop iterations */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/*
 * @brief  Pin Multiplexing Initialization
 * Configures GPIO pins for input and output based on the defined pin configurations.
 */
void PinMux_init()
{
    /* Configure pins for input interrupt and output pins */
    GPIO_setPinConfig(myGPIOInputInterrupt0_CONFIG);
    GPIO_setPinConfig(myGPIOOutput0_CONFIG);
    GPIO_setPinConfig(myGPIOOutput1_CONFIG);

    /* Ensure the GPIO pins are not in analog mode if they are not of GPIO type */
    if ((check_pin_type(myGPIOInputInterrupt0) != HW_PIN_TYPE_GPIO)) {
        GPIO_setAnalogMode(myGPIOInputInterrupt0, GPIO_ANALOG_DISABLED);
    }
    if (check_pin_type(myGPIOOutput0) != HW_PIN_TYPE_GPIO) {
        GPIO_setAnalogMode(myGPIOOutput0, GPIO_ANALOG_DISABLED);
    }
    if (check_pin_type(myGPIOOutput1) != HW_PIN_TYPE_GPIO) {
        GPIO_setAnalogMode(myGPIOOutput1, GPIO_ANALOG_DISABLED);
    }
}

/*
 * @brief  GPIO Input Interrupt Configuration
 * Configures the input GPIO pin to trigger an interrupt on a rising edge.
 */
void myGPIOInputInterrupt0_init()
{
    GPIO_setQualificationPeriod(myGPIOInputInterrupt0, 510);                              /* Set qualification period for input */
    GPIO_setPadConfig(myGPIOInputInterrupt0, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLDOWN); /* Standard input with pull-down resistor */
    GPIO_setQualificationMode(myGPIOInputInterrupt0, GPIO_QUAL_SYNC);                     /* Synchronous qualification mode */
    GPIO_setDirectionMode(myGPIOInputInterrupt0, GPIO_DIR_MODE_IN);                       /* Set pin as input */
}

/*
 * @brief  GPIO Output Pin 0 Configuration
 * Configures GPIO pin 0 as an output pin.
 */
void myGPIOOutput0_init()
{
    GPIO_setPadConfig(myGPIOOutput0, GPIO_PIN_TYPE_STD);      /* Standard output pin configuration */
    GPIO_setQualificationMode(myGPIOOutput0, GPIO_QUAL_SYNC); /* Synchronous qualification mode */
    GPIO_setDirectionMode(myGPIOOutput0, GPIO_DIR_MODE_OUT);  /* Set pin as output */
}

/*
 * @brief  GPIO Output Pin 1 Configuration
 * Configures GPIO pin 1 as an output pin.
 */
void myGPIOOutput1_init()
{
    GPIO_setPadConfig(myGPIOOutput1, GPIO_PIN_TYPE_STD);      /* Standard output pin configuration */
    GPIO_setQualificationMode(myGPIOOutput1, GPIO_QUAL_SYNC); /* Synchronous qualification mode */
    GPIO_setDirectionMode(myGPIOOutput1, GPIO_DIR_MODE_OUT);  /* Set pin as output */
}

/*
 * @brief  General GPIO Initialization
 * Initializes the input and output GPIO pins.
 */
void GPIO_init()
{
    myGPIOInputInterrupt0_init();
    myGPIOOutput0_init();
    myGPIOOutput1_init();
}

/*
 * @brief  Board Initialization
 * Initializes the pin multiplexing and GPIO configurations.
 */
void Board_init(void)
{
    PinMux_init();
    GPIO_init();
}

/*
 * @brief  GPIO Interrupt Handler
 * This interrupt service routine toggles an output pin and increments the interrupt count.
 */
#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void GPIO_IRQHandler(void)
{
    SAVE_IRQ_CSR_CONTEXT();

    GPIO_togglePin(myGPIOOutput1);
    __DSB();
    xint1Count++;

    RESTORE_IRQ_CSR_CONTEXT();
}
#else
void GPIO_IRQHandler(void)
{
    GPIO_togglePin(myGPIOOutput1);
    __DSB();
    xint1Count++;
}
#endif

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/*
 * @brief  main function.
 */
int main(void)
{
    /* Temporary variable to store the interrupt count */
    uint32_t xint1CountTemp;

    /* Initialize device */
    Device_init();

    /* Initialize the board (GPIO and pinmux) */
    Board_init();

    /* Configure GPIO interrupt to trigger on a rising edge on input pin */
    GPIO_setInterruptType(GPIO_INT_XINT1, GPIO_INT_TYPE_RISING_EDGE);
    /* Enable the interrupt */
    GPIO_setInterruptPin(myGPIOInputInterrupt0, GPIO_INT_XINT1);
    /* Enable the interrupt */
    GPIO_enableInterrupt(GPIO_INT_XINT1);

    /* Register the interrupt handler */
    Interrupt_register(INT_XINT1, &GPIO_IRQHandler);
    /* Enable the interrupt in the interrupt controller */
    Interrupt_enable(INT_XINT1);

    /* Set GPIO Output Pin 0 to low */
    GPIO_writePin(myGPIOOutput0, 0);
    /* Set GPIO Output Pin 1 to high */
    GPIO_writePin(myGPIOOutput1, 1);

    /* Enable global interrupts */
    __enable_irq();

    while (1) {
        /* Store the current interrupt count */
        xint1CountTemp = xint1Count;

        GPIO_writePin(myGPIOOutput0, 1); /* Set Output Pin 0 to high */

        /* Wait for the interrupt to increment the counter */
        while (xint1Count == xint1CountTemp) {
            ;
        }

        /* If the interrupt count has increased by 1, increment loopCount and toggle Output Pin 0 */
        if (xint1Count == (xint1CountTemp + 1)) {
            loopCount++;                     /* Increment the loop count */
            GPIO_writePin(myGPIOOutput0, 0); /* Set Output Pin 0 to low */
        }

        DELAY_US(1000 * 1000); /* Delay for 1 second */
    }

    return 0; /* Program will never reach this point */
}
