/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * @examplespi master slave fifo interrupt
 * @brief  
 *   SPI Digital Loopback with FIFO Interrupts.
 *
 *   This test routine uses two SPI modules and employs FIFO mode as well as
 *   interrupt mode. SPIA is configured as a slave device and receives data from
 *   SPIB, which is configured as a master device. The master device sends data
 *   and the slave device receives the data and verifies it.
 * 
 *   Watch Variables:
 *                   - sData - Data to send
 *                   - rData - Received data
 *                   - rDataPoint - Used to keep track of the last position in the
 *                                  receive stream for error checking.
 *
 * @note:
 *   1.This routine requires the chip to support two SPI channels.
 *
 */

/**
 * @examplespi master slave fifo interrupt
 * @brief  
 *    FIFO жϵ SPI ֻء
 * 
 *   ʹSPIģ飬õFIFOģʽԼжģʽSPIAΪ豸
 *   Ϊ豸SPIBݡ豸ᷢݣ豸ݲУ顣
 *
 *   :
 *           - sData - ͵
 *           - rData - յ
 *           - rDataPoint - ڼ¼еλãԱд顣
 *
 * @note:
 *   1.ҪоƬ֧·SPI
 *
 */

#include <spi_master_slave_fifo_interrupt.h>
#include "driverlib.h"
#include "device.h"

/**
 * @brief Globals Variables
 */
uint16_t sData[2];                  /** @brief Send data buffer */
uint16_t rData[2];                  /** @brief Receive data buffer */
uint16_t rDataPoint = 0;            /** @brief To keep track of where we are in the
                                        data stream to check received data */

void spiATxFIFOISR(void);
void spiBRxFIFOISR(void);

int main(void)
{
    uint16_t i;

    /**
     * @brief Initialize device clock and peripherals
     */
    Device_init();

    __disable_irq();

    spi_master_init();
    spi_slave_init();

    /**
     * @brief Initialize the data buffers
     */
    for(i = 0; i < 2; i++)
    {
        sData[i] = i;
        rData[i]= 0;
    }

    __enable_irq();

    /**
     * @brief Loop forever. Suspend or place breakpoints to observe the buffers.
     */
    while(1)
    {

    }

    return 0;
}

void spiBRxFIFOISR(void)
{
    uint16_t i;
    uint16_t status;
    status = SPI_getInterruptStatus(SPIB_BASE);

    for(i = 0; i < 2; i++)
    {
        rData[i] = SPI_readDataNonBlocking(SPIB_BASE);
    }

    for(i = 0; i < 2; i++)
    {
        if(rData[i] != (uint16_t)(rDataPoint + i))
        {

            ESTOP0;
        }
    }

    rDataPoint++;

    /**
     * @brief Stop after receiving a total of 100 data points.
     */
    if(rDataPoint == 100){
    	ESTOP0;
    }

    SPI_clearInterruptStatus(CONFIG_SPI_SLAVE_BASE, SPI_INT_RXFF);

}

void spiATxFIFOISR(void)
{
    uint16_t i;
    uint16_t status;
    status = SPI_getInterruptStatus(SPIA_BASE);

    for(i = 0; i < 2; i++)
    {
       SPI_writeDataNonBlocking(SPIA_BASE, sData[i]);
    }

    /**
     ** @brief Increment data for next cycle
     */
    for(i = 0; i < 2; i++)
    {
       sData[i] = sData[i] + 1;
    }

    SPI_clearInterruptStatus(CONFIG_SPI_MASTER_BASE, SPI_INT_TXFF);
}

