#include "uid_ip_8.h"

// Reference time: 00:00:00, January 1, 200
#define REFERENCE_YEAR 2000

// Number of days in a month (non-leap year)
static const uint8_t days_in_month[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

// Determine whether it is a leap year
static uint8_t is_leap_year(uint16_t year) {
    return ((year % 4 == 0) && (year % 100 != 0)) || (year % 400 == 0);
}

// Obtain the full year from the two-digit year
static uint16_t get_full_year(uint8_t year_short) {
    return 2000 + year_short;
}

// Obtain the total number of days prior to the specified year
static uint32_t days_since_reference(uint16_t year) {
    uint32_t days = 0;
    for (uint16_t y = REFERENCE_YEAR; y < year; y++) {
        days += is_leap_year(y) ? 366 : 365;
    }
    return days;
}

// Obtain the total number of days before the specified month in the specified year
static uint16_t days_in_year_to_month(uint16_t year, uint8_t month) {
    uint16_t days = 0;
    for (uint8_t m = 1; m < month; m++) {
        days += days_in_month[m - 1];
        // Leap year has an extra day in February
        if (m == 2 && is_leap_year(year)) {
            days += 1;
        }
    }
    return days;
}

uint32_t datetime_to_seconds(const DateTime* dt) {
	// Obtain the complete year
    uint16_t full_year = get_full_year(dt->year);

    // Calculate the total number of days
    uint32_t total_days = days_since_reference(full_year);
    total_days += days_in_year_to_month(full_year, dt->month);
    total_days += (dt->day - 1);

    // Calculate the total number of seconds
    uint32_t total_seconds = total_days * 86400UL;
    total_seconds += dt->hour * 3600UL;
    total_seconds += dt->minute * 60UL;
    total_seconds += dt->second;

    return total_seconds;
}

void seconds_to_datetime(uint32_t seconds, DateTime* dt) {
    uint32_t remaining_seconds = seconds;

    // Calculate the number of days
    uint32_t days = remaining_seconds / 86400UL;
    remaining_seconds %= 86400UL;

    // Calculate the year
    uint16_t year = REFERENCE_YEAR;
    while (days >= (is_leap_year(year) ? 366 : 365)) {
        days -= is_leap_year(year) ? 366 : 365;
        year++;
    }
    dt->year = year - 2000;

    // Calculate the month
    dt->month = 1;
    uint8_t days_in_current_month;
    while (days >= (days_in_current_month = days_in_month[dt->month - 1] +
                   (dt->month == 2 && is_leap_year(year) ? 1 : 0))) {
        days -= days_in_current_month;
        dt->month++;
    }

    // Calculate the day
    dt->day = days + 1;

    // Calculate time
    dt->hour = remaining_seconds / 3600UL;
    remaining_seconds %= 3600UL;
    dt->minute = remaining_seconds / 60UL;
    dt->second = remaining_seconds % 60UL;
}

void create_unique_id(const DateTime* dt, uint32_t ip, uint8_t serial, uint8_t* output) {
	// Convert to timestamp (in seconds)
    uint32_t timestamp = datetime_to_seconds(dt);

    // Combine the high 24 bits of the IP address with the low 8 bits of the serial number (discard the low 8 bits of the IP address)
    uint32_t ip_serial = ((ip & 0xFFFFFF00) >> 8) | ((uint32_t)serial << 24);

    // Store the timestamp in big-endian format
    output[0] = (timestamp >> 24) & 0xFF;
    output[1] = (timestamp >> 16) & 0xFF;
    output[2] = (timestamp >> 8) & 0xFF;
    output[3] = timestamp & 0xFF;

    // Store ip_serial in big-endian format
    output[4] = (ip_serial >> 24) & 0xFF;
    output[5] = (ip_serial >> 16) & 0xFF;
    output[6] = (ip_serial >> 8) & 0xFF;
    output[7] = ip_serial & 0xFF;
}

// Extract the IP address from ip_serial (take the high 24 bits and pad the low 8 bits with zeros)
uint32_t extract_ip_from_serial(uint32_t ip_serial) {
    return (ip_serial & 0x00FFFFFF) << 8;
}

uint8_t extract_serial_from_ip_serial(uint32_t ip_serial) {
    return (ip_serial >> 24) & 0xFF;
}

void parse_unique_id(const uint8_t* input, UniqueID* unique_id) {
    unique_id->timestamp = ((uint32_t)input[0] << 24) |
                          ((uint32_t)input[1] << 16) |
                          ((uint32_t)input[2] << 8) |
                          (uint32_t)input[3];

    unique_id->ip_serial = ((uint32_t)input[4] << 24) |
                          ((uint32_t)input[5] << 16) |
                          ((uint32_t)input[6] << 8) |
                          (uint32_t)input[7];
}

uint32_t ip_string_to_uint(const char* ip_str) {
    uint32_t ip = 0;
    uint8_t segments[4] = {0};
    uint8_t segment_index = 0;
    uint16_t current_value = 0;

    for (const char* p = ip_str; *p != '\0'; p++) {
        if (*p >= '0' && *p <= '9') {
            current_value = current_value * 10 + (*p - '0');
        } else if (*p == '.') {
            if (segment_index < 4) {
                segments[segment_index++] = (uint8_t)current_value;
                current_value = 0;
            }
        }
    }
    if (segment_index < 4) {
        segments[segment_index] = (uint8_t)current_value;
    }

    ip = ((uint32_t)segments[0] << 24) |
         ((uint32_t)segments[1] << 16) |
         ((uint32_t)segments[2] << 8) |
         (uint32_t)segments[3];

    return ip;
}

void ip_uint_to_string(uint32_t ip, char* buffer) {
    uint8_t segments[4];
    char temp[4];
    uint8_t buffer_index = 0;

    segments[0] = (ip >> 24) & 0xFF;
    segments[1] = (ip >> 16) & 0xFF;
    segments[2] = (ip >> 8) & 0xFF;
    segments[3] = ip & 0xFF;

    for (int i = 0; i < 4; i++) {
        if (segments[i] >= 100) {
            buffer[buffer_index++] = '0' + (segments[i] / 100);
            segments[i] %= 100;
        }

        if (segments[i] >= 10) {
            buffer[buffer_index++] = '0' + (segments[i] / 10);
            segments[i] %= 10;
        }

        buffer[buffer_index++] = '0' + segments[i];

        if (i < 3) {
            buffer[buffer_index++] = '.';
        }
    }

    buffer[buffer_index] = '\0';
}
uint64_t get_uid_8(){
	uint8_t unique_id[8];
	uint64_t result;
    DateTime* const dt_ptr = (DateTime*)DATE_TIME_ADDRESS;
    uint32_t* const ip =(uint32_t*)DATE_IP_ADDRESS;
    uint8_t* const serial =(uint8_t*)DATE_serial_ADDRESS;
    create_unique_id(dt_ptr, *ip, *serial, unique_id);
    memcpy(&result, unique_id, sizeof(unique_id));
    return result;
}
