/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file     main.c
*   @brief    template main.c file
*   @details  toggle GPIO to drive LED1, print output via UART.
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"
#include "gs32_math.h"
#include "cfft_f32.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define FFT_SIZE 256
#define PI (3.1415926536f)

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

CFFT_F32_STRUCT fft_struct;
float32_t fft_in_buf[2 * FFT_SIZE];    // Ҫռ
float32_t fft_mag_buf[FFT_SIZE];
float32_t fft_phase_buf[FFT_SIZE];

riscv_cfft_instance_f32 fft_inst;
float32_t riscv_fft_input[2*FFT_SIZE]; // Ҫռ
float32_t riscv_fft_mag[FFT_SIZE];

uint32_t dlog_fft_ts[5];
uint32_t dlog_fft_time[4];

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/**
 * @brief ɲź
 * @param signal_buf
 */
void generate_test_signal(float32_t *signal_buf)
{
    for (int i = 0; i < FFT_SIZE; i++) {
        signal_buf[2*i] = 0.7f * __cos(2.0f * PI * i * 2 / FFT_SIZE) +
                         0.3f * __cos(2.0f * PI * i * 10 / FFT_SIZE);
        signal_buf[2*i+1] = 0.0f;
    }
}


void test_cfft(void)
{
    printf("\nTesting CFFT_F32 style implementation:\n");

    /* ʼ FFT ṹ */
    CFFT_F32_STRUCT_Handle fft_handle = CFFT_f32_init(&fft_struct, FFT_SIZE, fft_in_buf);

    /* ɲź */
    generate_test_signal(fft_handle->InPtr);
    dlog_fft_ts[0] = CPU_getCycleCnt();

    /*  */
    fft_handle->CurrentOutPtr = fft_mag_buf;
    CFFT_f32s_mag(fft_handle);
    dlog_fft_ts[1] = CPU_getCycleCnt();

    /* λ */
    fft_handle->CurrentOutPtr = fft_phase_buf;
    CFFT_f32_phase(fft_handle);
    dlog_fft_ts[2] = CPU_getCycleCnt();

    /* ִ FFT */
    CFFT_f32(fft_handle);
    dlog_fft_ts[3] = CPU_getCycleCnt();

    /* ִ IFFT */
    ICFFT_f32(fft_handle);
    dlog_fft_ts[4] = CPU_getCycleCnt();

    dlog_fft_time[0] = dlog_fft_ts[1] - dlog_fft_ts[0];
    dlog_fft_time[1] = dlog_fft_ts[2] - dlog_fft_ts[1];
    dlog_fft_time[2] = dlog_fft_ts[3] - dlog_fft_ts[2];
    dlog_fft_time[3] = dlog_fft_ts[4] - dlog_fft_ts[3];

    printf("Performance:\n");
    printf("Magnitude calculation time: %u cycles\n", dlog_fft_time[0]);
    printf("Phase calculation time: %u cycles\n", dlog_fft_time[1]);
    printf("FFT execution time: %u cycles\n", dlog_fft_time[2]);
    printf("IFFT execution time: %u cycles\n", dlog_fft_time[3]);
}


void test_riscv_cfft(void)
{
    printf("\nTesting RISCV_CFFT style implementation:\n");

    /* ʼ FFT ʵ */
    fft_inst = riscv_cfft_sR_f32_len256;

    /* ɲź */
    generate_test_signal(riscv_fft_input);
    dlog_fft_ts[0] = CPU_getCycleCnt();

    /*  */
    riscv_cmplx_mag_f32(riscv_fft_input, riscv_fft_mag, FFT_SIZE);
    dlog_fft_ts[1] = CPU_getCycleCnt();

    /* ִ FFT (0ʾFFT1ʾλת) */
    riscv_cfft_f32(&fft_inst, riscv_fft_input, 0, 1);
    dlog_fft_ts[2] = CPU_getCycleCnt();

    /* ִ IFFT */
    riscv_cfft_f32(&fft_inst, riscv_fft_input, 1, 1);
    dlog_fft_ts[3] = CPU_getCycleCnt();

    dlog_fft_time[0] = dlog_fft_ts[1] - dlog_fft_ts[0];
    dlog_fft_time[1] = dlog_fft_ts[2] - dlog_fft_ts[1];
    dlog_fft_time[2] = dlog_fft_ts[3] - dlog_fft_ts[2];

    printf("Performance:\n");
    printf("Magnitude calculation time: %u cycles\n", dlog_fft_time[0]);
    printf("FFT execution time: %u cycles\n", dlog_fft_time[1]);
    printf("IFFT execution time: %u cycles\n", dlog_fft_time[2]);
}

void loop_test(void)
{
    uint16_t i;
    CFFT_F32_STRUCT_Handle fft_handle = CFFT_f32_init(&fft_struct, FFT_SIZE, fft_in_buf);

    GPIO_writePin(23,0);
    for (i=0;i<10000;i++){CFFT_f32(fft_handle);}

    GPIO_writePin(23,1);
    for (i=0;i<10000;i++){CFFT_f32s_mag(fft_handle);}

    GPIO_writePin(23,0);
    for (i=0;i<10000;i++){CFFT_f32_phase(fft_handle);}

    GPIO_writePin(23,1);
    for (i=0;i<10000;i++){ICFFT_f32(fft_handle);}

    GPIO_writePin(23,1);
    for (i=0;i<10000;i++){riscv_cmplx_mag_f32(riscv_fft_input, riscv_fft_mag, FFT_SIZE);}

    GPIO_writePin(23,0);
    for (i=0;i<10000;i++){riscv_cfft_f32(&fft_inst, riscv_fft_input, 0, 1);}

    GPIO_writePin(23,1);
    for (i=0;i<10000;i++){riscv_cfft_f32(&fft_inst, riscv_fft_input, 1, 1);}

}

int main(void)
{
    __disable_irq();

    Device_init();

    GPIO_enableWritePin(23);

    UartPrint_init(LOG_SCI_BASE, 115200);

    test_cfft();
    test_riscv_cfft();

    /*IO23ƽ仯Զμʱ*/
    loop_test();

    for(;;);
}

#ifdef __cplusplus
}
#endif
