/*
/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
//#############################################################################
//
// FILE:   pto_pulsegen.c
//
// Description: Example project for using PM pto_pulsegen Library.
//              Includes pto_pulsegen library and corresponding include
//              files. Initializes the encoders and performs delay compensation.
//              Runs pto_pulsegen command set.
//              Continuously Read position value in an infinite loop
//
/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "driverlib.h"
#include "device.h"
#include "stdint.h"
#include "pto_pulsegen.h"

#include "clb_config.h"
#include "clb.h"

//*****************************************************************************
//
//! Setup CLB tiles
//!
//! \param None
//!
//! Setup the CLB and other interconnect XBARs used in this application.
//! This is called during the CLB initialization. This function needs to be
//! called after every system reset.
//!
//! \return None
//
//*****************************************************************************
void
pto_pulsegen_setupPeriph(void)
{
    pto_pulsegen_resetCLB();
    initTILE1(myCLB0_BASE);


    //
    // SW-controlled input select (GP_REG)
    //
    CLB_configGPInputMux(myCLB0_BASE, CLB_IN0, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(myCLB0_BASE, CLB_IN1, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(myCLB0_BASE, CLB_IN2, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(myCLB0_BASE, CLB_IN3, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(myCLB0_BASE, CLB_IN4, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(myCLB0_BASE, CLB_IN5, CLB_GP_IN_MUX_EXTERNAL);
    CLB_configGPInputMux(myCLB0_BASE, CLB_IN6, CLB_GP_IN_MUX_EXTERNAL);
    CLB_configGPInputMux(myCLB0_BASE, CLB_IN7, CLB_GP_IN_MUX_EXTERNAL);

    //
    // Input Filtering (Edge detects, or passthru)
    //
    CLB_selectInputFilter(myCLB0_BASE, CLB_IN0, CLB_FILTER_NONE);
    CLB_selectInputFilter(myCLB0_BASE, CLB_IN1, CLB_FILTER_RISING_EDGE);
    CLB_selectInputFilter(myCLB0_BASE, CLB_IN2, CLB_FILTER_NONE);
    CLB_selectInputFilter(myCLB0_BASE, CLB_IN3, CLB_FILTER_NONE);
    CLB_selectInputFilter(myCLB0_BASE, CLB_IN4, CLB_FILTER_NONE);
    CLB_selectInputFilter(myCLB0_BASE, CLB_IN5, CLB_FILTER_NONE);
    CLB_selectInputFilter(myCLB0_BASE, CLB_IN6, CLB_FILTER_NONE);
    CLB_selectInputFilter(myCLB0_BASE, CLB_IN7, CLB_FILTER_NONE);

    pto_pulsegen_initCLBXBAR();   // Initialize the CLB XBAR
}

//*****************************************************************************
//
//! Reset pulsegen
//!
//! \param None
//!
//! This function resets the pulsegen parameters set by earlier
//! configuration and start a fresh setup. The reset is done by calling the
//! reset CLB function.
//!
//! \return None
//
//*****************************************************************************
void
pto_pulsegen_reset(void)
{
    pto_pulsegen_resetCLB();
    CLB_setGPREG(myCLB0_BASE, 0);
}

//*****************************************************************************
//
//! Run pulse generator by loading PTO options (pulses, period, duration
//! into HLC registers
//!
//! \param None
//! \param pulseLo is low pulse width
//! \param pulseHi is high pulse width
//! \param ptoActivePeriod is the period the pulses are sent out
//!        less than ptoFullPeriod
//! \param ptoFullPeriod is full PTO period
//! \param ptoInterruptTime is the time when the interrupt is generated to
//!        the CPU
//! \param ptoDirection is the output direction output. The output direction is
//!        latched at the beginning of new period
//! \param run is the run status. 1 indicates run and 0 is stop.
//!        This value is sampled at the beginning of the new period to determine
//!         to continue or halt the pulse generation
//!
//! This function is a runtime function to be called periodically
//! for dynamically configuring and changing the pulse generation requirements
//! as needed by the application
//!
//! \return ptoFullPeriod return the value, if the function is
//! executed successfully
//
//*****************************************************************************
uint16_t
pto_pulsegen_runPulseGen(uint32_t pulseLo, uint32_t pulseHi,
                            uint32_t ptoActivePeriod, uint32_t ptoFullPeriod,
                            uint32_t ptoInterruptTime, uint16_t ptoDirection,
                            uint16_t run)
{
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_HLC_R0, pulseLo - 1);
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_HLC_R1, pulseLo + pulseHi - 1);
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_HLC_R2, ptoActivePeriod - 1);
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_HLC_R3, ptoFullPeriod - 1);
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_COUNTER_2_MATCH1,
                       ptoInterruptTime - 1);
    CLB_setGPREG(myCLB0_BASE,(run*6) + (ptoDirection*16) +1);
    return(ptoFullPeriod);
}

//*****************************************************************************
//
//! This function initiates the pulse generation.
//!
//! \param None
//!
//! This function must be called after pto_pulsegen_setupPeriph.
//! Hence, the pto_pulsegen_startOperation kick starts the pulse generation that
//! was set up earlier.
//!
//! \return None.
//
//*****************************************************************************
void
pto_pulsegen_startOperation(void)
{
	  CLB_enableCLB(myCLB0_BASE);


    CLB_setGPREG(myCLB0_BASE, 0x15);

}

//*****************************************************************************
//
//! Reset the CLB
//!
//! \param None
//!
//! This function configures Output-XBars to route CLB outputs to GPIO pins
//!
//! \return None
//
//*****************************************************************************
void
pto_pulsegen_resetCLB(void)
{
    //
    // Turn OFF the CLB functionality
    //
    CLB_setGPREG(myCLB0_BASE, 0);

    CLB_disableCLB(myCLB0_BASE);
    //
    // Clear Counters - this will clear counter REGs.
    // Add_shift_on_even_en should be Zero for this to take effect.
    //
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_COUNTER_0_LOAD, 0x0);
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_COUNTER_1_LOAD, 0x0);
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_COUNTER_2_LOAD, 0x0);

    //
    // Clear and reload the HLC registers
    //
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_HLC_R0, 0x9);
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_HLC_R1, 0x0);
    CLB_writeInterface(myCLB0_BASE, CLB_ADDR_COUNTER_1_MATCH1, 0x2);

}

//*****************************************************************************
//
//! Initialize the CLB XBAR-connect
//!
//! \param None
//!
//! This function configures Output-XBars to route CLB outputs to GPIO pins
//!
//! \return None
//
//*****************************************************************************
void
pto_pulsegen_initCLBXBAR(void) // see configuration for controlCARD
{
    //
    // Configure CLB1 o/p 4 and 5 to OUTPUT XBAR to OUTPUT3 and OUTPUT4
    // respectively
    //
#if defined(F2837x_F28004x)
    XBAR_setOutputMuxConfig(XBAR_OUTPUT3, XBAR_OUT_MUX01_CLB1_OUT4);
    XBAR_enableOutputMux(XBAR_OUTPUT3, XBAR_MUX01);

    XBAR_setOutputMuxConfig(XBAR_OUTPUT4, XBAR_OUT_MUX03_CLB1_OUT5);
    XBAR_enableOutputMux(XBAR_OUTPUT4, XBAR_MUX03);
#elif defined(F28002x_F28003x)
    XBAR_setOutputMuxConfig(OUTPUTXBAR_BASE, XBAR_OUTPUT3, XBAR_OUT_MUX01_CLB1_OUT4);
	XBAR_enableOutputMux(OUTPUTXBAR_BASE, XBAR_OUTPUT3, XBAR_MUX01);

    XBAR_setOutputMuxConfig(OUTPUTXBAR_BASE, XBAR_OUTPUT4, XBAR_OUT_MUX03_CLB1_OUT5);
	XBAR_enableOutputMux(OUTPUTXBAR_BASE, XBAR_OUTPUT4, XBAR_MUX03);
#else
    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE, XBAR_OUTPUT3,
    		XBAR_OUT_MUX03_CLB1_OUT4);
    XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE, XBAR_OUTPUT3, XBAR_MUX01);

    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE, XBAR_OUTPUT4,
    		XBAR_OUT_MUX04_CLB1_OUT5);
    XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE, XBAR_OUTPUT4, XBAR_MUX03);
#endif
}

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

//
// End of File
//
