/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    Test_CpuCRC.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "device.h"
#include <time.h>
#include <stdlib.h>
#include "driverlib.h"
#include "load_img.h"
#include "log.h"            //smp thread safe
#include "board_cfg.h"
#include "interrupt.h"
#include <CRC_ex02_misaligned_calc.h>
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define LED_PIN             LED1_PIN

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

const CRC_8 crc_8 = {0x07,0x00,0x00,FALSE,FALSE};
const CRC_8 crc_8_ITU = {0x07,0x00,0x55,FALSE,FALSE};
const CRC_8 crc_8_ROHC = {0x07,0xff,0x00,TRUE,TRUE};
const CRC_8 crc_8_MAXIM = {0x31,0x00,0x00,TRUE,TRUE};

const CRC_16 crc_16_IBM = {0x8005,0x0000,0x0000,TRUE,TRUE};
const CRC_16 crc_16_MAXIM = {0x8005,0x0000,0xffff,TRUE,TRUE};
const CRC_16 crc_16_USB = {0x8005,0xffff,0xffff,TRUE,TRUE};
const CRC_16 crc_16_MODBUS = {0x8005,0xffff,0x0000,TRUE,TRUE};
const CRC_16 crc_16_CCITT = {0x1021,0x0000,0x0000,TRUE,TRUE};
const CRC_16 crc_16_CCITT_FALSE = {0x1021,0xffff,0x0000,FALSE,FALSE};
const CRC_16 crc_16_X5 = {0x1021,0xffff,0xffff,TRUE,TRUE};
const CRC_16 crc_16_XMODEM = {0x1021,0x0000,0x0000,FALSE,FALSE};
const CRC_16 crc_16_DNP = {0x3d65,0x0000,0xffff,TRUE,TRUE};

const CRC_32 crc_32 = {0x04c11db7,0xffffffff,0xffffffff,TRUE,TRUE};
const CRC_32 crc_32_MPEG2 = {0x04c11db7,0xffffffff,0x00000000,FALSE,FALSE};
const CRC_32 crc_32_AIXM = {0x814141AB, 0x0000000, 0x00000000, FALSE, FALSE};
const CRC_32 crc_32_AUTOSAR = {0xF4ACFB13,0xffffffff,0xffffffff,TRUE,TRUE};

uint8_t testCrc8Result;
uint8_t testCrc8Result1;
uint16_t testCrc16Result;
uint16_t testCrc16Result1;
uint16_t testCrc16Result2;
uint32_t testCrc32Result;
uint32_t testCrc32Result1;
uint32_t testCrc32Result2;

uint8_t testBuffer[15] = {0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,\
					  0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F};
volatile uint32_t coreTicks = 0;

void TIMER0_IRQHandler(void)
{
    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    coreTicks++;
}

void Timer_init(void)
{
    CPUTimer_init(CPUTIMER0_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER0, TIMER0_IRQHandler);
    Interrupt_enable(INT_TIMER0);
}

//*****************************************************************************
//! This function is used to byte reverse.
//!
//!
//! \return Returns crc8result.
//!
//
//*****************************************************************************
static uint8_t reverse8(uint8_t data)
{
	 uint8_t i;
	 uint8_t temp=0;
    for(i=0;i<8;i++)
        temp |= ((data>>i) & 0x01)<<(7-i);
    return temp;
}

//*****************************************************************************
//! This function is used to half word reverse.
//!
//!
//! \return Returns crc8result.
//!
//
//*****************************************************************************
static uint16_t reverse16(uint16_t data)
{
	 uint8_t i;
    uint16_t temp=0;
    for(i=0;i<16;i++)
        temp |= ((data>>i) & 0x0001)<<(15-i);
    return temp;
}

//*****************************************************************************
//! This function is used to word reverse.
//!
//!
//! \return Returns crc8result.
//!
//
//*****************************************************************************
static uint32_t reverse32(uint32_t data)
{
    uint8_t i;
    uint32_t temp=0;
    for(i=0;i<32;i++)
        temp |= ((data>>i) & 0x01)<<(31-i);
    return temp;
}

//*****************************************************************************
//! calculate crc8 by software.
//!
//! \param  addr specifies the "array" base address.
//!         num  specifies the "array" quantity.
//!        CRC_8 specifies the crc calculation's polyValue, initValue.
//!              refInValue, refOutValue, xoroutValue
//! This function is used to calculate crc8.
//!
//!
//! \return Returns crc8result.
//!
//
//*****************************************************************************
uint8_t Crc8Calc_BySoftware(uint8_t *addr, uint32_t num, CRC_8 type)
{
	 uint8_t data;
	 uint8_t crc = type.InitValue;
    int i;
    for (; num > 0; num--)
    {
        data = *addr++;
        if(type.InputReverse == TRUE)
        data = reverse8(data);
        crc = crc ^ data ;
        for (i = 0; i < 8; i++)
        {
            if (crc & 0x80)
                crc = (crc << 1) ^ type.poly;
            else
                crc <<= 1;
        }
    }
    if(type.OutputReverse == TRUE)
        crc = reverse8(crc);
    crc = crc^type.xor;
    return(crc);
}

//*****************************************************************************
//! calculate crc16 by software.
//!
//! \param  addr specifies the "array" base address.
//!         num  specifies the "array" quantity.
//!        CRC_16 specifies the crc calculation's polyValue, initValue.
//!              refInValue, refOutValue, xoroutValue
//! This function is used to calculate crc16.
//!
//!
//! \return Returns crc16result.
//!
//
//*****************************************************************************
uint16_t Crc16Calc_BySoftware(uint8_t *addr, uint32_t num, CRC_16 type)
{
    uint8_t data;
    uint16_t crc = type.InitValue;
    int i;
    for (; num > 0; num--)
    {
        data = *addr++;
        if(type.InputReverse == TRUE)
            data = reverse8(data);
        crc = crc ^ (data<<8) ;
        for (i = 0; i < 8; i++)
        {
            if (crc & 0x8000)
                crc = (crc << 1) ^ type.poly;
            else
                crc <<= 1;
        }
    }
    if(type.OutputReverse == TRUE)
        crc = reverse16(crc);
    crc = crc^type.xor;
    return(crc);
}

//*****************************************************************************
//! calculate crc32 by software.
//!
//! \param  addr specifies the "array" base address.
//!         num  specifies the "array" quantity.
//!        CRC32 specifies the crc calculation's polyValue, initValue.
//!              refInValue, refOutValue, xoroutValue
//! This function is used to calculate crc32.
//!
//!
//! \return Returns crc32result.
//!
//
//*****************************************************************************
uint32_t Crc32Calc_BySoftware(uint8_t *addr, uint32_t num, CRC_32 type)
{
	 uint8_t data;
	 uint32_t crc = type.InitValue;
    int i;
    for (; num > 0; num--)
    {
        data = *addr++;
        if(type.InputReverse == TRUE)
            data = reverse8(data);
        crc = crc ^ (data<<24) ;
        for (i = 0; i < 8; i++)
        {
            if (crc & 0x80000000)
                crc = (crc << 1) ^ type.poly;
            else
                crc <<= 1;
        }
    }
    if(type.OutputReverse == TRUE)
        crc = reverse32(crc);
    crc = crc^type.xor;
    return(crc);
}
 //*****************************************************************************
 //! calculate crc8 by crcmodule.
 //!
 //! \param  pData specifies the "array" base address.
 //!         Length  specifies the "array" quantity.
 //!        CRC_8 specifies the crc calculation's polyValue, initValue.
 //!              refInValue, refOutValue, xoroutValue
 //! This function is used to calculate crc8.
 //!
 //!
 //! \return Returns crc8result.
 //!
 //
 //*****************************************************************************
 uint8_t Crc8Calc_Misaligned(uint8_t *pData, uint32_t length, CRC_8 type)
 {
	 uint32_t tmpLegenth;
	 uint8_t crcResult;

	 CRC_setConfig(CRC_BASE, CRC_POLYSIZE_8BIT, type.poly, type.InputReverse,\
			 type.OutputReverse, type.InitValue);

	 crcResult = CRC_calcResult8(CRC_BASE, pData, length);
	 crcResult ^= type.xor;

	 return crcResult;
 }


 //16
 #define BSWAP_16(x) \
     (uint16_t)((((uint16_t)(x) & 0x00ff) << 8) | \
               (((uint16_t)(x) & 0xff00) >> 8) \
              )

 //32
 #define BSWAP_32(x) \
     (uint32_t)((((uint32_t)(x) & 0xff000000) >> 24) | \
               (((uint32_t)(x) & 0x00ff0000) >> 8) | \
               (((uint32_t)(x) & 0x0000ff00) << 8) | \
               (((uint32_t)(x) & 0x000000ff) << 24) \
              )


 //*****************************************************************************
 //! calculate crc16 by crcmodule.
 //!
 //! \param  pData specifies the "array" base address.
 //!         Length  specifies the "array" quantity.
 //!        CRC_16 specifies the crc calculation's polyValue, initValue.
 //!              refInValue, refOutValue, xoroutValue
 //! This function is used to calculate crc16.
 //!
 //!
 //! \return Returns crc16result.
 //!
 //
 //*****************************************************************************
 uint16_t Crc16Calc_Misaligned(uint8_t *pData, uint32_t length, CRC_16 type)
 {
	 uint8_t tmpLegenth;
	 uint16_t crcResult;
	 tmpLegenth = length % 2;
	 uint16_t pTmpData[10];

	 for(int i = 0; i < length/2; i++)
	 {
		 pTmpData[i] = BSWAP_16(*(uint16_t*)(pData + 2*i));
	 }
	 if(tmpLegenth != 0 && type.OutputReverse == 1)
	 {
		 CRC_setConfig(CRC_BASE, CRC_POLYSIZE_16BIT, type.poly, type.InputReverse,\
						 0, type.InitValue);
	 }
	 else
	 {
		 CRC_setConfig(CRC_BASE, CRC_POLYSIZE_16BIT, type.poly, type.InputReverse,\
				 type.OutputReverse, type.InitValue);
	 }
	 crcResult = CRC_calcResult16(CRC_BASE, pTmpData, length/2);
	 if(tmpLegenth != 0)
	 {
		 type.InitValue = crcResult;
		 crcResult =  Crc16Calc_BySoftware(pData +length - tmpLegenth, tmpLegenth, type);
	 }
	 else
	 {
		 crcResult ^= type.xor;
	 }
	 return crcResult;
 }

 //*****************************************************************************
 //! calculate crc32 by crcmodule.
 //!
 //! \param  pData specifies the "array" base address.
 //!         Length  specifies the "array" quantity.
 //!        CRC_32 specifies the crc calculation's polyValue, initValue.
 //!              refInValue, refOutValue, xoroutValue
 //! This function is used to calculate crc32.
 //!
 //!
 //! \return Returns crc32result.
 //!
 //
 //*****************************************************************************
 uint32_t Crc32Calc_Misaligned(uint8_t *pData, uint32_t length, CRC_32 type)
 {
	 uint8_t tmpLegenth;
	 uint32_t crcResult;
	 tmpLegenth = length % 4;
	 uint32_t pTmpData[10];

	 for(int i = 0; i < length/4; i++)
	 {
		 pTmpData[i] = BSWAP_32(*(uint32_t*)(pData + 4*i));
	 }
	 if(tmpLegenth != 0 && type.OutputReverse == 1)
	 {
		 CRC_setConfig(CRC_BASE, CRC_POLYSIZE_32BIT, type.poly, type.InputReverse,\
						 0, type.InitValue);
	 }
	 else
	 {
		 CRC_setConfig(CRC_BASE, CRC_POLYSIZE_32BIT, type.poly, type.InputReverse,\
				 type.OutputReverse, type.InitValue);
	 }
	 crcResult = CRC_calcResult32(CRC_BASE, pTmpData, length/4);

	 if(tmpLegenth != 0)
	 {
		 type.InitValue = crcResult;
		 crcResult =  Crc32Calc_BySoftware(pData +length - tmpLegenth, tmpLegenth, type);
	 }
	 else
	 {
	 	crcResult^= type.xor;
	 }
	 return crcResult;
 }

 void TestCrcCalc(void)
 {
	 testCrc32Result = Crc32Calc_Misaligned(testBuffer, 15, crc_32_AUTOSAR);
	 testCrc32Result1 = Crc32Calc_BySoftware(testBuffer, 15, crc_32_AUTOSAR);
//	 testCrc32Result2 = Crc32Calc_ByCrcModule(testBuffer, 15, crc_32_AUTOSAR);

	 testCrc16Result = Crc16Calc_Misaligned(testBuffer, 15, crc_16_MODBUS);
	 testCrc16Result1 = Crc16Calc_BySoftware(testBuffer, 15, crc_16_MODBUS);

	 testCrc8Result = Crc8Calc_Misaligned(testBuffer, 15, crc_8_ROHC);
	 testCrc8Result1 = Crc8Calc_BySoftware(testBuffer, 15, crc_8_ROHC);

 }

 /* Private functions ---------------------------------------------------------*/
 /**
   * @brief
   * @param  None
   * @retval None
   */
 int main(void)
 {
     uint32_t oldTicks;
     uint32_t cycles = __get_rv_cycle();

     __disable_irq();

     Device_init();

     GPIO_enableWritePin(LED_PIN);
     GPIO_enableWritePin(22);

     UartPrint_init(LOG_SCI_BASE, 115200);

     log_info("Hello DSP300 Template Project!\r\n");
     log_info("Core running @ %d MHz\r\n", DEVICE_SYSCLK_FREQ/1000/1000);
     log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);
     log_info("Reset record: %d, cycles %d\r\n", SysCtl_getDspCpu1RstRecord(), cycles);

     Timer_init();
     TestCrcCalc();

     __enable_irq();

     oldTicks = coreTicks;

     while (1) {
         if (coreTicks != oldTicks) {
             oldTicks = coreTicks;

             if ((oldTicks % 100) == 0) {
                 log_debug("ticks: %d\r\n", coreTicks);
                 GPIO_togglePin(LED_PIN);      //not thread safe
                 GPIO_togglePin(22);
             }
         }
     }

     for(;;);

     return 0;
 }
