/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    buffdac_ex1_enable.c
*   @brief
*   @details
*
*   This example enables DACA_OUT and DACB_OUT, and loopback to ADC_A0/1 input;
*
*   DACA_OUT is the same pin as ADC_A0;
*   DACB_OUT is the same pin as ADC_A1;
*   DACA output is configured as a sinusoidal voltage.f(x)=sin(x)+1.6
*   DACB output is configured to increment from 0 to VREFHI, and back to 0.
*
*   ADCA is initialized to sample the DAC output.
*   Watch the variable ADCA0_Result and ADCA1_Result in live view and Graph
*   to see the actual DAC output signals.
*   ADC_A0 and ADC_A1 should not be connected to external circuit.
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "printf.h"
#include "gs32_math.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define myDAC0_BASE  DACA_BASE
#define myDAC1_BASE  DACB_BASE
#define myADC0_BASE  ADCA_BASE

#define ADC_SAMPLE_TIME    40U
#define VREFHI_VAL         1.65	// VREFHI
#define SIN_SAMPLE         128 	// 正弦波采样率
/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

uint16_t ADCA0_Result[128];
uint16_t ADCA1_Result[128];

uint32_t AdcResultIdx;
uint16_t g_dac_sin_array[SIN_SAMPLE];

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

uint16_t DACA_Val = 0;
uint16_t DACB_Val = 4095;

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/*
 * @breif Generates an array of sine waves
 */
void DAC_CreateSinArray(float valAmplitude,float valDisplacement, uint16_t samples)
{
	/*
	 * Sine amplitude, since DACA is configured with a gain mode of 2,
	 * the reference voltage here is (VREFHI_VAL*2)
	 */
    float amplitude = valAmplitude * 4095 / (VREFHI_VAL * 2);
    /* Sinusoidal displacement */
    float displacement = valDisplacement * 4095 / (VREFHI_VAL * 2);
    /* Calculate the X-axis spacing between two adjacent points */
    float increment = (2*3.141592653589793) / samples;
    /* Store the calculated amount of numbers */
    uint16_t outputData = 0;

    for (uint8_t i = 0; i < samples; i++)
    {
        /* Sine wave analytic expression:y=Asin(ωx+Φ)+b*/
        outputData = amplitude * __sin(i * increment)+ displacement;
        outputData = (outputData > 4095) ? 4095 : outputData;
        g_dac_sin_array[i] = outputData;
    }
}

/* DAC Configurations */
void myDAC0_init(void)
{
	/* Set DAC reference voltage. */
	DAC_setReferenceVoltage(myDAC0_BASE, DAC_REF_INTERNAL);

	/* Set DAC load mode. */
	DAC_setLoadMode(myDAC0_BASE, DAC_LOAD_SYSCLK);

	/* configured with a gain mode of 2 */
	DAC_setGainMode(myDAC0_BASE, DAC_GAIN_TWO);

	/* Enable the DAC output */
	DAC_enableOutput(myDAC0_BASE);

	/* Enable the DAC */
	DAC_enable(myDAC0_BASE);

	/* Set the DAC shadow output */
	DAC_setShadowValue(myDAC0_BASE, 0U);

	/* Delay for buffered DAC to power up. */
	DEVICE_DELAY_US(500);
}

void myDAC1_init(void)
{
	/* Set DAC reference voltage. */
	DAC_setReferenceVoltage(myDAC1_BASE, DAC_REF_INTERNAL);

	/* Set DAC load mode. */
	DAC_setLoadMode(myDAC1_BASE, DAC_LOAD_SYSCLK);


	/* configured with a gain mode of 1 */
	DAC_setGainMode(myDAC1_BASE, DAC_GAIN_ONE);

	/* Enable the DAC output */
	DAC_enableOutput(myDAC1_BASE);

	/* Enable the DAC */
	DAC_enable(myDAC1_BASE);

	/* Set the DAC shadow output */
	DAC_setShadowValue(myDAC1_BASE, 0U);

	/* Delay for buffered DAC to power up. */
	DEVICE_DELAY_US(500);
}

/* ADCA Configurations for sampling DACA_OUT */
void myADC0_init()
{
	/* Set ADC AnaSysctrl */
	ADC_setVREF(myADC0_BASE,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);

	ADC_setINLTrim(myADC0_BASE);

	/* Configures the analog-to-digital converter module prescaler */
	ADC_setPrescaler(myADC0_BASE, ADC_CLK_DIV_4_0);

	/* Sets the timing of the end-of-conversion pulse */
	ADC_setInterruptPulseMode(myADC0_BASE, ADC_PULSE_END_OF_CONV);

	/* Powers up the analog-to-digital converter core. */
	ADC_enableConverter(myADC0_BASE);

	/* Delay for 5ms to allow ADC time to power up */
	DEVICE_DELAY_US(5000);

	/*
	 * Start of Conversion 0 Configuration
	 *
	 * Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	 *  	  	SOC number		: 0
	 *  	  	Trigger			: ADC_TRIGGER_CPU1_TINT0
	 *  	  	Channel			: ADC_CH_ADCIN0
	 *  	  	Sample Window	: 50 SYSCLK cycles
	 *  	  	Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	ADC_setupSOC(myADC0_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_CPU1_TINT0, ADC_CH_ADCIN0, ADC_SAMPLE_TIME);
	ADC_setInterruptSOCTrigger(myADC0_BASE, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);

	/*
	 * Start of Conversion 1 Configuration
	 *
	 * Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	 *  	  	SOC number		: 1
	 *  	  	Trigger			: ADC_TRIGGER_SW_ONLY
	 *  	  	Channel			: ADC_CH_ADCIN1
	 *  	  	Sample Window	: 50 SYSCLK cycles
	 *  	  	Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	ADC_setupSOC(myADC0_BASE, ADC_SOC_NUMBER1, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN1, ADC_SAMPLE_TIME);
	ADC_setInterruptSOCTrigger(myADC0_BASE, ADC_SOC_NUMBER1, ADC_INT_SOC_TRIGGER_NONE);

	/*
	 * ADC Interrupt 1 Configuration
	 * 		SOC/EOC number	: 1
	 * 		Interrupt Source: enabled
	 * 		Continuous Mode	: disabled
	 */
	ADC_setInterruptSource(myADC0_BASE, ADC_INT_NUMBER1, ADC_SOC_NUMBER1);
	ADC_enableInterrupt(myADC0_BASE, ADC_INT_NUMBER1);
	ADC_clearInterruptStatus(myADC0_BASE, ADC_INT_NUMBER1);
	ADC_disableContinuousMode(myADC0_BASE, ADC_INT_NUMBER1);
}


//
// Main
//
void main(void)
{
	/* Initialize device clock and peripherals */
    Device_init();

    UartPrint_init(LOG_SCI_BASE,115200);
	printf("SCI print OK.\r\n");

    /*
     * Board initialization
     * Configure DAC - Setup the reference voltage and output value for the DAC
     */
	DAC_CreateSinArray(1.0,1.6,SIN_SAMPLE);		// f(x)=1.0sin(x)+1.6
	myDAC0_init();
	myDAC1_init();

	myADC0_init();
	uint16_t i = 0;

    /* Continuously set the DAC output value */
    while(1)
    {
        /* Set DAC1 output value - 0~4095 = 0V~VREFHI_VAL */
    	DAC_setShadowValue(myDAC0_BASE, DACA_Val);
    	DAC_setShadowValue(myDAC1_BASE, DACB_Val);
        DEVICE_DELAY_US(10);

        /* Software trigger ADC conversion SOC0 and SOC1 */
        ADC_forceMultipleSOC(myADC0_BASE, ADC_FORCE_SOC0 | ADC_FORCE_SOC1);
        /* wait till ADC result ready */
        while(ADC_getInterruptStatus(myADC0_BASE, ADC_INT_NUMBER1) == 0);
        /* read ADC_A0 result */
        ADCA0_Result[AdcResultIdx] = ADC_readResult(ADCARESULT_BASE, ADC_SOC_NUMBER0);
        /* read ADC_A1 result (it's for DACB_OUT test) */
        ADCA1_Result[AdcResultIdx] =ADC_readResult(ADCARESULT_BASE, ADC_SOC_NUMBER1);
        /* clear ADC int flag */
        ADC_clearInterruptStatus(myADC0_BASE, ADC_INT_NUMBER1);


        printf("ADCA0_result:%.3fV.\r\n",((float)ADCA0_Result[AdcResultIdx]/4095.0)*3.3);
        printf("ADCA1_result:%.3fV.\r\n",((float)ADCA1_Result[AdcResultIdx]/4095.0)*3.3);

        AdcResultIdx++;
        AdcResultIdx &= (128-1);

        /* DAC_A output value */
        DACA_Val = g_dac_sin_array[i++];
        if(i>=SIN_SAMPLE){i=0;}

        /* decrement DAC_B output value */
        DACB_Val -= (4096/128);
        DACB_Val %= 4096;

        /* delay 3907uS for next cycle, 3907uS * 128cycles = 0.5S for one period */
        DEVICE_DELAY_US(7813/2);	// 2Hz
    }
}

//
// End of File
//

#ifdef __cplusplus
}
#endif



