/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    xdma_dsp_ex01_ecap_pwm.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "board_cfg.h"
#include "xdma_dsp_ex01_ecap_pwm_board.h"
#include "log.h"
#include "load_img.h"
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define PWM_FREQUENCY       10000U // 10 Khz
#define PWM_DUTY            0.3f   // 30% duty
#define PWM_PRD_VAL         20000000U / PWM_FREQUENCY
#define PWM_CMP_VAL         (uint32_t)(PWM_DUTY * PWM_PRD_VAL)
#define ECAP_COUNT 16

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

vuint16_t cap3Count;
vuint16_t cap4Count;
uint32_t ecapData[ECAP_COUNT*2] = {0};

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/**
 * \brief   "// initEPWM - Configure ePWM" 
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void initEPWM(uint32_t base)
{
    //EPWM_setEmulationMode(base, EPWM_EMULATION_STOP_AFTER_FULL_CYCLE);

    //
    // Set-up TBCLK
    //
//    EPWM_setTimeBasePeriod(base, PWM_FREQUENCY);
//    EPWM_setPhaseShift(base, 0U);
//    EPWM_setTimeBaseCounter(base, 0U);

    //
    // Set up counter mode
    //
//    EPWM_setTimeBaseCounterMode(base, EPWM_COUNTER_MODE_UP);
//    EPWM_disablePhaseShiftLoad(base);

    EPWM_setClockPrescaler(base,
                           EPWM_CLOCK_DIVIDER_2,
                           EPWM_HSCLOCK_DIVIDER_2);
    //EPWM_setTimeBasePeriod(base, PWM_PRD_VAL);
    EPWM_setTimeBasePeriod(base, 2000);
    EPWM_setTimeBaseCounter(base,0U);
    EPWM_setTimeBaseCounterMode(base, EPWM_COUNTER_MODE_UP);
    EPWM_disablePhaseShiftLoad(base);
    EPWM_setPhaseShift(base, 0U);
    //EPWM_setCounterCompareValue(base,EPWM_COUNTER_COMPARE_A,PWM_CMP_VAL);
    EPWM_setCounterCompareValue(base,EPWM_COUNTER_COMPARE_A,1000);
    EPWM_setCounterCompareShadowLoadMode(base,EPWM_COUNTER_COMPARE_A,EPWM_COMP_LOAD_ON_CNTR_ZERO);
    //EPWM_enableSyncOutPulseSource(base,EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);
    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);

    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
    EPWM_setHrpwmDllCfg0(PREEPWM_BASE,0x1);
    // Enable sync and clock to PWM
    //
    // SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}

/**
 * \brief   "// initDMA - Configure DMA" 
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void initDMA()
{
    XDMA_ConfigParams dmaCfg = {0};
        
    XDMA_initController(DMA1_BASE);

    /* Stop XDMA channel first - why the DMA channel is still pending after reset? */
    XDMA_stopChannel(DMA1_CH1_BASE);
    dmaCfg.enableInterrupt = 1;
    dmaCfg.dmaSrcReqId = ReqID_ECAP; 
    dmaCfg.srcAddr = (myECAP_BASE + ECAP_O_CAP1); // ecap3寄存器地址
    dmaCfg.destAddr = (uint32_t)ecapData;
    dmaCfg.blockTS = 4;  // 每次只传?2个数据，ecap3和ecap4各一个数?
    dmaCfg.ttfc    = XDMA_TT_FC_2_P2M_DMAC;
    dmaCfg.srcBtl  = XDMA_BTL_4;
    dmaCfg.destBtl = XDMA_BTL_4;
    dmaCfg.srcAddrDirect = XDMA_ADDR_INCRE; // 需要传输ecap3和ecap4两个寄存器的数据，所以设置为增长
    dmaCfg.destAddrDirect = XDMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = XDMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes= XDMA_TR_WIDTH_BYTE_4;
    dmaCfg.reloadSrc = XDMA_RELOAD; // 设置源地址自动reload，传输完可产生block中断，不会产生tfr中断
    
    XDMA_configChannel(DMA1_CH1_BASE, &dmaCfg);

    /* Clear INT flag set in previous operation */
    /* Enable the IRQ so that we can poll the Status_Tfr flag */
    XDMA_clearInterrupt(DMA1_CH1_BASE, XDMA_INT_TFR | XDMA_INT_BLOCK | XDMA_INT_ERR);
    XDMA_unMaskInterrupt(DMA1_CH1_BASE, XDMA_INT_TFR | XDMA_INT_BLOCK | XDMA_INT_ERR);

    /* XDMA channel enable */
    XDMA_startChannel(DMA1_CH1_BASE);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

void main(void){

    uint8_t status = 0;
    uint8_t reload_cnt = 0;
    uint32_t tmp32;

    __disable_irq();
    
#if ((GS32F3xx & 0xFF00) || (GS32F00xx & 0xFF00))
    Device_init();
#endif

#if ((GS32F3xx & 0xFF) || (GS32F00xx & 0xFF))
    //
    // config arm clock different from RISCV
    //
    HWREG(CRG_CFG_BASE + 0x008) = 1;        //Slow 2 normal
    HWREG(CRG_CFG_BASE + 0x010) = 0x0;      //arm clk div = 0
    HWREG(CRG_CFG_BASE + 0x01C) = 1;        //update
#endif

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    
#if defined(LOAD_ARM)
    bringup_arm();
#endif
    __enable_irq();

    printf("hello: %s \r\n", __FILE__);

    // Board initialization
    // Configure GPIO 16 as eCAP input
    // Enable interrupts required for this example
    //
    Board_init();

    // Initialize counters
    cap3Count = 0U;
    cap4Count = 0U;

#if ((GS32F3xx & 0xFF) || (GS32F00xx & 0xFF))
    //
    // config for pwm output
    //
    //HWREG(0x4003F000 + 0x120) = 0x10000;
    HWREG(0x4003F000 + 0x44) = 0x40000;
    HWREG(0x40008000 + 0x31C) = 0x3;
#endif

    // Configure ePWM
    initEPWM(myEPWM_BASE);

    // Configure DMA
    initDMA();

    while((status & XDMA_INT_TFR) == 0)
        {
            status = XDMA_getInterruptStatus(DMA1_CH1_BASE);
            if (status & XDMA_INT_BLOCK)
            {
                reload_cnt++;
                if (reload_cnt == ECAP_COUNT - 1)
                    XDMA_disableReload(DMA1_CH1_BASE);//传输最后一个数据前，关闭reload，则最后一个数据传输完可产生TFR中断
                XDMA_clearInterrupt(DMA1_CH1_BASE, XDMA_INT_BLOCK); //传输完blockts个数据后，源寄存器auto reload，通过clear中断触发再次传输
            }
        }

    // Loop forever. Suspend or place breakpoints to observe the buffers.
    while(1)
    {
        cap3Count = ECAP_getEventTimeStamp(myECAP_BASE, ECAP_EVENT_3);
        cap4Count = ECAP_getEventTimeStamp(myECAP_BASE, ECAP_EVENT_4);
    }
}

/**
 * \brief   "add brief description of this function" 
 *
 * \param   parameter1   description of parameter1
 * \param   parameter2   description of parameter2
 *
 * \retval  None
 */

#ifdef __cplusplus
}
#endif
