/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    xdma_dsp_ex02_adc_soc_continuous.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif


/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "xdma_dsp_ex02_adc_soc_continue_board.h"
#include "load_img.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define RESULTS_BUFFER_SIZE     (512)
#define CHECK_SIZE              (32)

#if (GS32F3xx == 0x2000 || GS32F3xx == 0x0020)
#define DMA_INT_IRQn            (INT_DSP_DMA)
#else
#define DMA_INT_CH1_IRQn        (INT_DMA1_CH1)
#define DMA_INT_CH2_IRQn        (INT_DMA1_CH2)
#define DMA_INT_CH3_IRQn        (INT_DMA1_CH3)
#define DMA_INT_CH4_IRQn        (INT_DMA1_CH4)
#define DMA_INT_CH5_IRQn        (INT_DMA1_CH5)
#define DMA_INT_CH6_IRQn        (INT_DMA1_CH6)
#define DMA_INT_CH7_IRQn        (INT_DMA1_CH7)
#define DMA_INT_CH8_IRQn        (INT_DMA1_CH8)
#endif

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

uint16_t myADC0DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC1DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC2DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC3DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC4DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC5DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC6DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC7DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];

volatile uint16_t done_chanel1;
volatile uint16_t done_chanel2;
volatile uint16_t done_chanel3;
volatile uint16_t done_chanel4;
volatile uint16_t done_chanel5;
volatile uint16_t done_chanel6;
volatile uint16_t done_chanel7;
volatile uint16_t done_chanel8;

volatile uint16_t idx_chanel1;
volatile uint16_t idx_chanel2;
volatile uint16_t idx_chanel3;
volatile uint16_t idx_chanel4;
volatile uint16_t idx_chanel5;
volatile uint16_t idx_chanel6;
volatile uint16_t idx_chanel7;
volatile uint16_t idx_chanel8;


/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

void Test_adc_ex06(void);
void myADC0_init(void);
void myADC1_init(void);
void configureEPWM(uint32_t epwmBase);
void initializeDMA(void);
void configureDMAChannels(void);
void deConfDMAChannels(void);
void disableDMAreload(uint32_t DMA_CH_BASE);

void DMA_CH1_IRQHandler()
{
    if (idx_chanel1 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel1 = 1;
        XDMA_stopChannel(DMA1_CH1_BASE);
    }
    idx_chanel1 ++;
    XDMA_clearInterrupt(DMA1_CH1_BASE, XDMA_INT_BLOCK);
    __DSB();
}

void DMA_CH2_IRQHandler()
{
    if (idx_chanel2 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
       done_chanel2 = 1;
       XDMA_stopChannel(DMA1_CH2_BASE);
    
    }

    idx_chanel2 ++;
    XDMA_clearInterrupt(DMA1_CH2_BASE, XDMA_INT_BLOCK);
    __DSB();
}

void DMA_CH3_IRQHandler()
{
    if(idx_chanel3 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel3 = 1;
        XDMA_stopChannel(DMA1_CH3_BASE);
    }

    idx_chanel3++;
    
    XDMA_clearInterrupt(DMA1_CH3_BASE, XDMA_INT_BLOCK);
    __DSB();
}

void DMA_CH4_IRQHandler()
{
    if (idx_chanel4 >= (RESULTS_BUFFER_SIZE/16 -1))
    {
        done_chanel4 = 1;
        XDMA_stopChannel(DMA1_CH4_BASE);
    }
    idx_chanel4++;
    
    XDMA_clearInterrupt(DMA1_CH4_BASE, XDMA_INT_BLOCK);
    __DSB();
}

void DMA_CH5_IRQHandler()
{
    if (idx_chanel5 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel5 = 1;
        XDMA_stopChannel(DMA1_CH5_BASE);
    }
    idx_chanel5 ++;
    XDMA_clearInterrupt(DMA1_CH5_BASE, XDMA_INT_BLOCK);
    __DSB();
}

void DMA_CH6_IRQHandler()
{
    if (idx_chanel6 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel6 = 1;
        XDMA_stopChannel(DMA1_CH6_BASE);
    }
    idx_chanel6++;
    XDMA_clearInterrupt(DMA1_CH6_BASE, XDMA_INT_BLOCK);
    __DSB();
}

void DMA_CH7_IRQHandler()
{
    if (idx_chanel7 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel7 = 1;
        XDMA_stopChannel(DMA1_CH7_BASE);
    }
    idx_chanel7++;
    XDMA_clearInterrupt(DMA1_CH7_BASE, XDMA_INT_BLOCK);
    __DSB();
}

void DMA_CH8_IRQHandler()
{
    if ((done_chanel1 == 1) &&
         (done_chanel2 == 1) &&
         (done_chanel3 == 1) &&
         (done_chanel4 == 1) &&
         (done_chanel5 == 1) &&
         (done_chanel6 == 1) &&
         (done_chanel7 == 1))
    {
        done_chanel8 = 1;
        XDMA_stopChannel(DMA1_CH8_BASE);
    }
    idx_chanel8++;
    XDMA_clearInterrupt(DMA1_CH8_BASE, XDMA_INT_BLOCK);
    __DSB();
}


void DSP_IRQ_Rigster(IRQn_Type idx, void* handler)
{
    CIDU_BroadcastExtInterrupt(IRQn_MAP_TO_EXT_ID(idx), CIDU_RECEIVE_INTERRUPT_EN(0));   //cpu1 only
    CIDU_ResetFirstClaimMode(IRQn_MAP_TO_EXT_ID(idx));
    ECLIC_Register_IRQ(idx, ECLIC_NON_VECTOR_INTERRUPT, ECLIC_LEVEL_TRIGGER, 1, 0, handler);
}


/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/**
 * @brief   "add brief description of this function"
 *
 * @param   parameter1   description of parameter1
 * @param   parameter2   description of parameter2
 *
 * @retval  None
 */
void Test_adc_ex06(void){

    uint16_t resultsIndex;

    __disable_irq();

    /* Board Initializatrion
     *   - Configure the ADCA & ADCC and power it up
     *   - Setup the ADC for continuous conversions on channels A3 and C3
     *   - Set up ISR for ADCA INT1 - occurs after first conversion
     *   - Enable specific PIE & CPU interrupts: ADCA INT1 - Group 1, interrupt 1

     */
    Board_init();

    /*  Enable specific PIE & CPU interrupts:
     * DMA interrupt - Group 7, interrupt 1
     */
    DSP_IRQ_Rigster(DMA_INT_CH2_IRQn, DMA_CH1_IRQHandler);
    DSP_IRQ_Rigster(DMA_INT_CH3_IRQn, DMA_CH2_IRQHandler);
    DSP_IRQ_Rigster(DMA_INT_CH4_IRQn, DMA_CH3_IRQHandler);
    DSP_IRQ_Rigster(DMA_INT_CH5_IRQn, DMA_CH4_IRQHandler);
    DSP_IRQ_Rigster(DMA_INT_CH6_IRQn, DMA_CH5_IRQHandler);
    DSP_IRQ_Rigster(DMA_INT_CH7_IRQn, DMA_CH6_IRQHandler);
    DSP_IRQ_Rigster(DMA_INT_CH8_IRQn, DMA_CH7_IRQHandler);
    DSP_IRQ_Rigster(DMA_INT_CH1_IRQn, DMA_CH8_IRQHandler);

    //
    // Stop the ePWM clock
    //
    /* Stop the ePWM clock */
    //SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    /* Call the set up function for ePWM 2 */
    //configureEPWM(EPWM2_BASE);

    /* Start the ePWM clock */
    //SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    /* Initialize the DMA & configure DMA channels 1 & 2 */
    initializeDMA();
    configureDMAChannels();
    /* Initialize results buffer */
    for(resultsIndex = 0; resultsIndex < RESULTS_BUFFER_SIZE + CHECK_SIZE; resultsIndex++)
    {
        myADC0DataBuffer[resultsIndex] = 0xA55A;
        myADC1DataBuffer[resultsIndex] = 0xA55A;
        myADC2DataBuffer[resultsIndex] = 0xA55A;
        myADC3DataBuffer[resultsIndex] = 0xA55A;
        myADC4DataBuffer[resultsIndex] = 0xA55A;
        myADC5DataBuffer[resultsIndex] = 0xA55A;
        myADC6DataBuffer[resultsIndex] = 0xA55A;
        myADC7DataBuffer[resultsIndex] = 0xA55A;
    }

    HWREGH(testADC_BASE + ADC_O_INTFLGCLR) = 0x3U; // ADCA

    /* Start DMA */
    done_chanel1 = 0;
    done_chanel2 = 0;
    done_chanel3 = 0;
    done_chanel4 = 0;
    done_chanel5 = 0;
    done_chanel6 = 0;
    done_chanel7 = 0;
    done_chanel8 = 0;

    idx_chanel1 = 0;
    idx_chanel2 = 0;
    idx_chanel3 = 0;
    idx_chanel4 = 0;
    idx_chanel5 = 0;
    idx_chanel6 = 0;
    idx_chanel7 = 0;
    idx_chanel8 = 0;

    __enable_irq();

    XDMA_startChannel(DMA1_CH1_BASE);
    XDMA_startChannel(DMA1_CH2_BASE);
    XDMA_startChannel(DMA1_CH3_BASE);
    XDMA_startChannel(DMA1_CH4_BASE);
    XDMA_startChannel(DMA1_CH5_BASE);
    XDMA_startChannel(DMA1_CH6_BASE);
    XDMA_startChannel(DMA1_CH7_BASE);
    XDMA_startChannel(DMA1_CH8_BASE);

    /*  Loop until the ISR signals the transfer is complete */
    while((done_chanel1 == 0) ||
           (done_chanel2 == 0) ||
           (done_chanel3 == 0) ||
           (done_chanel4 == 0) ||
           (done_chanel5 == 0) ||
           (done_chanel6 == 0) ||
           (done_chanel7 == 0) ||
           (done_chanel8 == 0))
    {

        ADC_forceMultipleSOC(testADC_BASE, 0xFFFF);
        DEVICE_DELAY_US(5000);
        __asm(" NOP");
    }


    deConfDMAChannels();
    for(;;);

}

void initializeDMA(void)
{
	/* Perform a hard reset on DMA */
    XDMA_initController(DMA1_BASE);
}

/* configureDMAChannels - Initialize DMA ch 0~ch6 to transfer ADC results
 * and DMA ch 7 to transfer adc PPB EVT esults
 */
void configureDMAChannels(void)
{
    XDMA_ConfigParams dmaCfg = {0};
    dmaCfg.enableInterrupt = 1U;
    //enable DMA Trigger by DMA MUX
    dmaCfg.dmaSrcReqId = testDMAMUX_Request_ID;
    dmaCfg.srcAddr = (uint32_t)testADC_RESULT;
    dmaCfg.destAddr = (uint32_t)myADC0DataBuffer;
    dmaCfg.blockTS = 16;
    dmaCfg.ttfc    = XDMA_TT_FC_2_P2M_DMAC;
    dmaCfg.srcBtl  = XDMA_BTL_16;
    dmaCfg.reloadSrc = XDMA_RELOAD;
    dmaCfg.destBtl = XDMA_BTL_16;
    dmaCfg.srcAddrDirect = XDMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = XDMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = XDMA_TR_WIDTH_BYTE_2;
    dmaCfg.destTrWidthBytes= XDMA_TR_WIDTH_BYTE_2;
    XDMA_configChannel(DMA1_CH1_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH1_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH1_BASE, XDMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC1DataBuffer;
    XDMA_configChannel(DMA1_CH2_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH2_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH2_BASE, XDMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC2DataBuffer;
    XDMA_configChannel(DMA1_CH3_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH3_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH3_BASE, XDMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC3DataBuffer;
    XDMA_configChannel(DMA1_CH4_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH4_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH4_BASE, XDMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC4DataBuffer;
    XDMA_configChannel(DMA1_CH5_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH5_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH5_BASE, XDMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC5DataBuffer;
    XDMA_configChannel(DMA1_CH6_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH6_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH6_BASE, XDMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC6DataBuffer;
    XDMA_configChannel(DMA1_CH7_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH7_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH7_BASE, XDMA_INT_BLOCK);

    dmaCfg.dmaSrcReqId = testDMAMUX_Request_ID_Evt;
    dmaCfg.destAddr = (uint32_t)myADC7DataBuffer;
    XDMA_configChannel(DMA1_CH8_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH8_BASE, XDMA_INT_BLOCK);
    XDMA_unMaskInterrupt(DMA1_CH8_BASE, XDMA_INT_BLOCK);
}

/* eConfDMAChannels - Deinitialize DMA ch 0~ch6 to transfer ADC results
 *  and DMA ch 7 to transfer adc PPB EVT esults
 */
void deConfDMAChannels(void)
{
    XDMA_ConfigParams dmaCfg = {0};
    dmaCfg.dmaSrcReqId = testDMAMUX_Request_ID;
    dmaCfg.ttfc    = XDMA_TT_FC_2_P2M_DMAC;

    XDMA_DeConfChannel(DMA1_CH1_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH1_BASE, XDMA_INT_BLOCK);
    XDMA_maskInterrupt(DMA1_CH1_BASE, XDMA_INT_BLOCK);
    
    XDMA_DeConfChannel(DMA1_CH2_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH2_BASE, XDMA_INT_BLOCK);
    XDMA_maskInterrupt(DMA1_CH2_BASE, XDMA_INT_BLOCK);

    XDMA_DeConfChannel(DMA1_CH3_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH3_BASE, XDMA_INT_BLOCK);
    XDMA_maskInterrupt(DMA1_CH3_BASE, XDMA_INT_BLOCK);

    XDMA_DeConfChannel(DMA1_CH4_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH4_BASE, XDMA_INT_BLOCK);
    XDMA_maskInterrupt(DMA1_CH4_BASE, XDMA_INT_BLOCK);

    XDMA_DeConfChannel(DMA1_CH5_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH5_BASE, XDMA_INT_BLOCK);
    XDMA_maskInterrupt(DMA1_CH5_BASE, XDMA_INT_BLOCK);

    XDMA_DeConfChannel(DMA1_CH6_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH6_BASE, XDMA_INT_BLOCK);
    XDMA_maskInterrupt(DMA1_CH6_BASE, XDMA_INT_BLOCK);

    XDMA_DeConfChannel(DMA1_CH7_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH7_BASE, XDMA_INT_BLOCK);
    XDMA_maskInterrupt(DMA1_CH7_BASE, XDMA_INT_BLOCK);

	dmaCfg.dmaSrcReqId = testDMAMUX_Request_ID_Evt;
    XDMA_DeConfChannel(DMA1_CH8_BASE, &dmaCfg);
    XDMA_clearInterrupt(DMA1_CH8_BASE, XDMA_INT_BLOCK);
    XDMA_maskInterrupt(DMA1_CH8_BASE, XDMA_INT_BLOCK);
}

int main(void)
{
    int ret;
    
#if ((GS32F3xx & 0xFF00) || (GS32F00xx & 0xFF00))
    Device_init();
#endif

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_INFO);

    log_info("Hello Cortex-M4!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);

#if defined(LOAD_ARM)
    bringup_arm();
#endif

    __enable_irq();
    Test_adc_ex06();

    /* We should never get here as control is now taken by the scheduler */
    for(;;);
}

#ifdef __cplusplus
}
#endif

