/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* Includes ------------------------------------------------------------------*/
#include <sys/stat.h>

#include "device.h"
#include "board_cfg.h"
#include "load_img.h"
#include "log.h"
#include "gs32_math.h"

#include "sram.h"

#define LED_PIN                LED1_PIN

#define FSMC_MEM_BASE          0x78000000
#define TEST_LEN               256

uint8_t buf[TEST_LEN];

/* Private functions ---------------------------------------------------------*/
/**
  * @brief
  * @param  None
  * @retval None
  */
volatile uint32_t coreTicks = 0;

//void TIMER0_INT_IRQHandler(void)
//{
//    coreTicks += 1;
//	CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
//	NVIC_ClearPendingIRQ(INT_TIMER0);
//}

void InitIoForDSP(void)
{
    GPIO_setPinConfig(SCIB_TX_PIN);
    GPIO_setPinConfig(SCIB_RX_PIN);

    SysCtl_PinMux_setIOMux(LED2_PIN, 0);
}

void Mem_show(void * mem, uint32_t len)
{
    uint32_t i;

    printf("--- 8bits reading:\r\n");
    for (i=0; i<len; i+=1) {
        if ((i%16) == 0) {
            printf("%08X: ", ((uint32_t)mem) + i);
        }

        printf("%02X ", ((uint8_t *)mem)[i]);

        if ((i%16) == 15) {
            printf("\r\n");
        }
    }

    printf("--- 16bits reading:\r\n");
    for (i=0; i<len/2; i+=1) {
        if ((i%8) == 0) {
            printf("%08X: ", ((uint32_t)mem) + i*2);
        }

        printf("%04X ", ((uint16_t *)mem)[i]);

        if ((i%8) == 7) {
            printf("\r\n");
        }
    }

    printf("--- 32bits reading:\r\n");
    for (i=0; i<len/4; i+=1) {
        if ((i%4) == 0) {
            printf("%08X: ", ((uint32_t)mem) + i*4);
        }

        printf("%08X ", ((uint32_t *)mem)[i]);

        if ((i%4) == 3) {
            printf("\r\n");
        }
    }
}

void FSMC_test(void)
{
    volatile uint8_t *byte = (uint8_t *)FSMC_MEM_BASE;
    volatile uint16_t *halfword = (uint16_t *)FSMC_MEM_BASE;
    volatile uint32_t *word = (uint32_t *)FSMC_MEM_BASE;
    uint32_t error = 0;
    printf("---------- 8bit test -------\r\n");
    for (uint32_t i=0; i<TEST_LEN; i+=1) {
        byte[i] = i;
    }

    Mem_show(byte, TEST_LEN);
    for(uint32_t i = 0; i < TEST_LEN; i++)
    {

    	if(byte[i] != i)
    	{
    		error++;
    	}
    }
    if(error){
    	printf("8bit test error\r\n");
    	error = 0;
    }
    else {
		printf("8bit test success\r\n");
	}


    printf("---------- 16bits test -------\r\n");
    for (uint32_t i=0; i<TEST_LEN/2; i+=1) {
        halfword[i] = i;
    }

    Mem_show(halfword, TEST_LEN);
    for(uint32_t i = 0; i < TEST_LEN / 2; i++)
       {

       	if(halfword[i] != i)
       	{
       		error++;
       	}
       }
       if(error){
       	printf("16bit test error\r\n");
       	error = 0;
       }
       else {
   		printf("16bit test success\r\n");
   	}

    printf("---------- 32bits test -------\r\n");
    for (uint32_t i=0; i<TEST_LEN/4; i+=1) {
        word[i] = i;
    }

    Mem_show(word, TEST_LEN);
    for(uint32_t i = 0; i < TEST_LEN / 4; i++)
       {

       	if(word[i] != i)
       	{
       		error++;
       	}
       }
       if(error){
       	printf("32bit test error\r\n");
       	error = 0;
       }
       else {
   		printf("32bit test success\r\n");
   	}

}
/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
	uint32_t i;

    Device_init();

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP EMIF/FSMC test!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);

    SRAM_GPIO_Config();
	FSMC_Config();

	FSMC_test();

	/* We should never get here as control is now taken by the scheduler */
    for(;;);
}

/*****END OF FILE****/
