/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* Includes ------------------------------------------------------------------*/
#include <sys/stat.h>

#include "device.h"
#include "board_cfg.h"
#include "load_img.h"
#include "log.h"
#include "gs32_math.h"

#include "norflash.h"

#define LED_PIN                LED1_PIN

#define FSMC_MEM_BASE          0x74000000
#define TEST_LEN               512

uint8_t buf[TEST_LEN];

/* Private functions ---------------------------------------------------------*/
/**
  * @brief
  * @param  None
  * @retval None
  */
volatile uint32_t coreTicks = 0;

uint16_t g_CFI_Query[11] = {
0x0051, 0x0052, 0x0059, 0x0002, 0x0000, 0x0040, 0x0000, 0x0000,
0x0000, 0x0000, 0x0000};
uint16_t g_CFI_SystemInterface[12] = {
0x0027, 0x0036, 0x0000, 0x0000, 0x0003, 0x0000, 0x0009, 0x0000,
0x0005, 0x0000, 0x0004, 0x0000};
uint16_t g_CFI_DeviceGeometryDefinition[22] = {
0x0015, 0x0002, 0x0000, 0x0000, 0x0000, 0x0004, 0x0000, 0x0000,
0x0040, 0x0000, 0x0001, 0x0000, 0x0020, 0x0000, 0x0000, 0x0000,
0x0080, 0x0000, 0x001E, 0x0000, 0x0000, 0x0001};
uint16_t g_CFI_PrimaryVendorSpecific[17] = {
0x0050, 0x0052, 0x0049, 0x0031, 0x0033, 0x000C, 0x0002, 0x0001,
0x0001, 0x0004, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0002,
0x0000};

void NorFlash_Program(uint16_t addr, uint16 data);
void NorFlash_ScanCFI(void);
void NorFlash_test();
void NorFlash_ChipErase(void);

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
	uint32_t i;

    Device_init();

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP EMIF/FSMC test!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);

    NORFLASH_GPIO_Config();
	FSMC_Config();
	NorFlash_ChipErase();
	NorFlash_ScanCFI();
	NorFlash_test();

	/* We should never get here as control is now taken by the scheduler */
    for(;;);
}

void NorFlash_test()
{
	uint16_t offset = 0;
	uint16_t halfword[TEST_LEN] = {0};
	uint16_t error = 0;
	for(uint16_t i = 0; i < TEST_LEN; i++)
	{
		NorFlash_Program(offset + i, i);
		halfword[i] = Nor_dat(offset + i);
	}
	for(uint16_t i = 0; i < TEST_LEN; i++)
	{
		if(halfword[i] != i)
		{
			error++;
		}
		if((i % 16) == 0)
		{
			printf("%08X: ", (uint32_t)(NORFLASH_BASE + offset + i));
		}
		printf("%04X ", halfword[i]);
		if((i % 16) == 15)
		{
			printf("\r\n");
		}
	}
	if(error)
	{
		printf("%d error\r\n", error);
	}else{
		printf("test success\r\n");
	}

}
void NorFlash_wait_ready(uint16_t addr)
{
	uint16_t val, pre;

	pre = Nor_dat(addr);
	val = Nor_dat(addr);
	while((val & (1 << 6)) != (pre & (1 << 6)))
	{
		pre = val;
		val = Nor_dat(addr);
	}

}
void NorFlash_Program(uint16_t addr, uint16 data)
{
	Nor_cmd(0x555, 0xaa);
	Nor_cmd(0x2aa, 0x55);
	Nor_cmd(0x555, 0xa0);
	Nor_cmd(addr, data);
	DELAY_US(10);
	NorFlash_wait_ready(addr);

}
// Ƭܻ
void NorFlash_ChipErase(void)
{
	Nor_cmd(0x555, 0xaa);
	Nor_cmd(0x2aa, 0x55);
	Nor_cmd(0x555, 0x80);
	Nor_cmd(0x555, 0xaa);
	Nor_cmd(0x2aa, 0x55);
	Nor_cmd(0x555, 0x10);
	DELAY_US(500000);
	NorFlash_wait_ready(0x00);
}
// 
void NorFlash_SectorErase(uint16_t addr)
{
	Nor_cmd(0x555, 0xaa);
	Nor_cmd(0x2aa, 0x55);
	Nor_cmd(0x555, 0x80);
	Nor_cmd(0x555, 0xaa);
	Nor_cmd(0x2aa, 0x55);
	Nor_cmd(addr, 30);
	DELAY_US(500000);
	NorFlash_wait_ready(0x00);
}
void NorFlash_ScanCFI(void)
{
	uint16_t CFI_Query[11] = {0};
	uint16_t CFI_SystemInterface[12] = {0};
	uint16_t CFI_DeviceGeometryDefinition[22] = {0};
	uint16_t CFI_PrimaryVendorSpecific[17] = {0};
	uint16_t vendor, device;
	uint16_t offset = 0x10;
	uint16_t i = 0;
	uint8_t error = 0 , error_flag = 0;
	Nor_cmd(0x555, 0xaa);
	Nor_cmd(0x2aa, 0x55);
	Nor_cmd(0x555, 0x90);
	vendor = Nor_dat(0);
	device = Nor_dat(1);
	Nor_cmd(0, 0xf0);
	Nor_cmd(0x55, 0x98);
	printf("CFI_Query:\r\n");
	for(i = 0; i < 11; i++)
	{
		CFI_Query[i] = Nor_dat(offset);
		offset += 1;
		printf("0x%04X, ", CFI_Query[i]);
		if((i % 8) == 7)
		{
			printf("\r\n");
		}
	}
	printf("\n");
	printf("CFI_SystemInterface:\r\n");
	for(i = 0; i < 12; i++)
	{
		CFI_SystemInterface[i] = Nor_dat(offset);
		offset += 1;
		printf("0x%04X, ", CFI_SystemInterface[i]);
		if((i % 8) == 7)
		{
			printf("\r\n");
		}
	}
	printf("\n");
	printf("CFI_DeviceGeometryDefinition:\r\n");
	for(i = 0; i < 22; i++)
	{
		CFI_DeviceGeometryDefinition[i] = Nor_dat(offset);
		offset += 1;
		printf("0x%04X, ", CFI_DeviceGeometryDefinition[i]);
		if((i % 8) == 7)
		{
			printf("\r\n");
		}
	}
	printf("\n");
	offset = 0x40;
	printf("CFI_PrimaryVendorSpecific:\r\n");
	for(i = 0; i < 17; i++)
	{
		CFI_PrimaryVendorSpecific[i] = Nor_dat(offset);
		offset += 1;
		printf("0x%04X, ", CFI_PrimaryVendorSpecific[i]);
		if((i % 8) == 7)
		{
			printf("\r\n");
		}
	}
	printf("\n");
	Nor_cmd(0, 0xf0);
	for(i = 0; i < 11; i++)
	{
		if(CFI_Query[i] != g_CFI_Query[i])
			error++;
	}
	if(error)
	{
		printf("CFI_Query has encountered %d issues\r\n", error);
		error = 0;
		error_flag++;
	}
	for(i = 0; i < 12; i++)
	{
		if(CFI_SystemInterface[i] != g_CFI_SystemInterface[i])
			error++;
	}
	if(error)
	{
		printf("CFI_SystemInterface has encountered %d issues\r\n", error);
		error = 0;
		error_flag++;
	}
	for(i = 0; i < 22; i++)
	{
		if(CFI_DeviceGeometryDefinition[i] != g_CFI_DeviceGeometryDefinition[i])
			error++;
	}
	if(error){
		printf("CFI_DeviceGeometryDefinition has encountered %d issues\r\n", error);
		error = 0;
		error_flag++;
	}
	for(i = 0; i < 15; i++)
	{
		if(CFI_PrimaryVendorSpecific[i] != g_CFI_PrimaryVendorSpecific[i])
			error++;
		error = 0;
	}
	if(error)
	{
		printf("CFI_PrimaryVendorSpecific has encountered %d issues\r\n", error);
		error = 0;
		error_flag++;
	}
	if(error_flag)
	{
		printf("%d fatal errors \n", error_flag);
	}else
	{
		printf("CFI check 0 error\r\n");
	}

}

/*****END OF FILE****/
