/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file     main.c
*   @brief    template main.c file
*   @details  toggle GPIO to drive LED, print output via UART.
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "driverlib.h"
#include "log.h"            //smp thread safe
#include "board_cfg.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
__SHARED_BSS__ volatile uint32_t cpu1Ticks;
__SHARED_BSS__ volatile uint32_t cpu2Ticks;

/* ========================================================================== */
/*                          External Function Prototypes                      */
/* ========================================================================== */
extern void Shared_show(void);
extern void Shared_bss_clean(void);

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
/*
 * @brief    CPUTimer IRQ handler
 */
#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void TIMER1_IRQHandler(void)
{
    SAVE_IRQ_CSR_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
    cpu2Ticks++;

    RESTORE_IRQ_CSR_CONTEXT();
}
#else
void TIMER1_IRQHandler(void)
{
    SAVE_FPU_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
    cpu2Ticks++;

    RESTORE_FPU_CONTEXT();
}
#endif

/*
 * @brief    CPUTimer initialization
 */
void Timer_init(void)
{
    CPUTimer_init(CPUTIMER1_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER1, TIMER1_IRQHandler);
    Interrupt_enable(INT_TIMER1);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/**
  * @brief    main function
  * @param    None
  * @retval   None
  */
int main(void)
{
    uint32_t oldTicks;

    IPC_init(IPC_CPU2_L_CPU1_R);
    IPC_sync(IPC_CPU2_L_CPU1_R, IPC_FLAG0);

    __disable_irq();

    Shared_bss_clean();

    GPIO_enableWritePin(GPIO_PIN_LED2);

    log_info("Hello DSP300 Dualcore Template Project!\r\n");
    log_info("Core%d running @ %d MHz\r\n", __get_hart_id(), DEVICE_SYSCLK_FREQ/1000/1000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);
    log_info("%d Cores, %d IRQn\r\n", CIDU_GetCoreNum(), CIDU_GetIntNum());

    Timer_init();

    __enable_irq();

    oldTicks = cpu2Ticks;

    while (1) {
        if (cpu2Ticks != oldTicks) {
            oldTicks = cpu2Ticks;

            if ((oldTicks % 100) == 0) {
                log_debug("cpu1Ticks: %d cpu2Ticks: %d\r\n", cpu1Ticks, cpu2Ticks);
                Shared_show();

                GPIO_togglePin(GPIO_PIN_LED2);      //not thread safe
            }
        }
    }

    for(;;);

    return 0;
}

#ifdef __cplusplus
}
#endif
