/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <time.h>
#include <stdlib.h>

#include "device.h"
#include "driverlib.h"
#include "load_img.h"
#include "log.h"            //smp thread safe
#include "board_cfg.h"
#include "interrupt.h"

#define LED_PIN             LED1_PIN

#define GPIO_INPUT 1
#define GPIO_OUT   2

//0:edge trigger,1:level trigger
#define TEST_CASE 0

volatile uint32_t xint1_count=0;

volatile uint32_t xint6_count=0;

#if TEST_CASE==1

void xint1_fun(void);

void xint1_fun(void)
{
    xint1_count++;
    if (xint1_count<10000) {
    	printf("xint1_count=%d\r\n",xint1_count);
	}else {
	 	xint1_count=0;
	}
}
#else
void xint6_fun(void);

void xint6_fun(void)
{
    xint6_count++;
    if (xint6_count<10000) {
    	printf("xint6_count=%d\r\n",xint6_count);
	}else {
	 	xint6_count=0;
	}
}
#endif
/* Private functions ---------------------------------------------------------*/
/**
  * @brief test inputxbar trigger XINT，There are three ways to trigger a CPU interrupt: level trigger, 
  * rising edge trigger, and falling edge trigger by calling API ECLIC_SetTrigIRQ configuration
  * @param  None
  * @retval None
  * @note This case takes GPIO1 as input, connects GPIO1 to GPIO2, and GPIO2 
  * as output, initially pulls GPIO2 down, and the xbar is configured so that GPIO2 is 
  * pulled up, so that the edge trigger only enters the interrupt once, and the level 
  * trigger enters the interrupt all the time
  */
int main(void)
{
    uint32_t oldTicks;
    uint32_t cycles = __get_rv_cycle();

    __disable_irq();

    Device_init();

    GPIO_enableWritePin(LED_PIN);
    GPIO_enableWritePin(GPIO_OUT);

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_info("Hello DSP300 Template Project!\r\n");
    log_info("Core running @ %d MHz\r\n", DEVICE_SYSCLK_FREQ/1000/1000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);
    log_info("Reset record: %d, cycles %d\r\n", SysCtl_getDspCpu1RstRecord(), cycles);

#if TEST_CASE==1
    /* XINT1: level triggerred */
    XBAR_setInputPin(INPUTXBAR_BASE,XBAR_INPUT1 , GPIO_INPUT);
    /*
    * config triger way:
    * ECLIC_LEVEL_TRIGGER         0x0  
    * ECLIC_POSTIVE_EDGE_TRIGGER  0x1
    * ECLIC_NEGTIVE_EDGE_TRIGGER  0x3
    */
    ECLIC_SetTrigIRQ(INT_XINT6,ECLIC_LEVEL_TRIGGER);
    Interrupt_register(INT_XINT1,xint1_fun);
    Interrupt_enable(INT_XINT1);
#else
    /* XINT6: edge triggerred interrupt  */
    XBAR_setInputPin(INPUTXBAR_BASE,XBAR_INPUT6 , GPIO_INPUT);
    Interrupt_register(INT_XINT6,xint6_fun);
    /*
    * config triger way:
    * ECLIC_LEVEL_TRIGGER         0x0  
    * ECLIC_POSTIVE_EDGE_TRIGGER  0x1
    * ECLIC_NEGTIVE_EDGE_TRIGGER  0x3
    */
    ECLIC_SetTrigIRQ(INT_XINT6,ECLIC_POSTIVE_EDGE_TRIGGER);
    Interrupt_enable(INT_XINT6);
#endif
    __enable_irq();
    DELAY_US(10000);
    GPIO_writePin(GPIO_OUT, 1);


    for(;;);

    return 0;
}


