/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

//
// Included Files
//
#include "driverlib.h"
#include "device.h"
#include "stdio.h"
//
// Defines
//
#define TARGET_ADDRESS   0x4C

#define I2CA_GPIO_PIN_SDA 0U
#define I2CA_GPIO_PIN_SCL 1U

#define I2CA_GPIO_CFG_SDA	GPIO_0_I2CA_SDA
#define I2CA_GPIO_CFG_SCL	GPIO_1_I2CA_SCL

#define I2CB_GPIO_PIN_SDA 2U
#define I2CB_GPIO_PIN_SCL 3U

#define I2CB_GPIO_CFG_SDA	GPIO_2_I2CB_SDA
#define I2CB_GPIO_CFG_SCL	GPIO_3_I2CB_SCL

//
// Globals
//
uint16_t sData[2] = {0,0};                  // Send data buffer
uint16_t rData[2] = {0,0};                  // Receive data buffer
uint16_t rDataPoint = 0;                    // To keep track of where we are in the
static uint16_t index123;                                      // data stream to check received data

//
// Function Prototypes
//
void initI2CAFIFO();
void initI2CBFIFO();
void i2cAFIFOISR(void);
void i2cBFIFOISR(void);

//
// Main
//
void main(void)
{
    uint16_t i;
    index123 = 0;
    //
    // Initialize device clock and peripherals
    //
    Device_init();

	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_I2C);

	SysCtl_resetI2c();

    //
	// Interrupts that are used in this example are re-mapped to ISR functions
	// found within this file.
	//
	Interrupt_register(INT_I2CA_FIFO, &i2cAFIFOISR);
	Interrupt_register(INT_I2CB_FIFO, &i2cBFIFOISR);
	//
	// Set I2C use, initializing it for FIFO mode
	//
	initI2CAFIFO();
	initI2CBFIFO();
	//
	// Initialize the data buffers
	//
	for(i = 0; i < 2; i++)
	{
	   sData[i] = i;
	   rData[i]= 0;
	}

	//
	// Enable interrupts required for this example
	//
	Interrupt_enable(INT_I2CA_FIFO);
	Interrupt_enable(INT_I2CB_FIFO);

	//
	// Enable Global Interrupt (INTM) and realtime interrupt (DBGM)
	//
	EINT;
	ERTM;

	//
	// Loop forever. Suspend or place breakpoints to observe the buffers.
	//
	while(1)
	{
		// A FIFO interrupt will be generated for each Tx and Rx based
		// on the Interrupt levels configured.
		// The ISR will handle pushing/pulling data to/from the TX and
		// RX FIFOs resp.
		if(index123 <= 100){
//			I2C_sendStartCondition(I2CA_BASE);
			I2caRegs.I2CMDR.bit.STT = 1;
		}

		DEVICE_DELAY_US(50);

	}

}




//
// Function to configure I2C A in FIFO mode.
//
void initI2CAFIFO()
{
    GPIO_setPinConfig(I2CA_GPIO_CFG_SDA);
    GPIO_setPadConfig(I2CA_GPIO_PIN_SDA, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CA_GPIO_PIN_SDA, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CA_GPIO_PIN_SDA,GPIO_DRV_20MA);

    GPIO_setPinConfig(I2CA_GPIO_CFG_SCL);
    GPIO_setPadConfig(I2CA_GPIO_PIN_SCL, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CA_GPIO_PIN_SCL, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CA_GPIO_PIN_SCL,GPIO_DRV_20MA);
    //
    // Must put I2C into reset before configuring it
    //
//    I2C_disableModule(I2CA_BASE);
    I2caRegs.I2CMDR.bit.IRS = 0;

    //
    // I2C configuration. Use a 100kHz I2CCLK with a 50% duty cycle.
    //
//    I2C_initController(I2CA_BASE, DEVICE_APBCLK_FREQ, 100000, I2C_DUTYCYCLE_50);
    I2caRegs.I2CPSC = 5;
    I2caRegs.I2CCLKH = 45;
    I2caRegs.I2CCLKL = 45;

//    I2C_setConfig(I2CA_BASE, I2C_CONTROLLER_RECEIVE_MODE);
    I2caRegs.I2CMDR.bit.MST = 1;
    I2caRegs.I2CMDR.bit.TRX = 0;

//    I2C_setAddressMode(I2CA_BASE, I2C_ADDR_MODE_7BITS);
    I2caRegs.I2CMDR.bit.XA = 0;

//    I2C_setDataCount(I2CA_BASE, 2);
    I2caRegs.I2CCNT.all = 2;

//    I2C_setBitCount(I2CA_BASE, I2C_BITCOUNT_8);
    I2caRegs.I2CMDR.bit.BC = 0;

    // Configure for external loopback
    //
//    I2C_setTargetAddress(I2CA_BASE, TARGET_ADDRESS);
    I2caRegs.I2CSAR.bit.SAR = TARGET_ADDRESS;

//    I2C_setEmulationMode(I2CA_BASE, I2C_EMULATION_FREE_RUN);
    I2caRegs.I2CMDR.bit.FREE = 1;

    //
    // FIFO and interrupt configuration
    //
//    I2C_enableFIFO(I2CA_BASE);
    I2caRegs.I2CFFTX.bit.I2CFFEN = 1;
    I2caRegs.I2CFFTX.bit.TXFFRST = 1;
    I2caRegs.I2CFFRX.bit.RXFFRST = 1;

//    I2C_clearInterruptStatus(I2CA_BASE, I2C_INT_RXFF);
    I2caRegs.I2CFFRX.bit.RXFFINTCLR = 1;

    //
    // Transmit FIFO interrupt levels are set to generate an interrupt
    // when the 16 byte TX fifo contains 2 or lesser bytes of data.
    //
//    I2C_setFIFOInterruptLevel(I2CA_BASE, I2C_FIFO_TX2, I2C_FIFO_RX2);
    I2caRegs.I2CFFTX.bit.TXFFIL = 2;
    I2caRegs.I2CFFRX.bit.RXFFIL = 2;

//    I2C_enableInterrupt(I2CA_BASE, I2C_INT_RXFF | I2C_INT_STOP_CONDITION);
    I2caRegs.I2CIER.bit.SCD = 1;
    I2caRegs.I2CFFRX.bit.RXFFIENA = 1;

//    I2C_enableModule(I2CA_BASE);
    I2caRegs.I2CMDR.bit.IRS = 1;
}





//
// Function to configure I2C A in FIFO mode.
//
void initI2CBFIFO()
{
    GPIO_setPinConfig(I2CB_GPIO_CFG_SDA);
    GPIO_setPadConfig(I2CB_GPIO_PIN_SDA, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CB_GPIO_PIN_SDA, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CB_GPIO_PIN_SDA,GPIO_DRV_20MA);

    GPIO_setPinConfig(I2CB_GPIO_CFG_SCL);
    GPIO_setPadConfig(I2CB_GPIO_PIN_SCL, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CB_GPIO_PIN_SCL, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CB_GPIO_PIN_SCL,GPIO_DRV_20MA);
    //
    // Must put I2C into reset before configuring it
    //
//    I2C_disableModule(I2CB_BASE);
    I2cbRegs.I2CMDR.bit.IRS = 0;

    //
    // I2C configuration. Use a 100kHz I2CCLK with a 50% duty cycle.
    //
//    I2C_initController(I2CB_BASE, DEVICE_APBCLK_FREQ, 100000, I2C_DUTYCYCLE_50);
    I2cbRegs.I2CPSC = 5;
    I2cbRegs.I2CCLKH = 45;
    I2cbRegs.I2CCLKL = 45;

//    I2C_setConfig(I2CB_BASE, I2C_TARGET_SEND_MODE);
    I2cbRegs.I2CMDR.bit.MST = 0;
    I2cbRegs.I2CMDR.bit.TRX = 1;

//    I2C_setAddressMode(I2CB_BASE, I2C_ADDR_MODE_7BITS);
    I2cbRegs.I2CMDR.bit.XA = 0;

//    I2C_setDataCount(I2CB_BASE, 2);
    I2cbRegs.I2CCNT.all = 2;

//    I2C_setBitCount(I2CB_BASE, I2C_BITCOUNT_8);
    I2cbRegs.I2CMDR.bit.BC = 0;
    //
    // Configure for external loopback
    //
//    I2C_setOwnAddress(I2CB_BASE, TARGET_ADDRESS);
    I2cbRegs.I2COAR.bit.OAR = TARGET_ADDRESS;

//    I2C_setEmulationMode(I2CB_BASE, I2C_EMULATION_FREE_RUN);
    I2cbRegs.I2CMDR.bit.FREE = 1;

    //
    // FIFO and interrupt configuration
    //
//    I2C_enableFIFO(I2CB_BASE);
    I2cbRegs.I2CFFTX.bit.I2CFFEN = 1;
    I2cbRegs.I2CFFTX.bit.TXFFRST = 1;
    I2cbRegs.I2CFFRX.bit.RXFFRST = 1;

//    I2C_clearInterruptStatus(I2CB_BASE, I2C_INT_TXFF|I2C_INT_RXFF);
    I2cbRegs.I2CFFTX.bit.TXFFINTCLR = 1;
    I2cbRegs.I2CFFRX.bit.RXFFINTCLR = 1;

    //
    // Transmit FIFO interrupt levels are set to generate an interrupt
    // when the 16 byte TX fifo contains 2 or lesser bytes of data.
    //
//    I2C_setFIFOInterruptLevel(I2CB_BASE, I2C_FIFO_TX2, I2C_FIFO_RX2);
    I2cbRegs.I2CFFTX.bit.TXFFIL = 2;
    I2cbRegs.I2CFFRX.bit.RXFFIL = 2;

//    I2C_enableInterrupt(I2CB_BASE, I2C_INT_TXFF | I2C_INT_STOP_CONDITION);
    I2cbRegs.I2CIER.bit.SCD = 1;
    I2cbRegs.I2CFFTX.bit.TXFFIENA = 1;

//    I2C_enableModule(I2CB_BASE);
    I2cbRegs.I2CMDR.bit.IRS = 1;

}





//
// I2C TX and Receive FIFO ISR
//
 void i2cAFIFOISR(void)
{
    uint16_t i;
    //
    // If receive FIFO interrupt flag is set, read data
    //
//    if((I2C_getInterruptStatus(I2CA_BASE) & I2C_INT_RXFF) != 0)
    if(I2caRegs.I2CFFRX.bit.RXFFINT != 0)
    {
        for(i = 0; i < 2; i++)
        {
//            rData[i] = I2C_getData(I2CA_BASE);
        	rData[i] = I2caRegs.I2CDRR;
        }

        //
        // Check received data
        //
        for(i = 0; i < 2; i++)
        {

		   if(rData[i] != ((rDataPoint + i) & 0xFF))
		   {
			   while(1);
			   //
			   // Something went wrong. rData doesn't contain expected data.
			   //
			   //ESTOP0;
		   }
	   }

	   //
	   // Used to keep track of the last position in the receive
	   // stream for error checking
	   //
	   rDataPoint = (rDataPoint + 1) & 0xFF;

        //
        // Clear interrupt flag
        //
//        I2C_clearInterruptStatus(I2CA_BASE, I2C_INT_RXFF);
	   I2caRegs.I2CFFRX.bit.RXFFINTCLR = 1;
    }

}

 //
 // I2C TX and Receive FIFO ISR
 //
  void i2cBFIFOISR(void)
 {
     uint16_t i;

     //
     // If transmit FIFO interrupt flag is set, put data in the buffer
     //
//     if((I2C_getInterruptStatus(I2CB_BASE) & I2C_INT_TXFF) != 0)
     if(I2cbRegs.I2CFFTX.bit.TXFFINT != 0)
     {
         for(i = 0; i < 2; i++)
         {
//             I2C_putData(I2CB_BASE, sData[i]);
        	 I2cbRegs.I2CDXR = sData[i];
         }

         //
         // Increment data for next cycle
         //
         for(i = 0; i < 2; i++)
         {
            sData[i] = (sData[i] + 1) & 0xFF;
         }

         //
         // Clear interrupt flag
         //
//         I2C_clearInterruptStatus(I2CB_BASE, I2C_INT_TXFF);
         I2cbRegs.I2CFFTX.bit.TXFFINTCLR = 1;

         //
         // Turn Off an LED to depict data transfer
         //
        // GPIO_writePin(DEVICE_GPIO_PIN_LED1, 1);
		if(index123 > 100){
//			I2C_sendStopCondition(I2CA_BASE);
			I2caRegs.I2CMDR.bit.STP = 1;
		}
		index123++;

     }

 }

 //
 // End of File
 //

