/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * @file    gs32_pwm_clu.h
 * @brief
 *
 * @Commit History:
 * 2024.12.19, ralf, provide the PWM clu support APIs.
 *
 */

#ifndef GS32_PWM_CLU_H_
#define GS32_PWM_CLU_H_

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "inc/hw_types.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define OVERMOD_EN 			0
#define PHASE_INVER_EN 		0
#define NARROR_PWM_EN 		0

#define NARROR_PWM_VALUE 	0.03f
#define SQRT3 				1.7320508f
#define ONE_PI 				3.1415926f

#define OVERMOD_FACT_A 		0.907f
#define OVERMOD_FACT_B 		0.9535f
#define OVERMOD_FACT_C 		10.5f
#define OVERMOD_FACT_D 		1.05f
#define KI_FACT 			0.6f

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef struct {
    float32_t Ualpha;   //Input: reference alpha-axis phase voltage
    float32_t Ubeta;    //Input: reference beta-axis phase voltage
    float32_t Ta;       //Output: reference phase-a switching function
    float32_t Tb;       //Output: reference phase-b switching function
    float32_t Tc;       //Output: reference phase-c switching function
    uint16 Sector;     	//output: sector detect
    uint16 PRD_pwm;     //Input: PWM period, pwm must work in up&down mode
    float32_t Udc;      //Input: Bus Voltage
    float32_t k;        //output: k Parameter
    uint16 CMPa;        //Output: PWMx cmp value for phase A
    uint16 CMPb;        //Output: PWMx cmp value for phase B
    uint16 CMPc;        //Output: PWMx cmp value for phase C
    float32_t X;		//Output:
    float32_t Y;		//Output:
    float32_t Z;		//Output:
    float32_t t1;		//Output:
    float32_t t2;		//Output:
} PWM_VECT_3P2L_Obj;

typedef struct {
    float32_t Ualpha;   //Input: reference alpha-axis phase voltage
    float32_t Ubeta;    //Input: reference beta-axis phase voltage
    float32_t Ta;       //Output: reference phase-a switching function
    float32_t Tb;       //Output: reference phase-b switching function
    float32_t Tc;       //Output: reference phase-c switching function
    float32_t Udc;      //Input: Bus Voltage
    float32_t Speed;    //Input: Motor Speed, revolution per minute
    uint16 PRD_pwm;     //Input: Period Counter of EPWM, EPWM in up&down mode
    uint16 CMPa;        //Output: PWMx CMPa value for phase A
    uint16 CMPb;        //Output: PWMx CMPa value for phase B
    uint16 CMPc;        //Output: PWMx CMPa value for phase C
    float32_t Uzero;	//Output: Harmonic Voltage
} PWM_ZERO_Obj;

typedef enum {
    SVPWM = 0,
    VECT_DPWM0,
	VECT_DPWM1,
	VECT_DPWM2,
	VECT_DPWM3,
	VECT_DPWMMAX,
	VECT_DPWMMIN
} PWM_VECT_3P2L_MODE;

typedef enum {
    SPWM = 0,
	ZERO_DPWM0,
	ZERO_DPWM1,
	ZERO_DPWM2,
	ZERO_DPWM3,
	ZERO_DPWMMAX,
	ZERO_DPWMMIN,
    SYPWM
} PWM_ZERO_MODE;

typedef enum {
    LINEAR,
    OVER_MOD1,
    OVER_MOD2
} OVER_MOD_STATE;

typedef enum {
    LINEAR_COMP = 0,
    LINEAR_NOCOMP
} OVERMOD_TYPE;

typedef enum {
    OFF = 0,
    P_CLOOP,
    PI_CLOOP
} NVOLT_CONTROL_TYPE;

typedef struct {
    float32_t Ualpha;   //Input: reference alpha-axis phase voltage
    float32_t Ubeta;    //Input: reference beta-axis phase voltage
    float32_t Ta;       //Output: phase-a switching on time
    float32_t Tb;       //Output: phase-b switching on time
    float32_t Tc;       //Output: phase-c switching on time
    uint16 Sector_N;    //output: 6*big sector
    uint16 Sector_n;    //output: 4*small sector for each Sector_N
    float32_t Udc1;     //input: Bus Voltage plus
    float32_t Udc2;     //input: Bus voltage minus
    float32_t Udc;      //output: total bus voltage, Udc1+ Udc2
    float32_t Udc_ref;  //input: bus voltage reference for over-voltage protection
    float32_t coef;     //input: neutral voltage balance control co-efficiency, (0, 1]
    uint16 PRD_pwm;     //input: period counter of EPWM, EPWM in up&down mode
    uint16 CMPa1;		//Output: EPWMx CMPa1
    uint16 CMPa2;		//Output: EPWMx CMPa2
    uint16 CMPb1;		//Output: EPWMx CMPb1
    uint16 CMPb2;		//Output: EPWMx CMPb2
    uint16 CMPc1;		//Output: EPWMx CMPc1
    uint16 CMPc2;		//Output: EPWMx CMPc2
    float32_t t1;		//Output:
    float32_t t2;		//Output:
    float32_t X2;		//Output:
    float32_t Y2;		//Output:
    float32_t Z2;		//Output:
    float32_t X1;		//Output:
    float32_t X11;		//Output:
    float32_t Y1;		//Output:
    float32_t Y11;		//Output:
    float32_t Z1;		//Output:
    float32_t Z11;		//Output:
} SVPWM_3P3L_Obj;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/**
 * \brief   PWM Obj reset to 3p2l space vector pwm control
 */
static inline void
PWM_vect_3P2L_reset(PWM_VECT_3P2L_Obj *v) {
    v->Ta = 0;
    v->Tb = 0;
    v->Tc = 0;
    v->Sector = 0;
    v->Ualpha = 0;
    v->Ubeta = 0;
    v->Udc = 0;
    v->k = 0;
    v->CMPa = 0;
    v->CMPb = 0;
    v->CMPc = 0;
}

/**
 * \brief   PWM Obj reset to zero pwm control
 */
static inline void
PWM_zero_reset(PWM_ZERO_Obj *v) {
    v->Ta = 0;
    v->Tb = 0;
    v->Tc = 0;
    v->Ualpha = 0;
    v->Ubeta = 0;
    v->Speed = 0;
    v->Udc = 0;
    v->CMPa = 0;
    v->CMPb = 0;
    v->CMPc = 0;
    v->Uzero = 0;
}

/**
 * \brief   PWM Obj reset to 3p3l space vector pwm control
 */
static inline void
SVPWM_3P3L_reset(SVPWM_3P3L_Obj *v) {
    v->Ta = 0;
    v->Tb = 0;
    v->Tc = 0;
    v->Sector_N = 0;
    v->Sector_n = 0;
    v->Ualpha = 0;
    v->Ubeta = 0;
    v->Udc = 0;
    v->coef = 0;
}

/**
 * brief   select the 3p2l pwm vector mode
 */
void PWM_vect_3P2L_config(PWM_VECT_3P2L_MODE mode);

/**
 * brief   select the 3p2l pwm zero mode
 */
void PWM_zero_config(PWM_ZERO_MODE mode);

/**
 * \brief   3P2L PWM Vector Generator Function Definition
 */
void PWM_vect_3P2L_run(PWM_VECT_3P2L_Obj *v);

/**
 * \brief   PWM ZERO Generator Function Definition
 */
void PWM_zero_run(PWM_ZERO_Obj *v);

/**
 * \brief   3P3L PWM Vector Generator Function Definition
 */
void SVPWM_3P3L_run(SVPWM_3P3L_Obj *v, NVOLT_CONTROL_TYPE type);

#ifdef __cplusplus
}
#endif

#endif /* GS32_PWM_CLU_H_*/
