/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file     park.h
 *   @brief
 *
 */

#ifndef _PARK_H
#define _PARK_H

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#if __riscv && __ENABLE_NICE
#include "gs32_math.h"
#else
#include "math.h"
#endif

#include "IQmathLib.h"
#include "gs32_math.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/*
 * @brief		Default initalizer for the PARK object.
 */
#define PARK_DEFAULTS \
    {                 \
        0,            \
            0,        \
            0,        \
            0,        \
            0,        \
    }

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/*
 * @brief		Park Transformation Parameters
 */
typedef struct
{
    _iq ds;  // Input: stationary d-axis stator variable */
    _iq qs;  // Input: stationary q-axis stator variable */
    _iq ang; // Input: rotating angle (pu) */
    _iq de;  // Output: rotating d-axis stator variable */
    _iq qe;  // Output: rotating q-axis stator variable */
} PARK;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/*
 * @brief		Park transform calculation
 */
static inline void PARK_calc(register PARK *v) {

    register _iq cos_ang, sin_ang;

	#if __ENABLE_NICE //DSP instruction
    sin_ang = __sinpuf32(v->ang);
    cos_ang = __cospuf32(v->ang);

    __mv2_f2mpr(0, 0, 0, 1, v->ds, v->qs);
    v->de = __vmulsum2_f2_r1(cos_ang, sin_ang);
	v->qe = __vmulsum2_f2_r1_neg1(sin_ang, cos_ang);
	#else
    sin_ang = sinf(v->ang * TWO_PI);//sinf(v->ang * TWO_PI);
    cos_ang = cosf(v->ang * TWO_PI);//cosf(v->ang * TWO_PI);

    v->de = _IQmpy(v->ds, cos_ang) + _IQmpy(v->qs, sin_ang);
    v->qe = _IQmpy(v->qs, cos_ang) - _IQmpy(v->ds, sin_ang);
	#endif
}

#ifdef __cplusplus
}
#endif

#endif /* _PARK_H */
