/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <string.h>

#include "fdp_err.h"
#include "fdp_storage_port.h"
#include "fdp_cache.h"
#include "fdp_defconfig.h"

/**
 * @brief User define include header file.
 * 
 */
#include "gs32_flashAPI_lib.h"
#include "flash_programming_gs32.h"
#include "inc/hw_flash.h"

#include "device.h"

/**
 * @brief User define macro.
 *
 */

/* Flash sector size. */
#define FLASH_SECTOR_SIZE	FlashMainPageSize

int fdp_storage_dev_init(uint32_t start_addr, uint32_t size)
{
	uint32 oReturnCheck = Fapi_Status_Success;
	uint32 u32EfcCount = Fapi_GetEfcCount();
	uint32 u32FlashBank = 0;

	Flash_initModule(0, 0, 0);

	oReturnCheck = Fapi_initializeAPI( (Fapi_FmcRegistersType *)0, 0U);

	if(oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;

	for (u32FlashBank=0; u32FlashBank <u32EfcCount; u32FlashBank++) {
		oReturnCheck = Fapi_setActiveFlashBank(u32FlashBank);

		if(oReturnCheck != Fapi_Status_Success)
			return -FDP_CHECKERR;

		while (Fapi_checkFsmForReady(u32FlashBank) != Fapi_Status_FsmReady);
	}

	return FDP_SUCCESS;
}

#if (FPD_VERSION_NUMBER == 0x12)

int fdp_storage_dev_erase(uint32_t start_addr, uint32_t size)
{
	Fapi_StatusType oReturnCheck = Fapi_Status_Success;
	Fapi_FlashStatusType oFlashStatus = Fapi_Status_Success;
	Fapi_FlashStatusWordType oFlashStatusWord = {0};
	Fapi_FlashBankType start_bank;
	uint32_t bank0_addr = 0;
	uint32_t bank1_addr = 0;
	uint32_t bank0_size = 0;
	uint32_t bank1_size = 0;
	uint32_t sector_num;
	uint32_t sector_addr;
	uint32_t current_addr;

	/* Calculate which block the address belongs to. */
	start_bank = (start_addr >= BANK1_BASE_ADDR) ? Fapi_FlashBank1 : Fapi_FlashBank0;

	if (start_addr + size >= BANK1_BASE_ADDR && start_bank == Fapi_FlashBank0) {
		bank0_addr = start_addr;
		bank0_size = BANK1_BASE_ADDR - start_addr;
		bank1_addr = BANK1_BASE_ADDR;
		bank1_size = size - bank0_size;
	} else if (start_addr + size >= BANK1_BASE_ADDR && start_bank == Fapi_FlashBank1) {
		bank1_addr = start_addr;
		bank1_size = size;
		goto erase_bank1;
	} else {
		bank0_addr = start_addr;
		bank0_size = size;
	}

	/* Erase the Flash Bank 0. */
erase_bank0 :
	/* Calculates the sector in which the address is located. */
	sector_num = bank0_size / FLASH_SECTOR_SIZE;

	if (bank0_size % FLASH_SECTOR_SIZE)
		sector_num += 1;

	sector_addr = (bank0_addr / FLASH_SECTOR_SIZE) * FLASH_SECTOR_SIZE;

	/* Issue ClearMore command */
	oReturnCheck = Fapi_issueAsyncCommand(Fapi_FlashBank0, Fapi_ClearMore);

	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;

	/*
	* Erase the sector that is programmed in the above example
	* Erase Sector0 flash main block.
	*/
	for (current_addr = sector_addr; current_addr < bank0_addr + bank0_size;
		current_addr += FLASH_SECTOR_SIZE) {

		oReturnCheck = Fapi_issueAsyncCommandWithAddress(Fapi_FlashBank0,
						Fapi_EraseSector, (uint32 *)(uintptr_t)current_addr);

		/* Check Flash API documentation for possible errors */
		if (oReturnCheck != Fapi_Status_Success)
			return -FDP_CHECKERR;

		/* Wait until Flash Control is enable done with other operation */
		while (Fapi_checkFsmForReady(Fapi_FlashBank0) != Fapi_Status_FsmReady);

		/*
		* Read flash control status register contents to know the status of flash control after
		* erase command to see if there are any erase operation related errors
		*/
		oFlashStatus = Fapi_getFsmStatus(Fapi_FlashBank0);
		if ((FLASH_CONTROL_ERASE_OKAY != oFlashStatus) && (0x64 != oFlashStatus)) {
			oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
			return -FDP_CHECKERR;
		}
	}

	/*
	* Do blank check
	* Verify that flash block
	* The Erase command itself does a verify as it goes.
	* Hence erase verify by CPU reads (Fapi_doBlankCheck()) is optional.
	*/
#if (defined(FDP_IS_CHECK_STORAGE_ERASE) && FDP_IS_CHECK_STORAGE_ERASE)
	fdp_dcache_invaild(sector_addr, bank0_size);
	oReturnCheck = Fapi_doBlankCheck(Fapi_FlashBank0, (uint32 *)(uintptr_t)sector_addr,
									bank0_size/sizeof(uint32), &oFlashStatusWord);

	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;
#endif

	if (bank1_size == 0)
		return FDP_SUCCESS;

erase_bank1:
	/* Calculates the sector in which the address is located. */
	sector_num = bank1_size / FLASH_SECTOR_SIZE;

	if (bank1_size % FLASH_SECTOR_SIZE)
		sector_num += 1;

	sector_addr = (bank1_addr / FLASH_SECTOR_SIZE) * FLASH_SECTOR_SIZE;

	/* Issue ClearMore command */
	oReturnCheck = Fapi_issueAsyncCommand(Fapi_FlashBank1, Fapi_ClearMore);

	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;

	/*
	* Erase the sector that is programmed in the above example
	* Erase Sector0 flash main block.
	*/
	for (current_addr = sector_addr; current_addr < bank1_addr + bank1_size;
		current_addr += FLASH_SECTOR_SIZE) {

		oReturnCheck = Fapi_issueAsyncCommandWithAddress(Fapi_FlashBank1,
						Fapi_EraseSector, (uint32 *)(uintptr_t)current_addr);

		/* Check Flash API documentation for possible errors */
		if (oReturnCheck != Fapi_Status_Success)
			return -FDP_CHECKERR;

		/* Wait until Flash Control is enable done with other operation */
		while (Fapi_checkFsmForReady(Fapi_FlashBank1) != Fapi_Status_FsmReady);

		/*
		* Read flash control status register contents to know the status of flash
		* control after erase command to see if there are any erase operation
		* related errors.
		*/
		oFlashStatus = Fapi_getFsmStatus(Fapi_FlashBank1);
		if ((FLASH_CONTROL_ERASE_OKAY != oFlashStatus) && (0x64 != oFlashStatus)) {
			oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
			return -FDP_CHECKERR;
		}
	}

	/*
	* Do blank check
	* Verify that flash block
	* The Erase command itself does a verify as it goes.
	* Hence erase verify by CPU reads (Fapi_doBlankCheck()) is optional.
	*/
#if (defined(FDP_IS_CHECK_STORAGE_ERASE) && FDP_IS_CHECK_STORAGE_ERASE)
	fdp_dcache_invaild(sector_addr, bank1_size);
	oReturnCheck = Fapi_doBlankCheck(Fapi_FlashBank1, (uint32 *)(uintptr_t)sector_addr,
									bank1_size/sizeof(uint32), &oFlashStatusWord);

	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;
#endif

	return FDP_SUCCESS;
}

int fdp_storage_dev_write(uint8_t *buffer, uint32_t start_addr, uint32_t len)
{
	volatile uint32_t u32Index = 0;
	volatile uint32_t i = 0;
	Fapi_StatusType oReturnCheck = Fapi_Status_Success;
	Fapi_FlashStatusType oFlashStatus;
	Fapi_FlashStatusWordType oFlashStatusWord;
	Fapi_FlashBankType start_bank;

	uint16_t BufferSizeInWords = 4;

	uint32_t bank0_addr = 0;
	uint32_t bank1_addr = 0;
	uint32_t bank0_size = 0;
	uint32_t bank1_size = 0;

	if (buffer == NULL)
		return -FDP_ERRINPARAM;

	/* Calculate which block the address belongs to. */
	start_bank = (start_addr >= BANK1_BASE_ADDR) ? Fapi_FlashBank1 : Fapi_FlashBank0;

	if (start_addr + len >= BANK1_BASE_ADDR && start_bank == Fapi_FlashBank0) {
		bank0_addr = start_addr;
		bank0_size = BANK1_BASE_ADDR - start_addr;
		bank1_addr = BANK1_BASE_ADDR;
		bank1_size = len - bank0_size;
	} else if (start_addr + len >= BANK1_BASE_ADDR && start_bank == Fapi_FlashBank1) {
		bank1_addr = start_addr;
		bank1_size = len;
		goto write_bank1;
	} else {
		bank0_addr = start_addr;
		bank0_size = len;
	}

write_bank0:
	for (i=0, u32Index = bank0_addr;
		u32Index < (bank0_addr + bank0_size);
		i += BufferSizeInWords, u32Index += (BufferSizeInWords * sizeof(uint32_t))) {

		oReturnCheck = Fapi_issueProgrammingCommand(Fapi_FlashBank0,
													(uint32 *)(uintptr_t)u32Index,
													(uint16_t *)((uint32_t *)buffer + i),
													BufferSizeInWords * sizeof(uint16_t),
													0, 0, Fapi_AutoEccGeneration);

		if (oReturnCheck != Fapi_Status_Success)
			return -FDP_CHECKERR;

		/* Wait until the Flash program operation is over */
		while (Fapi_checkFsmForReady(Fapi_FlashBank0) == Fapi_Status_FsmBusy);

		/*
		* Read flash control register contents to know the status of flash control after
		* program command to see if there are any program operation related
		* errors
		*/
		oFlashStatus = Fapi_getFsmStatus(Fapi_FlashBank0);
		if ((oFlashStatus != FLASH_CONTROL_ERASE_OKAY) &&
			(oFlashStatus != FLASH_CONTROL_PROGRAM_OKAY) &&
			(oFlashStatus != FLASH_CONTROL_IDLE)) {

			return -FDP_CHECKERR;
		}
	}

	/* Wait until the Flash program operation is over */
	while (Fapi_checkFsmForReady(Fapi_FlashBank0) == Fapi_Status_FsmBusy);

	/*
	* Read flash control register contents to know the status of flash control after
	* program command to see if there are any program operation related
	* errors
	*/
	oFlashStatus = Fapi_getFsmStatus(Fapi_FlashBank0);
	if ((oFlashStatus != FLASH_CONTROL_ERASE_OKAY) &&
		(oFlashStatus != FLASH_CONTROL_PROGRAM_OKAY) &&
		(oFlashStatus != FLASH_CONTROL_IDLE)) {

		return -FDP_CHECKERR;
	}

	/* Verify the programmed values. */
#if (defined(FDP_IS_CHECK_STORAGE_WRITE) && FDP_IS_CHECK_STORAGE_WRITE)
	fdp_dcache_invaild(bank0_addr, bank0_size);
	oReturnCheck = Fapi_doVerify((uint32 *)(uintptr_t)bank0_addr,
								bank0_size / 4,
								(uint32 *)buffer,
								&oFlashStatusWord);
	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;
#endif

	if (bank1_size == 0)
		return FDP_SUCCESS;

write_bank1:
	for (i=0, u32Index = bank1_addr;
		u32Index < (bank1_addr + bank1_size);
		i += BufferSizeInWords, u32Index += (BufferSizeInWords * sizeof(uint32_t))) {

		oReturnCheck = Fapi_issueProgrammingCommand(Fapi_FlashBank1,
													(uint32 *)(uintptr_t)u32Index,
													(uint16_t *)((uint32_t *)buffer +
													(i + bank0_size/BufferSizeInWords)),
													BufferSizeInWords * sizeof(uint16_t),
													0, 0, Fapi_AutoEccGeneration);

		if (oReturnCheck != Fapi_Status_Success)
			return -FDP_CHECKERR;

		/* Wait until the Flash program operation is over */
		while (Fapi_checkFsmForReady(Fapi_FlashBank1) == Fapi_Status_FsmBusy);

		/*
		* Read flash control register contents to know the status of flash control after
		* program command to see if there are any program operation related
		* errors
		*/
		oFlashStatus = Fapi_getFsmStatus(Fapi_FlashBank1);
		if ((oFlashStatus != FLASH_CONTROL_ERASE_OKAY) &&
			(oFlashStatus != FLASH_CONTROL_PROGRAM_OKAY) &&
			(oFlashStatus != FLASH_CONTROL_IDLE)) {
			oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
			return -FDP_CHECKERR;
		}
	}

	/* Wait until the Flash program operation is over */
	while (Fapi_checkFsmForReady(Fapi_FlashBank1) == Fapi_Status_FsmBusy);

	/*
	* Read flash control register contents to know the status of flash control after
	* program command to see if there are any program operation related
	* errors
	*/
	oFlashStatus = Fapi_getFsmStatus(Fapi_FlashBank1);
	if ((oFlashStatus != FLASH_CONTROL_ERASE_OKAY) &&
		(oFlashStatus != FLASH_CONTROL_PROGRAM_OKAY) &&
		(oFlashStatus != FLASH_CONTROL_IDLE)) {

		oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
		return -FDP_CHECKERR;
	}

	/* Verify the programmed values. */
#if (defined(FDP_IS_CHECK_STORAGE_WRITE) && FDP_IS_CHECK_STORAGE_WRITE)
	fdp_dcache_invaild(bank1_addr, bank1_size);
	oReturnCheck = Fapi_doVerify((uint32 *)(uintptr_t)bank1_addr,
								bank1_size / 4,
								(uint32 *)buffer + (bank0_size/BufferSizeInWords),
								&oFlashStatusWord);

	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;
#endif

	return FDP_SUCCESS;
}

#elif (FPD_VERSION_NUMBER == 0x22)

int fdp_storage_dev_erase(uint32_t start_addr, uint32_t size)
{
	Fapi_StatusType oReturnCheck = Fapi_Status_Success;
	Fapi_FlashStatusType oFlashStatus = Fapi_Status_Success;
	Fapi_FlashStatusWordType oFlashStatusWord = {0};
	uint32_t bank0_addr = 0;
	uint32_t bank0_size = 0;
	uint32_t sector_num;
	uint32_t sector_addr;
	uint32_t current_addr;

	/* Calculate which block the address belongs to. */

	bank0_addr = start_addr;
	bank0_size = size;

	/* Erase the Flash Bank 0. */
	/* Calculates the sector in which the address is located. */
	sector_num = bank0_size / FLASH_SECTOR_SIZE;

	if (bank0_size % FLASH_SECTOR_SIZE)
		sector_num += 1;

	sector_addr = (bank0_addr / FLASH_SECTOR_SIZE) * FLASH_SECTOR_SIZE;

	/* Issue ClearMore command */
	oReturnCheck = Fapi_issueAsyncCommand(Fapi_FlashBank0, Fapi_ClearMore);

	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;

	/*
	* Erase the sector that is programmed in the above example
	* Erase Sector0 flash main block.
	*/
	for (current_addr = sector_addr; current_addr < bank0_addr + bank0_size;
		current_addr += FLASH_SECTOR_SIZE) {

		oReturnCheck = Fapi_issueAsyncCommandWithAddress(Fapi_FlashBank0, Fapi_EraseSector,
													(uint32 *)(uintptr_t)current_addr);
		/* Wait until Flash Control is enable done with other operation */
		while (Fapi_checkFsmForReady(Fapi_FlashBank0) != Fapi_Status_FsmReady);

		/* Check Flash API documentation for possible errors */
		if (oReturnCheck != Fapi_Status_Success)
			return -FDP_CHECKERR;

		/*
		* Read flash control status register contents to know the status of flash control after
		* erase command to see if there are any erase operation related errors
		*/
		oFlashStatus = Fapi_getFsmStatus(Fapi_FlashBank0);
		if ((FLASH_CONTROL_ERASE_OKAY != oFlashStatus) && (0x64 != oFlashStatus)) {
			oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
			return -FDP_CHECKERR;
		}
	}

	/*
	* Do blank check
	* Verify that flash block
	* The Erase command itself does a verify as it goes.
	* Hence erase verify by CPU reads (Fapi_doBlankCheck()) is optional.
	*/
#if (defined(FDP_IS_CHECK_STORAGE_ERASE) && FDP_IS_CHECK_STORAGE_ERASE)
	fdp_dcache_invaild(sector_addr, bank0_size);
	oReturnCheck = Fapi_doBlankCheck(Fapi_FlashBank0, (uint32 *)(uintptr_t)sector_addr,
									bank0_size/sizeof(uint32), &oFlashStatusWord);

	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;
#endif

	return FDP_SUCCESS;
}

int fdp_storage_dev_write(uint8_t *buffer, uint32_t start_addr, uint32_t len)
{
	uint32_t u32Index = 0;
	uint32_t i = 0;
	Fapi_StatusType oReturnCheck = Fapi_Status_Success;
	Fapi_FlashStatusType oFlashStatus;
	Fapi_FlashStatusWordType oFlashStatusWord;
	uint16_t BufferSizeInWords = 4;
	uint32_t bank0_addr = 0;
	uint32_t bank0_size = 0;

	if (buffer == NULL)
		return -FDP_ERRINPARAM;

	/* Calculate which block the address belongs to. */
	bank0_addr = start_addr;
	bank0_size = len;

	for (i=0, u32Index = bank0_addr;
		u32Index < (bank0_addr + bank0_size);
		i += BufferSizeInWords, u32Index += (BufferSizeInWords * sizeof(uint32_t))) {

		oReturnCheck = Fapi_issueProgrammingCommand(Fapi_FlashBank0,
													(uint32 *)(uintptr_t)u32Index,
													(uint16_t *)((uint32_t *)buffer + i),
													BufferSizeInWords * sizeof(uint16_t),
													0, 0, Fapi_AutoEccGeneration);

		/* Wait until the Flash program operation is over */
		while (Fapi_checkFsmForReady(Fapi_FlashBank0) == Fapi_Status_FsmBusy);

		if (oReturnCheck != Fapi_Status_Success)
			return -FDP_CHECKERR;

		/*
		* Read flash control register contents to know the status of flash control after
		* program command to see if there are any program operation related
		* errors
		*/
		oFlashStatus = Fapi_getFsmStatus(Fapi_FlashBank0);
		if ((oFlashStatus != FLASH_CONTROL_ERASE_OKAY) &&
			(oFlashStatus != FLASH_CONTROL_PROGRAM_OKAY) &&
			(oFlashStatus != FLASH_CONTROL_IDLE)) {

			oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
			return -FDP_CHECKERR;
		}
	}

	/* Wait until the Flash program operation is over */
	while (Fapi_checkFsmForReady(Fapi_FlashBank0) == Fapi_Status_FsmBusy);

	/* Check Flash API documentation for possible errors */
	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;

	/*
	* Read flash control register contents to know the status of flash control after
	* program command to see if there are any program operation related
	* errors
	*/
	oFlashStatus = Fapi_getFsmStatus(Fapi_FlashBank0);
	if ((oFlashStatus != FLASH_CONTROL_ERASE_OKAY) &&
		(oFlashStatus != FLASH_CONTROL_PROGRAM_OKAY) &&
		(oFlashStatus != FLASH_CONTROL_IDLE)) {

		oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
		return -FDP_CHECKERR;
	}

	/* Verify the programmed values. */
#if (defined(FDP_IS_CHECK_STORAGE_WRITE) && FDP_IS_CHECK_STORAGE_WRITE)

	fdp_dcache_invaild(bank0_addr, bank0_size);

	oReturnCheck = Fapi_doVerify((uint32 *)(uintptr_t)bank0_addr,
								bank0_size / 4,
								(uint32 *)buffer,
								&oFlashStatusWord);

	if (oReturnCheck != Fapi_Status_Success)
		return -FDP_CHECKERR;
#endif

	return FDP_SUCCESS;
}

#endif

int fdp_storage_dev_read(uint8_t *buffer, uint32_t start_addr, uint32_t len)
{
	if (buffer == NULL)
		return -FDP_ERRINPARAM;

	memcpy((void *)buffer, (void *)(uintptr_t)start_addr, len);

	return FDP_SUCCESS;
}

int fdp_security_mem_check(uintptr_t addr, uint32_t size)
{
	if (addr >= BANK_MIN_ADDR && addr < BANK_MAX_ADDR) {

		if (addr + size > BANK_MAX_ADDR)
			return -FDP_ERRMEMO;

		return FDP_SUCCESS;
	} else
		return -FDP_ERRMEMO;
}
