/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "fdp_err.h"
#include "fdp.h"

#include "uart.h"
#include "interrupt.h"
#include "board_cfg.h"
#include "device.h"
#include "bootrom_PIN.h"
#include "boot_gpio.h"

#define BITRATE					300000U

/* extern bootromPIN_t *bootromPIN; */
extern bootromPIN_t *bootromPIN;

uint32_t UART_BASE_ADDRESS;
uint32_t UART_RX_IRQ_NUM;

static void Uart_SetPinmux(void)
{
	boot_gpio_set_pin_config(bootromPIN->UART_TX_PINMUX_Value);
	boot_gpio_set_pin_config(bootromPIN->UART_RX_PINMUX_Value);

	boot_gpio_set_analog_mode(bootromPIN->UART_TX_PIN_NUM_Value, GPIO_ANALOG_DISABLED);
	boot_gpio_set_analog_mode(bootromPIN->UART_RX_PIN_NUM_Value, GPIO_ANALOG_DISABLED);

	boot_gpio_set_pad_config(bootromPIN->UART_TX_PIN_NUM_Value, GPIO_PIN_TYPE_PULLUP);
	boot_gpio_set_pad_config(bootromPIN->UART_RX_PIN_NUM_Value, GPIO_PIN_TYPE_PULLUP);
	
	boot_gpio_set_qualification_mode(bootromPIN->UART_TX_PIN_NUM_Value, GPIO_QUAL_SYNC);
	boot_gpio_set_qualification_mode(bootromPIN->UART_RX_PIN_NUM_Value, GPIO_QUAL_SYNC);
}

void SCI_IRQ_HANDLER(void)
{
	uint32_t irq_status = SCI_getInterruptStatus(UART_BASE_ADDRESS);

	if (irq_status == SCI_RX_DATA_AVAILABLE)
		fdp_receive_data_to_buffer();

	SCI_clearInterruptStatus(UART_BASE_ADDRESS);
	__DSB();
}

int Uart_Init(void)
{
	if(bootromPIN->UART_number == 0){
		UART_BASE_ADDRESS = SCIA_BASE;
		UART_RX_IRQ_NUM   = INT_SCIA;
	} else if (bootromPIN->UART_number == 1){
		UART_BASE_ADDRESS = SCIB_BASE;
		UART_RX_IRQ_NUM   = INT_SCIB;
	} else if (bootromPIN->UART_number == 2){
		UART_BASE_ADDRESS = SCIC_BASE;
		UART_RX_IRQ_NUM   = INT_SCIC;
	} else if (bootromPIN->UART_number == 3){
		UART_BASE_ADDRESS = SCID_BASE;
		UART_RX_IRQ_NUM   = INT_SCID;
	} else {
		return -FDP_ERRINPARAM;
	}

	Uart_SetPinmux();

	SCI_enableLoopback(UART_BASE_ADDRESS);

	SCI_reset(UART_BASE_ADDRESS);

	SCI_setConfig(UART_BASE_ADDRESS, DEVICE_APBCLK_FREQ, BITRATE,
				 (SCI_CONFIG_WLEN_8 | SCI_CONFIG_STOP_ONE | SCI_CONFIG_PAR_NONE));

	SCI_ClearReset(UART_BASE_ADDRESS);

	SCI_clearInterruptStatus(UART_BASE_ADDRESS);
	SCI_enableFIFO(UART_BASE_ADDRESS);
	SCI_setFIFOInterruptLevel(UART_BASE_ADDRESS, SCI_FIFO_TX0, SCI_FIFO_RX1);

	SCI_disableInterrupt(UART_BASE_ADDRESS, 0xff);
	SCI_enableInterrupt(UART_BASE_ADDRESS, SCI_RX_DATA_AVAILABLE_INT);

#if (FDP_IS_USE_INTERRUPT)

	Interrupt_register(UART_RX_IRQ_NUM, &SCI_IRQ_HANDLER);
	Interrupt_enable(UART_RX_IRQ_NUM);

#endif

	SCI_disableLoopback(UART_BASE_ADDRESS);

	return FDP_SUCCESS;
}

int Uart_Deinit(void)
{

	SCI_enableLoopback(UART_BASE_ADDRESS);

#if (FDP_IS_USE_INTERRUPT)

	Interrupt_unregister(UART_RX_IRQ_NUM);
	Interrupt_disable(UART_RX_IRQ_NUM);

#endif

	return FDP_SUCCESS;
}

int Uart_Receiver(uint8_t *buf, uint32_t *index, uint32_t buf_len)
{
	if (buf == NULL || index == NULL)
		return -FDP_ERRINPARAM;

	while(SCI_rxDataReady(UART_BASE_ADDRESS)) {
		buf[(*index)++] = (uint8_t)SCI_readCharNonBlocking(UART_BASE_ADDRESS);
		if (*index >= buf_len)
			*index = 0;
	}

	return FDP_SUCCESS;
}

int Uart_transmit(uint8_t *buf, uint32_t len)
{
	uint16_t temp;
	uint32_t i;

	if (buf == NULL || len == 0)
		return -FDP_ERRINPARAM;

	for (i = 0; i < len; i++) {
		temp = buf[i];
		SCI_writeCharArray(UART_BASE_ADDRESS, &temp, 1);
	}

	return FDP_SUCCESS;
}
