/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __FDP_STORAGE_PORT_H__
#define __FDP_STORAGE_PORT_H__

#include "inc/hw_types.h"

#include "fdp_defconfig.h"

/* Flash version. */
#define BANK_MIN_ADDR		0x08000000U
#define BANK_MAX_ADDR		0x08200000U

/**
 * @brief FDP storage device initialization function.
 * Users need to fill in this function themselves.
 * 
 * @param address Memory address(Memory address that needs to be initialized).
 * @param size Memory aera size(Memory area size that needs to be initialized).
 * @return int return non-value means initialize fail, else success.
 */
RAMFUNC_T int fdp_storage_dev_init(uint32_t start_addr, uint32_t size);

/**
 * @brief FDP storage device write function.
 * Users need to fill in this function themselves.
 * 
 * @param buffer Data buffer to be written to memory.
 * @param start_addr Memory address to be written to memory.
 * @param len Data length to be written to memory.
 * @return int return non-zero value is fail, else success.
 */
RAMFUNC_T int fdp_storage_dev_write(uint8_t *buffer, uint32_t start_addr, uint32_t len);

/**
 * @brief Erase memory area.
 * If the original data needs to be erased before writing new data,
 * please implement this interface. else return 0.
 * 
 * @param start_addr Memory address to be written to memory.
 * @param size Erase memory area size.
 * @return int return non-zero value is fail, else success.
 */
RAMFUNC_T int fdp_storage_dev_erase(uint32_t start_addr, uint32_t size);

/**
 * @brief Read data from memory.
 * 
 * @param buffer Received data buffer.
 * @param start_addr Read data from the memory address.
 * @param len Read the length of the data.
 * @return int return non-zero value is fail, else success.
 */
int fdp_storage_dev_read(uint8_t *buffer, uint32_t start_addr, uint32_t len);

/**
 * @brief Check if the memory address is in the security area.
 * The API function is used to check whether the memory address is in the security area.
 * If the address is in the security area, the API function returns an zero value.
 * else please return an non-zero value.
 * 
 * @param addr The memory address.
 * @param size The size of memory area.
 * @return int return non-zero value means check fail, else success.
 */
int fdp_security_mem_check(uintptr_t addr, uint32_t size);

#endif
