/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <stdbool.h>

#include "sysctl.h"

#include "fdp_app_info.h"
#include "fdp_crc.h"
#include "fdp_err.h"
#include "fdp_storage_port.h"

int fdp_check_image_info(uintptr_t info_addr)
{
	uint32_t crc32_val;

	fdp_app_image_info_t *info = (fdp_app_image_info_t *)info_addr;

	if (!info)
		return -FDP_CHECKERR;

	/* verify key value. */
	crc32_val = fdp_crc32((uint8_t *)&info->key, sizeof(info->key));
	if (crc32_val != info->key_check)
		return -FDP_CHECKERR;

	return FDP_SUCCESS;
}

int fdp_check_image_base_on_info(uintptr_t info_addr)
{
	volatile uint32_t crc32_val;
	fdp_app_image_info_t *info;

	if (fdp_check_image_info(info_addr) != FDP_SUCCESS)
		return -FDP_CHECKERR;

	info = (fdp_app_image_info_t *)info_addr;

//	/* Check if the address is in the range of the flash memory. */
//	if ((info->app_start_addr < BANK_MIN_ADDR) || (info->app_start_addr >= BANK_MAX_ADDR))
//		return -FDP_CHECKERR;

	crc32_val = fdp_crc32((uint8_t *)info->app_start_addr, info->image_size);

	if (crc32_val != info->image_check)
		return -FDP_CHECKERR;

	return FDP_SUCCESS;
}

bool fdp_is_entry_load_mode(void)
{
	if (SysCtl_getDigSysctlPorSpare0() != FDP_BOOT_UPDATE_FLAG)
		return false;
	else
		return true;
}

void fdp_set_entry_load_mode(bool enable)
{
	if (enable)
		SysCtl_setDigSysctlPorSpare0(FDP_BOOT_UPDATE_FLAG);
	else
		SysCtl_setDigSysctlPorSpare0(FDP_BOOT_UPDATE_DISABLE);
}
