/*
 * Copyright (c) 2019 Nuclei Limited. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/******************************************************************************
 * @file startup_gs32_dsp.S
 * @brief Boot ROM start up.
 * @version V1.00
 * @date 1. Jul 2024
 *
 * commit history:
 * 2024/07/1, Jihong, add DSP BootROM code.
 ******************************************************************************/

#if __riscv
#include "riscv_encoding.h"

/* If BOOT_HARTID is not defined, default value is 0 */
#ifndef BOOT_HARTID
	.equ BOOT_HARTID, 0
#endif

.macro DECLARE_INT_HANDLER  INT_HDL_NAME
#if defined(__riscv_xlen) && (__riscv_xlen == 32)
	.word \INT_HDL_NAME
#else
	.dword \INT_HDL_NAME
#endif
.endm

	.section .init

	.globl _start
	.type _start, @function

/**
 * Reset Handler called on controller reset
 */
_start:
	/* Disable Global Interrupt */
	csrc CSR_MSTATUS, MSTATUS_MIE

	/*enable nice instruction*/
	lui	a5,0x18
	csrs	mstatus,a5

	/* If SMP_CPU_CNT is not defined,
	* assume that only 1 core is allowed to run,
	* the core hartid is defined via BOOT_HARTID.
	* other harts if run to here, just do wfi in __amp_wait
	*/
#ifndef SMP_CPU_CNT
	/* take bit 0-7 for hart id in a local cluster */
	csrr a0, CSR_MHARTID
	andi a0, a0, 0xFF
	/* BOOT_HARTID is configurable in Makefile via BOOT_HARTID variable */
	li a1, BOOT_HARTID
	bne a0, a1, __smp_wait
#endif

	/* Initialize GP and TP */
	.option push
	.option norelax
#if defined(__USE_GP_REL16)
	la gp, __gp_rel16$
#else
	la gp, __global_pointer$
#endif
	la tp, __tls_base
	.option pop

	/* Set correct sp for current cpu */
	la sp, _sp

	/* Enable FPU and Vector Unit if f/d/v exist in march */
#if defined(__riscv_flen) && __riscv_flen > 0
	/* Enable FPU, and set state to initial */
	li t0, MSTATUS_FS
	csrc mstatus, t0
	li t0, MSTATUS_FS_INITIAL
	csrs mstatus, t0
#endif

	/* BPU cold bringup need time, so enable BPU before enter to main */
	li t0, MMISC_CTL_BPU
	csrs CSR_MMISC_CTL, t0

__init_common:
	la a4, __copy_table_start
	la a5, __copy_table_end

1:	bgeu a4, a5, 3f
	lw a0, (a4)			/* LMA */
	lw a1, 4(a4)		/* VMA */
	lw a2, 8(a4)		/* Length */
	addi a4, a4, 12

	beq a0, a1, 1b		/* LMA == VMA ? */
	beq a2, zero, 1b	/* Length == 0 ? */
2:
	lw t0, (a0)
	sw t0, (a1)
	addi a0, a0, 4
	addi a1, a1, 4
	addi a2, a2, -1
	bne a2, zero, 2b
	j 1b
3:

	la a4, __zero_table_start
	la a5, __zero_table_end

1:	bgeu a4, a5, 3f
	lw a0, (a4)			/* start */
	lw a2, 4(a4)		/* length */
	addi a4, a4, 8

	beq a2, zero, 1b
2:
	sw zero, (a0)
	addi a0, a0, 4
	addi a2, a2, -1
	bne a2, zero, 2b
	j 1b
3:

	call SystemInit

	call _premain_init

	/* argc = argv = 0 */
	li a0, 0
	li a1, 0
	call main

__smp_wait:
1:
	wfi
	j 1b

#endif /* __riscv */
