/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


//
// Included Files
//
#include "bootloader_can.h"
#include "boot_gpio.h"

//
// Defines
//
#define SELECT_EXTERNAL_OSC    0x1UL
#define CLOCK_DIVIDER_1        0x0U
#define CAN_DISABLE_PARITY     0x5UL
#define CAN_ENABLE_PARITY      0xAUL
#define CAN_11_BIT_ID_S        18U
#define CAN_RX_MSG_ID          0x1UL
#define CAN_TX_MSG_ID          0x2UL
#define CAN_MSG_OBJ_1          0x1U
#define CAN_MSG_OBJ_2          0x2U
#define CAN_DLC_SIZE           0x2U

//
// The key value for RAM initialization
//
#define CAN_RAM_INIT_KEY           (0xAU)

/* Configure CAN Receive Ring Buffer Size and the buffer size
can not be less than device frame size. */
#define CAN_RCV_RING_FRAME_BUFFER_SIZE			128U


typedef struct _CAN_RCV_RING_BUF{
	uint8_t  buffer[CAN_RCV_RING_FRAME_BUFFER_SIZE];
	uint32_t wIdx; /* write index to record received data ring buffer*/
	uint32_t rIdx; /* read index to record received data ring buffer*/
}CAN_RCV_RING_BUF;

//
// Globals
//
uint32_t CAN_TX_PINMUX;
uint32_t CAN_RX_PINMUX;
uint16_t CAN_TX_PINNUM;
uint16_t CAN_RX_PINNUM;
uint32_t CAN_CTRL_BASE;
uint32_t CAN_CTRL_IRQ_NUM;
uint32_t CAN_RX_ID = 0x123;
uint32_t CAN_TX_ID = 0x456;
uint32_t CanRcvReadIdex = 0U;
__attribute__((aligned(8))) CAN_RCV_RING_BUF bootCanRcvRingBuf;


//
// CAN bit timing settings for running CAN at 100kbps with a 20MHz crystal
//
// See CAN timing header for these values

//
// Function Prototypes
//
static void DCAN_Boot_GPIO(uint32_t bootMode);
static uint16_t DCAN_GetWordData(void);
static void DCAN_SendWordData(uint16_t data);
static void DCAN_ParseReservedWords(void);
static void CAN_CTRL_RcvBuf(CAN_RCV_RING_BUF *pbuf, uint32_t max_buf_len);

void can_ctrl_init(void)
{
	CAN_InitParams_t initParams = {0};
	CAN_FilterElement_t stdFiltelem = {0};

	CAN_CTRL_BASE    = CANA_BASE;
	CAN_CTRL_IRQ_NUM = INT_CANA;

	initParams.fdMode = false;
	initParams.txbmode = CAN_TS_MODE_FIFO;
	initParams.timestamp = CAN_TIMESTAMP_EOF;

	/* accept all CAN ID frames. */
	stdFiltelem.acode = CAN_RX_ID;
	stdFiltelem.amsk = 0;
	stdFiltelem.filerFrameType = CAN_IDTYPE_ONLY_STD;

	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);
	/* CAN clock source = 40MHz, PLL_CLK = 2 * SYS_CLK */
	SysCtl_setCanClkDiv((DEVICE_SYSCLK_FREQ * 2 / 40000000));

	/* Initial CAN module. */
	CAN_initModule(CAN_CTRL_BASE);

	/* Configuration CAN basic function. */
	CAN_initConfig(CAN_CTRL_BASE, &initParams);

	/* Set the arbitration segment bit rate based on sampling points. */
	CAN_setBitRateSamplePoint(CAN_CTRL_BASE, 40000000, 500 * 1000, 0.8);

	/* Set CAN ID acceptance filter 0. */
	CAN_configMsgFilter(CAN_CTRL_BASE, 0, &stdFiltelem);

	/* Enable CAN ID acceptance filter 0. */
	CAN_configMsgFilterEnable(CAN_CTRL_BASE, 0);

	/* Start running CAN module. */
	CAN_startModule(CAN_CTRL_BASE);

	/* Enable auto retransmission. */
	CAN_setAutoRetransmission(CAN_CTRL_BASE, true);

	CAN_enableInterrupt(CAN_CTRL_BASE, CAN_RTIE_RIE_M);

#if (FDP_IS_USE_INTERRUPT)
	Interrupt_SetPriority(CAN_CTRL_IRQ_NUM, 0, 0);
	Interrupt_register(CAN_CTRL_IRQ_NUM, &can_interrupt_irq_handler);
	Interrupt_enable(CAN_CTRL_IRQ_NUM);
#endif

}

void CAN_CTRL_RcvBuf(CAN_RCV_RING_BUF *pbuf, uint32_t max_buf_len)
{
	CAN_RxMessage_t RxMsgBuf = {0};
	int len;

	// read a frame
	while(!CAN_receiveMsg(CAN_CTRL_BASE, &RxMsgBuf));

	len = CAN_getMessageLength(RxMsgBuf.dlc);
	for (uint32_t i = 0; i < len; i++) {
		pbuf->buffer[pbuf->wIdx++] = RxMsgBuf.data[i];
		if (pbuf->wIdx >= max_buf_len)
			pbuf->wIdx = 0;
	}
}

void CAN_CTRL_transmit(uint8_t *buf, uint32_t len)
{
	CAN_TxMessage_t TxMsgBuf = {0};
	int datalen, maxlen;
	uint32_t i = 0;

	if (len == 0)
		return;

	TxMsgBuf.id = CAN_CTRL_TX_ID;

	maxlen = CAN_getMessageLength(CAN_MSG_MAX_LEN);

	while (len) {
		if (len >= maxlen) {
			TxMsgBuf.dlc = CAN_MSG_MAX_LEN;
			len -= maxlen;
		} else {
			TxMsgBuf.dlc = len;
			len = 0;
		}

		datalen = CAN_getMessageLength(TxMsgBuf.dlc);
		memcpy(TxMsgBuf.data, &buf[i], datalen);
		i += datalen;

		while(!CAN_transmitHighPriorityMsg(CAN_CTRL_BASE, &TxMsgBuf));
	}

	return;
}

/**
* DCAN_Boot - Run the CAN bootloader setup with the specified GPIOs for the
*             requested CAN boot mode
*
*
* \brief CAN Boot
*
* Design: did_can_boot_algo did_boot_first_instance_algo)
* Requirement: REQ_TAG(C2000BROM-208), REQ_TAG(C2000BROM-210)
*
* Start CAN Boot
*
*/
uint32_t DCAN_Boot(uint32_t bootMode)
{
    uint32_t appEntryAddress = 0xFFFFFFFFUL;

    //
    // Assign the CAN data reader function to the global
    // function pointer for loading data.
    //
    GetWordData = &DCAN_GetWordData;

    //
    // Set up the GPIO mux for the chosen pinout
    //
    DCAN_Boot_GPIO(bootMode);

    //
    // Initialize the CAN-A port for communications
    // with the host.
    //
    can_ctrl_init();


//    //
//    // Testing Only: Send two tests frames if the boot mode says so
//    //
//    if(bootMode >= CAN_BOOT_SENDTEST)
//    {
//        DCAN_SendWordData(0x0320U);
//        DCAN_SendWordData(0xf280U);
//    }

    //
    // The first data word should be a valid key. If it's not,
    // bypass the bootloader.
    //
    if(DCAN_GetWordData() != BROM_EIGHT_BIT_HEADER)
    {
        return FLASH_ENTRY_POINT;
    }

    //
    // Use the shared utility functions to load the data.
    //
    DCAN_ParseReservedWords();
    appEntryAddress = GetLongData();
    CopyData();

    return appEntryAddress;
}

/**
* DCAN_Boot_GPIO - Configure the peripheral mux to connect CAN-A to the
*                  chosen GPIOs
*
*
* \brief CAN Boot GPIO select
*
* Design: did_can_boot_algo
* Requirement: REQ_TAG(C2000BROM-206)
*
* Start I2C Boot
*
*/
static void DCAN_Boot_GPIO(uint32_t bootMode)
{
    //
    // Decode the GPIO selection, then set up the mux and configure the inputs
    // for asynchronous qualification.
    //
    switch (bootMode)
    {

        case CAN_BOOT_ALT1:
        case CAN_BOOT_ALT1_SENDTEST:
            //
            // GPIO04 = CANATX
            // GPIO05 = CANARX
            //
            CAN_TX_PINMUX = GPIO_4_CANA_TX;
            CAN_RX_PINMUX = GPIO_5_CANA_RX;
        	CAN_TX_PINNUM = 4UL;
        	CAN_RX_PINNUM = 5UL;
            break;

        case CAN_BOOT_ALT2:
        case CAN_BOOT_ALT2_SENDTEST:
            //
            // GPI19 = CANATX
            // GPI18 = CANARX
            //
            CAN_TX_PINMUX = GPIO_19_CANA_TX;
            CAN_RX_PINMUX = GPIO_18_CANA_RX;
        	CAN_TX_PINNUM = 19UL;
        	CAN_RX_PINNUM = 18UL;
            break;

        case CAN_BOOT_ALT3:
            //
            // GPIO37 = CANARX
            // GPIO36 = CANATX
            //
            CAN_TX_PINMUX = GPIO_37_CANA_TX;
            CAN_RX_PINMUX = GPIO_36_CANA_RX;
        	CAN_TX_PINNUM = 37UL;
        	CAN_RX_PINNUM = 36UL;
            break;

        case CAN_BOOT_ALT4:
            //
            // GPIO63 = CANARX
            // GPIO62 = CANATX
            //
            CAN_TX_PINMUX = GPIO_63_CANA_TX;
            CAN_RX_PINMUX = GPIO_62_CANA_RX;
        	CAN_TX_PINNUM = 63UL;
        	CAN_RX_PINNUM = 62UL;
            break;

        case CAN_BOOT:
        case CAN_BOOT_SENDTEST:
        default:
            //
            // GPIO59 = CANATX
            // GPIO58 = CANARX
            //
            CAN_TX_PINMUX = GPIO_59_CANA_TX;
            CAN_RX_PINMUX = GPIO_58_CANA_RX;
        	CAN_TX_PINNUM = 59UL;
        	CAN_RX_PINNUM = 58UL;
            break;

    }

    //
    // Set GPIO configuration for CAN
    //
    boot_gpio_set_pin_config(CAN_TX_PINMUX);
    boot_gpio_set_pin_config(CAN_RX_PINMUX);

    //
    // Disable analog func
    //
	boot_gpio_set_analog_mode(CAN_TX_PINNUM, GPIO_ANALOG_DISABLED);
	boot_gpio_set_analog_mode(CAN_RX_PINNUM, GPIO_ANALOG_DISABLED);

    //
    // Enable pull up on GPIOs pins
    //
	boot_gpio_set_pad_config(CAN_TX_PINNUM,GPIO_PIN_TYPE_PULLUP);
	boot_gpio_set_pad_config(CAN_RX_PINNUM,GPIO_PIN_TYPE_PULLUP);

    //
    // Configure GPIOs as async pins
    //
	boot_gpio_set_qualification_mode(CAN_TX_PINNUM,GPIO_QUAL_ASYNC);
	boot_gpio_set_qualification_mode(CAN_RX_PINNUM,GPIO_QUAL_ASYNC);
}


//
// DCAN_GetWordData - Read 16 bits from an incoming DCAN message sent to ID #1.
//                    If no message has been received, wait for one to arrive.
//
static uint16_t DCAN_GetWordData(void)
{
	uint16_t gap = 0;
	uint16_t worddata= 0;

	if (bootCanRcvRingBuf.wIdx >= bootCanRcvRingBuf.rIdx)
	{
		gap = bootCanRcvRingBuf.wIdx -  bootCanRcvRingBuf.rIdx;
	}
	else if (bootCanRcvRingBuf.wIdx < bootCanRcvRingBuf.rIdx)
	{
		gap = CAN_RCV_RING_FRAME_BUFFER_SIZE + bootCanRcvRingBuf.wIdx -  bootCanRcvRingBuf.rIdx;
	}

	while (gap < 2U)
	{
		CAN_CTRL_RcvBuf(&bootCanRcvRingBuf, CAN_RCV_RING_FRAME_BUFFER_SIZE);
		//Update  gap
		if (bootCanRcvRingBuf.wIdx >= bootCanRcvRingBuf.rIdx)
		{
			gap = bootCanRcvRingBuf.wIdx -  bootCanRcvRingBuf.rIdx;
		}
		else if (bootCanRcvRingBuf.wIdx < bootCanRcvRingBuf.rIdx)
		{
			gap = CAN_RCV_RING_FRAME_BUFFER_SIZE + bootCanRcvRingBuf.wIdx -  bootCanRcvRingBuf.rIdx;
		}
	}

	if (gap >= 2U)
	{
		worddata = bootCanRcvRingBuf.buffer[bootCanRcvRingBuf.rIdx++];
		if (bootCanRcvRingBuf.rIdx >= CAN_RCV_RING_FRAME_BUFFER_SIZE)
			bootCanRcvRingBuf.rIdx = 0U;
		worddata |= (bootCanRcvRingBuf.buffer[bootCanRcvRingBuf.rIdx++])<<8;
		if (bootCanRcvRingBuf.rIdx >= CAN_RCV_RING_FRAME_BUFFER_SIZE)
			bootCanRcvRingBuf.rIdx = 0U;
	}
	else
	{
		worddata = 0xFFFF;
		bootCanRcvRingBuf.wIdx= bootCanRcvRingBuf.rIdx;
	}
//	CAN_CTRL_transmit((uint8_t *)&worddata, sizeof(uint16_t));

	return worddata;
}

//
// DCAN_SendWordData - Send a CAN message to ID #2 for external testing and
//                     data rate measurement. Wait for the transmission to
//                     complete.
//
static void DCAN_SendWordData(uint16_t data)
{



}

//
// DCAN_ParseReservedWords - Parse the eight reserved words and check whether
//                           there's a new bit timing register value in the
//                           first pair.
//
static void DCAN_ParseReservedWords(void)
{
    uint16_t word;

    //
    // Skip the rest of the reserved words
    //
    for(word = 1U; word <= 8U; word++)
    {
        (void)DCAN_GetWordData();
    }
}

//
// End of File
//
