/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


//
// Included Files
//
#include "bootloader_sci.h"
#include "boot_gpio.h"

//
// Globals
//
uint32_t SCI_TX_PINMUX;
uint32_t SCI_RX_PINMUX;
uint16_t SCI_TX_PINNUM;
uint16_t SCI_RX_PINNUM;
uint32_t UART_BASE_ADDRESS;
uint32_t UART_RX_IRQ_NUM;

//
// Function Prototypes
//
uint16_t SCIA_GetWordData(void);

/**
* SCI_Init
*
* \brief SCI_Init
*/
void SCI_Init(void)
{
	UART_BASE_ADDRESS = SCIA_BASE;
	UART_RX_IRQ_NUM   = INT_SCIA_RX;

	SCI_enableLoopback(UART_BASE_ADDRESS);

	while (!SCI_getRxFIFOStatus(UART_BASE_ADDRESS) == SCI_FIFO_RX0) {
		SCI_readCharBlockingFIFO(UART_BASE_ADDRESS);
	}

	SCI_resetChannels(UART_BASE_ADDRESS);
	SCI_setConfig(UART_BASE_ADDRESS, DEVICE_APBCLK_FREQ, BITRATE,
				 (SCI_CONFIG_WLEN_8 | SCI_CONFIG_STOP_ONE | SCI_CONFIG_PAR_NONE));
	SCI_clearInterruptStatus(UART_BASE_ADDRESS, SCI_INT_RXFF | SCI_INT_TXFF | SCI_INT_TXRDY);
	SCI_enableFIFO(UART_BASE_ADDRESS);
	SCI_setFIFOInterruptLevel(UART_BASE_ADDRESS, SCI_FIFO_TX0, SCI_FIFO_RX1);

	SCI_disableInterrupt(UART_BASE_ADDRESS, 0xff);
	SCI_disableInterrupt(UART_BASE_ADDRESS, SCI_INT_TXFF | SCI_INT_TXRDY);
	SCI_enableInterrupt(UART_BASE_ADDRESS, SCI_INT_RXFF);

#if (FDP_IS_USE_INTERRUPT)

	Interrupt_register(UART_RX_IRQ_NUM, &SCI_IRQ_HANDLER);
	Interrupt_enable(UART_RX_IRQ_NUM);

#endif

	SCI_disableLoopback(UART_BASE_ADDRESS);
}

static void SCIBOOT_configure_gpio(uint32_t bootMode);

/**
* SCI_Boot - This module is the main SCI boot routine.
*            It will load code via the SCI-A port.
*
*            It will return a entry point address back
*            to the system initialization routine which in turn calls
*            the ExitBoot routine.
*
*
* \brief SCI Boot
*
* Design: did_sci_boot_algo did_boot_first_instance_algo)
* Requirement: REQ_TAG(C2000BROM-205), REQ_TAG(C2000BROM-210)
*
* Start sci Boot
*
*/
uint32_t SCI_Boot(uint32_t bootMode)
{
    uint32_t appEntryAddress = 0xFFFFFFFFUL;
    uint16_t byteData;

    //
    // Assign GetWordData to the SCI-A version of the
    // function. GetWordData is a pointer to a function.
    //
    GetWordData = SCIA_GetWordData;

    //
    // GPIO INIT
    //
    SCIBOOT_configure_gpio(bootMode);

    //
    // Initialize the SCI-A port for communications
    // with the host.
    //
    SCI_Init();


    //
    // Fetch the 'A' or 'a' and verify back to the host
    //
    byteData = SCI_readCharBlockingNonFIFO(UART_BASE_ADDRESS);
    SCI_writeCharNonBlocking(UART_BASE_ADDRESS,byteData);

    //
    // If the KeyValue was invalid, abort the load
    // and return the flash entry point.
    //
    if(SCIA_GetWordData() != BROM_EIGHT_BIT_HEADER)
    {
        return FLASH_ENTRY_POINT;
    }

    ReadReservedFn();

    appEntryAddress = GetLongData();

    CopyData();

    return appEntryAddress;
}

//
// SCIA_GetWordData - This routine fetches two bytes from the SCI-A
//                    port and puts them together to form a single
//                    16-bit value.  It is assumed that the host is
//                    sending the data in the order LSB followed by MSB.
//
uint16_t SCIA_GetWordData(void)
{
    uint16_t wordData;
    uint16_t byteData;

    //
    // Fetch the LSB and verify back to the host
    //
    wordData = SCI_readCharBlockingNonFIFO(UART_BASE_ADDRESS);

    SCI_writeCharNonBlocking(UART_BASE_ADDRESS,wordData);

    byteData = SCI_readCharBlockingNonFIFO(UART_BASE_ADDRESS);

    SCI_writeCharNonBlocking(UART_BASE_ADDRESS,byteData);

    //
    // Form the wordData from the MSB:LSB
    //
    wordData |= (byteData << 8U);

    return wordData;
}

//
// SCIBOOT_configure_gpio - This function configures the required GPIOs for the
//                          specified SCI boot mode
//
static void SCIBOOT_configure_gpio(uint32_t bootMode)
{
    switch (bootMode)
    {
        case SCI_BOOT_ALT1:

        	SCI_TX_PINMUX = GPIO_16_SCIA_TX;
        	SCI_RX_PINMUX = GPIO_17_SCIA_RX;
        	SCI_TX_PINNUM = 16;
        	SCI_RX_PINNUM = 17;
            break;

        case SCI_BOOT_ALT2:

        	SCI_TX_PINMUX = GPIO_8_SCIA_TX;
        	SCI_RX_PINMUX = GPIO_9_SCIA_RX;
        	SCI_TX_PINNUM = 8;
        	SCI_RX_PINNUM = 9;

            break;

        case SCI_BOOT_ALT3:

        	SCI_TX_PINMUX = GPIO_2_SCIA_TX;
        	SCI_RX_PINMUX = GPIO_3_SCIA_RX;
        	SCI_TX_PINNUM = 2;
        	SCI_RX_PINNUM = 3;

            break;

        case SCI_BOOT_ALT4:
        	SCI_TX_PINMUX = GPIO_16_SCIA_TX;
        	SCI_RX_PINMUX = GPIO_3_SCIA_RX;
        	SCI_TX_PINNUM = 16;
        	SCI_RX_PINNUM = 3;

            break;

        case SCI_BOOT:
        default:
            //
            // GPIO29 = SCIATX
            // GPIO28 = SCIARX
            //
        	SCI_TX_PINMUX = GPIO_29_SCIA_TX;
        	SCI_RX_PINMUX = GPIO_28_SCIA_RX;
        	SCI_TX_PINNUM = 29;
        	SCI_RX_PINNUM = 28;
            break;
    }

	boot_gpio_set_pin_config(SCI_TX_PINMUX);
	boot_gpio_set_pin_config(SCI_RX_PINMUX);

	boot_gpio_set_analog_mode(SCI_TX_PINNUM, GPIO_ANALOG_DISABLED);
	boot_gpio_set_analog_mode(SCI_RX_PINNUM, GPIO_ANALOG_DISABLED);

	boot_gpio_set_pad_config(SCI_TX_PINNUM, GPIO_PIN_TYPE_PULLUP);
	boot_gpio_set_pad_config(SCI_RX_PINNUM, GPIO_PIN_TYPE_PULLUP);

	boot_gpio_set_qualification_mode(SCI_TX_PINNUM, GPIO_QUAL_SYNC);
	boot_gpio_set_qualification_mode(SCI_RX_PINNUM, GPIO_QUAL_SYNC);
}

//
// End of File
//
