/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <string.h>

#include "device.h"
#include "can_transfer.h"
#include "msg_memory.h"

#define CAN_CTRL_RX_ID		0x456
#define CAN_CTRL_TX_ID		0x123

#define CAN_MSG_MAX_LEN		CAN_DATA_LENGTH_8

CAN_RxMessage_t RxMsgBuf = {0};
CAN_TxMessage_t TxMsgBuf = {0};

void can_interrupt_irq_handler(void)
{
	uint32_t status;
	uint8_t length;

	status = CAN_getIntrStatus(CAN_BASE);

	if (status & CAN_RTIF_RIF_M) {
		CAN_receiveMsg(CAN_BASE, &RxMsgBuf);
		length = CAN_getMessageLength(RxMsgBuf.dlc);

		for (uint8_t i = 0; i < length; i++)
			mem_write_buf(MEM_BUF_CAN, RxMsgBuf.data[i]);
	}

	/* Check if Rx buffer is empty.
	 * If the Rx buffer is not empty,
	 * do not clear the receive interrupt. */
	if (CAN_getRxBufState(CAN_BASE) == CAN_RXBUFFER_EMPTY)
		CAN_clearIntrStatus(CAN_BASE, CAN_RTIF_RIF_M);
}

static void can_config(void)
{
	CAN_InitParams_t initParams = {0};
	CAN_FilterElement_t stdFiltelem = {0};

	initParams.fdMode = false;
	initParams.txbmode = CAN_TS_MODE_FIFO;
	initParams.timestamp = CAN_TIMESTAMP_EOF;

	/* accept all CAN ID frames. */
	stdFiltelem.acode = CAN_CTRL_RX_ID;
	stdFiltelem.amsk = 0;
	stdFiltelem.filerFrameType = CAN_IDTYPE_BOTH_STD_EXT;

	/* Initial CAN module. */
	if (!CAN_initModule(CAN_BASE))
		while (1);

	/* Configuration CAN basic function. */
	if (!CAN_initConfig(CAN_BASE, &initParams))
		while (1);

	/* Set the arbitration segment bit rate based on sampling points. */
	if (!CAN_setBitRateSamplePoint(CAN_BASE, 40000000, 500 * 1000, 0.8))
		while (1);

	/* Set CAN ID acceptance filter 0. */
	if (!CAN_configMsgFilter(CAN_BASE, 0, &stdFiltelem))
		while (1);

	/* Enable CAN ID acceptance filter 0. */
	if (!CAN_configMsgFilterEnable(CAN_BASE, 0))
		while (1);

	/* Start running CAN module. */
	if (!CAN_startModule(CAN_BASE))
		while (1);

	/* Enable auto retransmission. */
	if (!CAN_setAutoRetransmission(CAN_BASE, true))
		while (1);
}

void can_pinmux_init(void)
{
	GPIO_setPinConfig(CAN_RX_PIN);
	GPIO_setPinConfig(CAN_TX_PIN);
}

void can_ctrl_init(void)
{
	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);

	/* CAN clock source = 40MHz, PLL_CLK = 2* SYS_CLK */
	SysCtl_setCanClkDiv((DEVICE_SYSCLK_FREQ * 2 / 40000000));

	can_pinmux_init();

	can_config();

	Interrupt_register(CAN_IRQ_NUM, &can_interrupt_irq_handler);
	Interrupt_enable(CAN_IRQ_NUM);
	CAN_enableInterrupt(CAN_BASE, CAN_RTIE_RIE_M);

	TxMsgBuf.id = CAN_CTRL_TX_ID;
}

void can_transmit_uart_msg(void)
{
	uint8_t data;
	uint8_t i = 0;
	uint64_t cpu_counter_start, cpu_counter;

	if (!mem_read_buf(MEM_BUF_UART, &data)) {
		cpu_counter_start = __get_rv_cycle();

		TxMsgBuf.data[i++] = data;

		while (1) {
			if (!mem_read_buf(MEM_BUF_UART, &data)) {
				TxMsgBuf.data[i++] = data;
				if (i == 8) {
					TxMsgBuf.dlc = CAN_DATA_LENGTH_8;
					while (!CAN_transmitMsg(CAN_BASE, &TxMsgBuf));
					i = 0;
				}
			} else {
				cpu_counter = __get_rv_cycle();
				/* Wait ms to check if there is still data. */
				if (cpu_counter - cpu_counter_start >= (DEVICE_SYSCLK_FREQ / 1000) * 2)
					break;
			}
		}
	}

	if (i > 0) {
		TxMsgBuf.dlc = i;
		while (!CAN_transmitMsg(CAN_BASE, &TxMsgBuf));
	}
}
