/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <stdio.h>

#include "msg_memory.h"

#define CAN_RECEIVED_BUF_SIZE			4096U
#define UART_RECEIVED_BUF_SIZE			4096U

static uint8_t CAN_RECEIVE_BUF[CAN_RECEIVED_BUF_SIZE];
static uint8_t UART_RECEIVE_BUF[UART_RECEIVED_BUF_SIZE];

static mem_buf_status_t mem_can_ctrl_status;
static mem_buf_status_t mem_uart_status;

void mem_init_status(void)
{
	/* initialize CAN status. */
	mem_can_ctrl_status.buf_size = CAN_RECEIVED_BUF_SIZE;
	mem_can_ctrl_status.buf = CAN_RECEIVE_BUF;
	mem_can_ctrl_status.read_index = 0;
	mem_can_ctrl_status.write_index = 0;

	/* initialize UART status. */
	mem_uart_status.buf_size = UART_RECEIVED_BUF_SIZE;
	mem_uart_status.buf = UART_RECEIVE_BUF;
	mem_uart_status.read_index = 0;
	mem_uart_status.write_index = 0;
}

static mem_buf_status_t *mem_get_buf_status(mem_buf_type_t mem)
{
	switch (mem) {
		case MEM_BUF_CAN :
			return &mem_can_ctrl_status;

		case MEM_BUF_UART :
			return &mem_uart_status;

		default :
			return NULL;
	}
}

int mem_write_buf(mem_buf_type_t mem, uint8_t data)
{
	mem_buf_status_t *status = mem_get_buf_status(mem);

	if (status == NULL)
		return -1;

	status->buf[status->write_index++] = data;

	if (status->write_index >= status->buf_size)
		status->write_index = 0;

	return 0;
}

int mem_read_buf(mem_buf_type_t mem, uint8_t *data)
{
	mem_buf_status_t *status = mem_get_buf_status(mem);

	if (status == NULL)
		return -1;

	if (status->read_index != status->write_index)
		*data = status->buf[status->read_index++];
	else
		return -2;

	if (status->read_index >= status->buf_size)
		status->read_index = 0;

	return 0;
}
