/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "uart.h"
#include "interrupt.h"
#include "device.h"

#include "msg_memory.h"

#define UART_BASE_ADDRESS		SCIA_BASE

#if (UART_BASE_ADDRESS == SCIA_BASE)

#define UART_RX_IRQ_NUM			INT_SCIA
#define BITRATE					300000

#define SCI_TX_PIN				GPIO_16_SCIA_TX
#define SCI_RX_PIN				GPIO_17_SCIA_RX

#define SCI_TX_PIN_NUM			16
#define SCI_RX_PIN_NUM			17

#elif (UART_BASE_ADDRESS == SCIB_BASE)

#define UART_RX_IRQ_NUM			INT_SCIB
#define BITRATE					300000

#define SCI_TX_PIN				GPIO_14_SCIB_TX
#define SCI_RX_PIN				GPIO_15_SCIB_RX

#define SCI_TX_PIN_NUM			14
#define SCI_RX_PIN_NUM			15

#endif

static void Uart_SetPinmux(void)
{
	GPIO_setPinConfig(SCI_TX_PIN);
	GPIO_setPinConfig(SCI_RX_PIN);
	GPIO_setPadConfig(SCI_TX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
	GPIO_setPadConfig(SCI_RX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
}

void SCI_IRQ_HANDLER(void)
{
	while(SCI_rxDataReady(UART_BASE_ADDRESS)) {
		mem_write_buf(MEM_BUF_UART, (uint8_t)SCI_readCharNonBlocking(UART_BASE_ADDRESS));
	}

	SCI_clearInterruptStatus(UART_BASE_ADDRESS);
}

void Uart_Init(void)
{
	Uart_SetPinmux();

	SCI_enableLoopback(UART_BASE_ADDRESS);

	while (SCI_rxDataReady(UART_BASE_ADDRESS)) {
		SCI_readData(UART_BASE_ADDRESS);
	}

	SCI_resetChannels(UART_BASE_ADDRESS);
	SCI_setConfig(UART_BASE_ADDRESS, DEVICE_APBCLK_FREQ, BITRATE,
				 (SCI_CONFIG_WLEN_8 | SCI_CONFIG_STOP_ONE | SCI_CONFIG_PAR_NONE));
	SCI_clearInterruptStatus(UART_BASE_ADDRESS);
	SCI_enableFIFO(UART_BASE_ADDRESS);
	SCI_setFIFOInterruptLevel(UART_BASE_ADDRESS, SCI_FIFO_TX0, SCI_FIFO_RX1);

	SCI_disableInterrupt(UART_BASE_ADDRESS, 0xff);
	SCI_enableInterrupt(UART_BASE_ADDRESS, SCI_RX_DATA_AVAILABLE_INT);

	Interrupt_register(UART_RX_IRQ_NUM, &SCI_IRQ_HANDLER);
	Interrupt_enable(UART_RX_IRQ_NUM);

	SCI_disableLoopback(UART_BASE_ADDRESS);
}

void uart_transmit_can_msg(void)
{
	uint16_t data;

	if (mem_read_buf(MEM_BUF_CAN, (uint8_t *)&data))
		return ;

	SCI_writeCharArray(UART_BASE_ADDRESS, &data, 1);
}
