/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __BOOT_FLASH_V3_0_H__
#define __BOOT_FLASH_V3_0_H__

#ifdef __cplusplus
extern "C"{
#endif

#if (GS32F00xx == 0x3000)

#include "boot_hw_type.h"
#include "inc/hw_memmap.h"

/**
 * @brief number of flash controller instance.
 *
 */
#define BOOT_FLASH_INSTANCE_NUM							2U

/**
 * @brief base address of flash controller.
 *
 */
#define BOOT_FLASH_EFC0_ADDR						FLASH_INTERFACE_EFC_BASE
#define BOOT_FLASH_EFC1_ADDR						FLASH_INTERFACE_EFC2_BASE

/**
 * @brief enable/disable the check base address switch.
 *
 */
#define BOOT_FLASH_CHECK_BASE_SWITCH					1U

/**
 * @brief flash spececific parameters.
 *
 */
#define BOOT_FLASH_BANK0_MIN_ADDR						0x08000000U
#define BOOT_FLASH_BANK0_MAX_ADDR						0x0803ffffU

#define BOOT_FLASH_BANK1_MIN_ADDR						0x08040000U
#define BOOT_FLASH_BANK1_MAX_ADDR						0x0807ffffU

#define BOOT_FLASH_BANK_MIN_ADDR						0x08000000U
#define BOOT_FLASH_BANK_MAX_ADDR						0x0807ffffU

#define BOOT_FLASH_MAIN_ADDR_MASK						0x0003FFF0U
#define BOOT_FLASH_INFO_ADDR_MASK						0x0007FFF0U

#define BOOT_FLASH_DUAL_BANK_SECTOR_SIZE				0x00002000U
#define BOOT_FLASH_SINGLE_BANK_SECTOR_SIZE				0x00004000U

#define BOOT_FLASH_MIN_WRITE_SIZE						16U

/**
 * @brief Set the storage area for boot flash functions.
 *
 */
#define BOOT_FLASH_SECTION		__attribute__((section(".boot_flash_sec")))

/**
 * @brief Flash controller registers offset.
 *
 */
#define BOOT_FLASH_CONTROL								0x0U
#define BOOT_FLASH_TIMING_0								0x4U
#define BOOT_FLASH_TIMING_1								0x8U
#define BOOT_FLASH_TIMING_2								0xCU
#define BOOT_FLASH_STATIC_CFG							0x10U
#define BOOT_FLASH_WRITE_PROTECT						0x14U
#define BOOT_FLASH_UNLOCK_1								0x18U
#define BOOT_FLASH_UNLOCK_2								0x1CU
#define BOOT_FLASH_ADDRESS								0x20U
#define BOOT_FLASH_DATA_0								0x24U
#define BOOT_FLASH_DATA_1								0x28U
#define BOOT_FLASH_DATA_2								0x2CU
#define BOOT_FLASH_DATA_3								0x30U
#define BOOT_FLASH_STATUS								0x44U
#define BOOT_FLASH_REDUN_ADDR_0							0x48U
#define BOOT_FLASH_REDUN_ADDR_1							0x4CU
#define BOOT_FLASH_SINGLE_ERR_COUNTER					0x50U
#define BOOT_FLASH_DOUBLE_ERR_COUNTER					0x54U
#define BOOT_FLASH_SMW_TIMER_OPT						0x5CU
#define BOOT_FLASH_SMW_SET_OPT1_REG0					0x60U
#define BOOT_FLASH_SMW_SET_OPT1_REG1					0x64U
#define BOOT_FLASH_SMW_SET_OPT2							0x68U
#define BOOT_FLASH_SMW_SET_OPT3							0x6CU
#define BOOT_FLASH_SMW_SMP_WHV_OPT0						0x70U
#define BOOT_FLASH_SMW_SMP_WHV_OPT1						0x74U
#define BOOT_FLASH_SMW_SME_OPT0							0x78U
#define BOOT_FLASH_SMW_SME_OPT1							0x7CU
#define BOOT_FLASH_MISC_CFG								0x80U
#define BOOT_FLASH_PREFETCH_CFG							0x84U
#define BOOT_FLASH_INFO_PROTECT_CFG						0x90U
#define BOOT_FLASH_OPT_UNLOCK							0x100U
#define BOOT_FLASH_OPT_LOCK								0x104U
#define BOOT_FLASH_OPT_MODE_CTRL						0x108U
#define BOOT_FLASH_OPT_MODE_STATUS						0x10CU
#define BOOT_FLASH_SET_USER_OPT_0						0x200U
#define BOOT_FLASH_SET_USER_OPT_1						0x204U
#define BOOT_FLASH_SET_RDP								0x208U
#define BOOT_FLASH_SET_PCROP_L							0x20CU
#define BOOT_FLASH_SET_PCROP_H							0x210U
#define BOOT_FLASH_SET_WRP_L							0x214U
#define BOOT_FLASH_SET_WRP_H							0x218U
#define BOOT_FLASH_SET_SEC_SIZE							0x21CU
#define BOOT_FLASH_SET_BOOT_LOCK						0x220U
#define BOOT_FLASH_SET_PCROP_RDP						0x224U
#define BOOT_FLASH_SET_SEC_DEBUG_EN						0x228U

/**
 * @brief FLASH Controller control register bit fields.
 *
 */
#define BOOT_FLASH_CONTROL_MAIN_MAS_ERASE_M				0x00000008U
#define BOOT_FLASH_CONTROL_MAIN_MAS_ERASE_S				3U
#define BOOT_FLASH_CONTROL_ERASE_PAGE_M					0x00000010U
#define BOOT_FLASH_CONTROL_ERASE_PAGE_S					4U
#define BOOT_FLASH_CONTROL_PROGRAM_START_M				0x00000020U
#define BOOT_FLASH_CONTROL_PROGRAM_START_S				5U
#define BOOT_FLASH_CONTROL_ERASE_REFERENCE_CELL_M		0x00000040U
#define BOOT_FLASH_CONTROL_ERASE_REFERENCE_CELL_S		6U
#define BOOT_FLASH_CONTROL_RECALL_TRIM_CODE_M			0x00000080U
#define BOOT_FLASH_CONTROL_RECALL_TRIM_CODE_S			7U
#define BOOT_FLASH_CONTROL_PROGRAM_END_M				0x00000100U
#define BOOT_FLASH_CONTROL_PROGRAM_END_S				8U

/**
 * @brief FLASH Controller timing 0 register bit fields.
 *
 */
#define BOOT_FLASH_TIMING_0_READ_TCYC_M					0x0000003FU
#define BOOT_FLASH_TIMING_0_READ_TCYC_S					0U
#define BOOT_FLASH_TIMING_0_PROG_CYCLES_M				0x0003FFC0U
#define BOOT_FLASH_TIMING_0_PROG_CYCLES_S				6U
#define BOOT_FLASH_TIMING_0_EIGHT_US_CYCLES_M			0xFFFC0000U
#define BOOT_FLASH_TIMING_0_EIGHT_US_CYCLES_S			18U

/**
 * @brief FLASH Controller timing 1 register bit fields.
 *
 */
#define BOOT_FLASH_TIMING_1_ERC_ERASE_CYCLES_M			0x003FFFFFU
#define BOOT_FLASH_TIMING_1_ERC_ERASE_CYCLES_S			0U
#define BOOT_FLASH_TIMING_1_TWO_US_CYCLES_M				0xFFC00000U
#define BOOT_FLASH_TIMING_1_TWO_US_CYCLES_S				22U

/**
 * @brief FLASH Controller timing 2 register bit fields.
 *
 */
#define BOOT_FLASH_TIMING_2_TEST_HOLD_CYCLES_M			0x0000FFFFU
#define BOOT_FLASH_TIMING_2_TEST_HOLD_CYCLES_S			0U
#define BOOT_FLASH_TIMING_2_HUNDRED_NS_CYCLES_M			0x00FF0000U
#define BOOT_FLASH_TIMING_2_HUNDRED_NS_CYCLES_S			16U
#define BOOT_FLASH_TIMING_2_TEN_NS_CYCLES_M				0x0F000000U
#define BOOT_FLASH_TIMING_2_TEN_NS_CYCLES_S				24U

/**
 * @brief FLASH Controller static configuration register bit fields.
 *
 */
#define BOOT_FLASH_STATIC_CFG_SLP_MODE_M				0x00000001U
#define BOOT_FLASH_STATIC_CFG_SLP_MODE_S				0U
#define BOOT_FLASH_STATIC_CFG_ECC_EN_M					0x00000002U
#define BOOT_FLASH_STATIC_CFG_ECC_EN_S					1U
#define BOOT_FLASH_STATIC_CFG_FAST_PRG_MODE_M			0x00000004U
#define BOOT_FLASH_STATIC_CFG_FAST_PRG_MODE_S			2U
#define BOOT_FLASH_STATIC_CFG_ERR_INDICATE_M			0x00000200U
#define BOOT_FLASH_STATIC_CFG_ERR_INDICATE_S			9U
#define BOOT_FLASH_STATIC_CFG_WRITE_CMP_INTR_EN_M		0x00000400U
#define BOOT_FLASH_STATIC_CFG_WRITE_CMP_INTR_EN_S		10U
#define BOOT_FLASH_STATIC_CFG_ECC_INTR_EN_M				0x00000800U
#define BOOT_FLASH_STATIC_CFG_ECC_INTR_EN_S				11U
#define BOOT_FLASH_STATIC_CFG_DED_INTR_EN_M				0x00001000U
#define BOOT_FLASH_STATIC_CFG_DED_INTR_EN_S				12U

/**
 * @brief FLASH Controller write protect register bit fields.
 *
 */
#define BOOT_FLASH_WRITE_PROTECT_WRITE_PROTECT_M		0x00000001U
#define BOOT_FLASH_WRITE_PROTECT_WRITE_PROTECT_S		0U

/**
 * @brief FLASH Controller FLASH UNLOCK 0 register bit fields.
 *
 */
#define BOOT_FLASH_UNLOCK_0_KEY0_M						0x0000FFFFU
#define BOOT_FLASH_UNLOCK_0_KEY0_S						0U

/**
 * @brief FLASH Controller FLASH UNLOCK 1 register bit fields.
 *
 */
#define BOOT_FLASH_UNLOCK_1_KEY1_M						0x0000FFFFU
#define BOOT_FLASH_UNLOCK_1_KEY1_S						0U

/**
 * @brief FLASH Controller address register bit fields.
 *
 */
#define BOOT_FLASH_ADDR_GROUP_ADDR_M					0x0000000FU
#define BOOT_FLASH_ADDR_GROUP_ADDR_S					0U
#define BOOT_FLASH_ADDR_Y_ADDR_M						0x000001F0U
#define BOOT_FLASH_ADDR_Y_ADDR_S						4U
#define BOOT_FLASH_ADDR_X_ADDR_M						0x0003FE00U
#define BOOT_FLASH_ADDR_X_ADDR_S						9U
#define BOOT_FLASH_ADDR_IFREN_ADDR_M					0x00040000U
#define BOOT_FLASH_ADDR_IFREN_ADDR_S					18U

/**
 * @brief FLASH Controller program data 0 register bit fields.
 *
 */
#define BOOT_FLASH_DATA_0_M								0xFFFFFFFFU
#define BOOT_FLASH_DATA_0_S								0U

/**
 * @brief FLASH Controller program data 1 register bit fields.
 *
 */
#define BOOT_FLASH_DATA_1_M								0xFFFFFFFFU
#define BOOT_FLASH_DATA_1_S								0U

/**
 * @brief FLASH Controller program data 2 register bit fields.
 *
 */
#define BOOT_FLASH_DATA_2_M								0xFFFFFFFFU
#define BOOT_FLASH_DATA_2_S								0U

/**
 * @brief FLASH Controller program data 3 register bit fields.
 *
 */
#define BOOT_FLASH_DATA_3_M								0xFFFFFFFFU
#define BOOT_FLASH_DATA_3_S								0U

/**
 * @brief FLASH Controller status register bit fields.
 *
 */
#define BOOT_FLASH_STATUS_EFC_BUSY_M					0x00000001U
#define BOOT_FLASH_STATUS_EFC_BUSY_S					0U
#define BOOT_FLASH_STATUS_WRITE_ERR_M					0x00000002U
#define BOOT_FLASH_STATUS_WRITE_ERR_S					1U
#define BOOT_FLASH_STATUS_DISABLE_REDUND_M				0x00000004U
#define BOOT_FLASH_STATUS_DISABLE_REDUND_S				2U
#define BOOT_FLASH_STATUS_DISCHARGED_STATUS_M			0x00000008U
#define BOOT_FLASH_STATUS_DISCHARGED_STATUS_S			3U
#define BOOT_FLASH_STATUS_BROWNOUT_STATUS_M				0x00000010U
#define BOOT_FLASH_STATUS_BROWNOUT_STATUS_S				4U
#define BOOT_FLASH_STATUS_WRITE_COMP_STATUS_M			0x00000020U
#define BOOT_FLASH_STATUS_WRITE_COMP_STATUS_S			5U
#define BOOT_FLASH_STATUS_SEC_DETECTED_M				0x00000040U
#define BOOT_FLASH_STATUS_SEC_DETECTED_S				6U
#define BOOT_FLASH_STATUS_DED_DETECTED_M				0x00000080U
#define BOOT_FLASH_STATUS_DED_DETECTED_S				7U
#define BOOT_FLASH_STATUS_SMW_ERR_STATUS_M				0x00000100U
#define BOOT_FLASH_STATUS_SMW_ERR_STATUS_S				8U
#define BOOT_FLASH_STATUS_SMW_LATEST_STATUS_M			0x0003FE00U
#define BOOT_FLASH_STATUS_SMW_LATEST_STATUS_S			9U
#define BOOT_FLASH_STATUS_SMW_LOOP_STATUS_M				0x0FFC0000U
#define BOOT_FLASH_STATUS_SMW_LOOP_STATUS_S				18U
#define BOOT_FLASH_STATUS_DEBUG_WR_ERR_M				0x10000000U
#define BOOT_FLASH_STATUS_DEBUG_WR_ERR_S				28U
#define BOOT_FLASH_STATUS_RDP_RD_ERR_M					0x20000000U
#define BOOT_FLASH_STATUS_RDP_RD_ERR_S					29U
#define BOOT_FLASH_STATUS_BUFFER_FULL_OR_HALT_M			0x40000000U
#define BOOT_FLASH_STATUS_BUFFER_FULL_OR_HALT_S			30U
#define BOOT_FLASH_STATUS_RWW_STATUS_M					0x80000000U
#define BOOT_FLASH_STATUS_RWW_STATUS_S					31U

/**
 * @brief FLASH Controller Redundant Address 0 register bit fields.
 *
 */
#define BOOT_FLASH_REDUN_ADDR_0_INDICATE_M				0x00000001U
#define BOOT_FLASH_REDUN_ADDR_0_INDICATE_S				0U
#define BOOT_FLASH_REDUN_ADDR_0_PAGE_ADDR_M				0x0007E000U
#define BOOT_FLASH_REDUN_ADDR_0_PAGE_ADDR_S				13U

/**
 * @brief FLASH Controller Redundant Address 1 register bit fields.
 *
 */
#define BOOT_FLASH_REDUN_ADDR_1_INDICATE_M				0x00000001U
#define BOOT_FLASH_REDUN_ADDR_1_INDICATE_S				0U
#define BOOT_FLASH_REDUN_ADDR_1_PAGE_ADDR_M				0x0007E000U
#define BOOT_FLASH_REDUN_ADDR_1_PAGE_ADDR_S				13U

/**
 * @brief FLASH Controller Single Error Counter register bit fields.
 *
 */
#define BOOT_FLASH_SINGLE_ERR_COUNTER_VALUE_M			0x0000FFFFU
#define BOOT_FLASH_SINGLE_ERR_COUNTER_VALUE_S			0U

/**
 * @brief FLASH Controller Double Error Counter register bit fields.
 *
 */
#define BOOT_FLASH_DOUBLE_ERR_COUNTER_VALUE_M			0x0000FFFFU
#define BOOT_FLASH_DOUBLE_ERR_COUNTER_VALUE_S			0U

/**
 * @brief FLASH Controller SMW Timer Option register bit fields.
 *
 */
#define BOOT_FLASH_SMW_TIMER_OPT_ONEUS_PULSE_CYCLES_M	0x000000FFU
#define BOOT_FLASH_SMW_TIMER_OPT_ONEUS_PULSE_CYCLES_S	0U
#define BOOT_FLASH_SMW_TIMER_OPT_PV_EV_READ_CYCLES_M	0x00001F00U
#define BOOT_FLASH_SMW_TIMER_OPT_PV_EV_READ_CYCLES_S	8U

/**
 * @brief FLASH Controller SMW Setting Option 1 register 0 bit fields.
 *
 */
#define BOOT_FLASH_SMW_SET_OPT1_REG0_FLASHUNUSED_M		0x000003FFU
#define BOOT_FLASH_SMW_SET_OPT1_REG0_FLASHUNUSED_S		0U
#define BOOT_FLASH_SMW_SET_OPT1_REG0_MV_CTRL_M			0x00FFC000U
#define BOOT_FLASH_SMW_SET_OPT1_REG0_MV_CTRL_S			14U
#define BOOT_FLASH_SMW_SET_OPT1_REG0_IPGM_CTRL_M		0x7F000000U
#define BOOT_FLASH_SMW_SET_OPT1_REG0_IPGM_CTRL_S		24U

/**
 * @brief FLASH Controller SMW Setting Option 1 register 1 bit fields.
 *
 */
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TERS_CTRL_M		0x00000007U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TERS_CTRL_S		0U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TPGM_CTRL_M		0x00000018U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TPGM_CTRL_S		3U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TNVS_CTRL_M		0x000000E0U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TNVS_CTRL_S		5U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TNVH_CTRL_M		0x00000700U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TNVH_CTRL_S		8U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TPGS_CTRL_M		0x00003800U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_TPGS_CTRL_S		11U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_MAX_ERASE_M		0x007FC000U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_MAX_ERASE_S		14U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_MAX_PROG_M			0x0F800000U
#define BOOT_FLASH_SMW_SET_OPT1_REG1_MAX_PROG_S			23U

/**
 * @brief FLASH Controller SMW Setting Option 2 register bit fields.
 *
 */
#define BOOT_FLASH_SMW_SET_OPT2_THVS_CTRL_M				0x00000007U
#define BOOT_FLASH_SMW_SET_OPT2_THVS_CTRL_S				0U
#define BOOT_FLASH_SMW_SET_OPT2_TRCV_CTRL_M				0x00000038U
#define BOOT_FLASH_SMW_SET_OPT2_TRCV_CTRL_S				3U
#define BOOT_FLASH_SMW_SET_OPT2_EXTRA_PULSE_M			0x000003C0U
#define BOOT_FLASH_SMW_SET_OPT2_EXTRA_PULSE_S			6U
#define BOOT_FLASH_SMW_SET_OPT2_WHV_COUNTER_M			0x0003FC00U
#define BOOT_FLASH_SMW_SET_OPT2_WHV_COUNTER_S			10U
#define BOOT_FLASH_SMW_SET_OPT2_POST_TERS_M				0x001C0000U
#define BOOT_FLASH_SMW_SET_OPT2_POST_TERS_S				18U
#define BOOT_FLASH_SMW_SET_OPT2_POST_TPGM_M				0x00600000U
#define BOOT_FLASH_SMW_SET_OPT2_POST_TPGM_S				21U
#define BOOT_FLASH_SMW_SET_OPT2_VERIFY_OPT_M			0x01800000U
#define BOOT_FLASH_SMW_SET_OPT2_VERIFY_OPT_S			23U
#define BOOT_FLASH_SMW_SET_OPT2_TPGM_OPT_M				0x06000000U
#define BOOT_FLASH_SMW_SET_OPT2_TPGM_OPT_S				25U
#define BOOT_FLASH_SMW_SET_OPT2_MASK0_OPT_M				0x08000000U
#define BOOT_FLASH_SMW_SET_OPT2_MASK0_OPT_S				27U
#define BOOT_FLASH_SMW_SET_OPT2_DISABLE_PRER_OPT_M		0x10000000U
#define BOOT_FLASH_SMW_SET_OPT2_DISABLE_PRER_OPT_S		28U

/**
 * @brief FLASH Controller SMW Setting Option 3 register bit fields.
 *
 */
#define BOOT_FLASH_SMW_SET_OPT3_HWM_WHV_COUNTER_M		0x000000FFU
#define BOOT_FLASH_SMW_SET_OPT3_HWM_WHV_COUNTER_S		0U
#define BOOT_FLASH_SMW_SET_OPT3_HEM_MAX_ERASE_M			0x0001FF00U
#define BOOT_FLASH_SMW_SET_OPT3_HEM_MAX_ERASE_S			8U

/**
 * @brief FLASH Controller SMW SMP WHV Option 0 register bit fields.
 *
 */
#define BOOT_FLASH_SMW_SMP_WHV_OPT0_M					0xFFFFFFFFU
#define BOOT_FLASH_SMW_SMP_WHV_OPT0_S					0U

/**
 * @brief FLASH Controller SMW SMP WHV Option 1 register bit fields.
 *
 */
#define BOOT_FLASH_SMW_SMP_WHV_OPT1_M					0xFFFFFFFFU
#define BOOT_FLASH_SMW_SMP_WHV_OPT1_S					0U

/**
 * @brief FLASH Controller SMW SME WHV Option 0 register bit fields.
 *
 */
#define BOOT_FLASH_SMW_SME_OPT0_M						0xFFFFFFFFU
#define BOOT_FLASH_SMW_SME_OPT0_S						0U

/**
 * @brief FLASH Controller SMW SME WHV Option 1 register bit fields.
 *
 */
#define BOOT_FLASH_SMW_SME_OPT1_M						0xFFFFFFFFU
#define BOOT_FLASH_SMW_SME_OPT1_S						0U

/**
 * @brief FLASH Controller FLASH Misc Config register bit fields.
 *
 */
#define BOOT_FLASH_MISC_CFG_WHV_CFG_M					0x0000000FU
#define BOOT_FLASH_MISC_CFG_WHV_CFG_S					0U
#define BOOT_FLASH_MISC_CFG_WMV_CFG_M					0x00000070U
#define BOOT_FLASH_MISC_CFG_WMV_CFG_S					4U
#define BOOT_FLASH_MISC_CFG_WIPGM_CFG_M					0x00000180U
#define BOOT_FLASH_MISC_CFG_WIPGM_CFG_S					7U
#define BOOT_FLASH_MISC_CFG_SW_ELETRICAL_CFG_EN_M		0x00000200U
#define BOOT_FLASH_MISC_CFG_SW_ELETRICAL_CFG_EN_S		9U
#define BOOT_FLASH_MISC_CFG_FLASH_MISC_M				0xFFFFFC00U
#define BOOT_FLASH_MISC_CFG_FLASH_MISC_S				10U

/**
 * @brief FLASH Controller Prefetch Config register bit fields.
 *
 */
#define BOOT_FLASH_PREFETCH_CFG_PREFETCH_ON_M			0x00000001U
#define BOOT_FLASH_PREFETCH_CFG_PREFETCH_ON_S			0U
#define BOOT_FLASH_PREFETCH_CFG_BUFFER_FLUSH_M			0x00000002U
#define BOOT_FLASH_PREFETCH_CFG_BUFFER_FLUSH_S			1U
#define BOOT_FLASH_PREFETCH_CFG_SKIP_COOL_M				0x00000004U
#define BOOT_FLASH_PREFETCH_CFG_SKIP_COOL_S				2U

/**
 * @brief FLASH Controller Info Block Write Protect register bit fields.
 *
 */
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_PRG_PROT0_M	0x00000001U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_PRG_PROT0_S	0U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_PRG_PROT1_M	0x00000002U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_PRG_PROT1_S	1U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_PRG_PROT2_M	0x00000004U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_PRG_PROT2_S	2U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_PRG_PROT3_M	0x00000008U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_PRG_PROT3_S	3U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_ERS_PROT0_M	0x00010000U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_ERS_PROT0_S	16U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_ERS_PROT1_M	0x00020000U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_ERS_PROT1_S	17U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_ERS_PROT2_M	0x00040000U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_ERS_PROT2_S	18U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_ERS_PROT3_M	0x00080000U
#define BOOT_FLASH_INFO_PROTECT_CFG_INFO_ERS_PROT3_S	19U

/**
 * @brief FLASH Controller Option Unlock register bit fields.
 *
 */
#define BOOT_FLASH_OPT_UNLOCK_M							0xFFFFFFFFU
#define BOOT_FLASH_OPT_UNLOCK_S							0U

/**
 * @brief FLASH Controller Option Lock register bit fields.
 *
 */
#define BOOT_FLASH_OPT_LOCK_M							0x00000001U
#define BOOT_FLASH_OPT_LOCK_S							0U

/**
 * @brief FLASH Controller Mode Ctrl register bit fields.
 *
 */
#define BOOT_FLASH_OPT_MODE_CTRL_OPT_MODE_START_M		0x00000001U
#define BOOT_FLASH_OPT_MODE_CTRL_OPT_MODE_START_S		0U
#define BOOT_FLASH_OPT_MODE_CTRL_OPT_LAUNCH_M			0x00000002U
#define BOOT_FLASH_OPT_MODE_CTRL_OPT_LAUNCH_S			1U

/**
 * @brief FLASH Controller OPT Mode Status register bit fields.
 *
 */
#define BOOT_FLASH_OPT_MODE_STATUS_OPT_MODE_STATE_M		0x00000001U
#define BOOT_FLASH_OPT_MODE_STATUS_OPT_MODE_STATE_S		0U
#define BOOT_FLASH_OPT_MODE_STATUS_OPT_MODE_ERROR_M		0x00000002U
#define BOOT_FLASH_OPT_MODE_STATUS_OPT_MODE_ERROR_S		1U
#define BOOT_FLASH_OPT_MODE_STATUS_OPT_LAUNCH_BUSY_M	0x00000004U
#define BOOT_FLASH_OPT_MODE_STATUS_OPT_LAUNCH_BUSY_S	2U

/**
 * @brief FLASH Controller User Option 0 register bit fields.
 *
 */
#define BOOT_FLASH_SET_USER_OPT_0_M						0xFFFFFFFFU
#define BOOT_FLASH_SET_USER_OPT_0_S						0U

/**
 * @brief FLASH Controller User Option 1 register bit fields.
 *
 */
#define BOOT_FLASH_SET_USER_OPT_1_M						0x00FFFFFFU
#define BOOT_FLASH_SET_USER_OPT_1_S						0U

/**
 * @brief FLASH Controller Rdp Set register bit fields.
 *
 */
#define BOOT_FLASH_SET_RDP_M							0x000000FFU
#define BOOT_FLASH_SET_RDP_S							0U

/**
 * @brief FLASH Controller Pcrop Low Set register bit fields.
 *
 */
#define BOOT_FLASH_SET_PCROP_L_M						0xFFFFFFFFU
#define BOOT_FLASH_SET_PCROP_L_S						0U

/**
 * @brief FLASH Controller Pcrop High Set register bit fields.
 *
 */
#define BOOT_FLASH_SET_PCROP_H_M						0xFFFFFFFFU
#define BOOT_FLASH_SET_PCROP_H_S						0U

/**
 * @brief FLASH Controller Wrp Low Set register bit fields.
 *
 */
#define BOOT_FLASH_SET_WRP_L_M							0xFFFFFFFFU
#define BOOT_FLASH_SET_WRP_L_S							0U

/**
 * @brief FLASH Controller Wrp High Set register bit fields.
 *
 */
#define BOOT_FLASH_SET_WRP_H_M							0xFFFFFFFFU
#define BOOT_FLASH_SET_WRP_H_S							0U

/**
 * @brief FLASH Controller Section Size Set register bit fields.
 *
 */
#define BOOT_FLASH_SET_SEC_SIZE_SET_SEC_SIZE1_M			0x000000FFU
#define BOOT_FLASH_SET_SEC_SIZE_SET_SEC_SIZE1_S			0U
#define BOOT_FLASH_SET_SEC_SIZE_SET_SEC_SIZE2_M			0x0000FF00U
#define BOOT_FLASH_SET_SEC_SIZE_SET_SEC_SIZE2_S			8U

/**
 * @brief FLASH Controller Boot Lock Set register bit fields.
 *
 */
#define BOOT_FLASH_SET_BOOT_LOCK_SET_BOOT_LOCK_M		0x00000001U
#define BOOT_FLASH_SET_BOOT_LOCK_SET_BOOT_LOCK_S		0U

/**
 * @brief FLASH Controller Pcrop Rdp Set register bit fields.
 *
 */
#define BOOT_FLASH_SET_PCROP_RDP_SET_PCROP_RDP_M		0x00000001U
#define BOOT_FLASH_SET_PCROP_RDP_SET_PCROP_RDP_S		0U

/**
 * @brief FLASH Controller Section Debug Enable Set register bit fields.
 *
 */
#define BOOT_FLASH_SET_SEC_DEBUG_EN_SET_SEC_DEBUG_EN_M	0x00000001U
#define BOOT_FLASH_SET_SEC_DEBUG_EN_SET_SEC_DEBUG_EN_S	0U

/**
 * @brief FLASH Controller unlock register write protect sequence.
 *
 */
#define BOOT_FLASH_UNLOCK_1_CODE0						0x7123U
#define BOOT_FLASH_UNLOCK_2_CODE1						0x812aU
#define BOOT_FLASH_UNLOCK_1_CODE2						0xbee1U

/**
 * @brief Flash write clear status flag.
 *
 * @param BOOT_FLASH_WRITE_ERROR write flash error flag.
 * @param BOOT_FLASH_DISCHARGED discharge flash flag.
 * @param BOOT_FLASH_WRITE_COMPLETED write or erase flash completed flag.
 * @param BOOT_FLASH_SEC_ERROR_CORRECTED single bit error flag.
 * @param BOOT_FLASH_DEC_ERROR_UNCORRECTED multi bit error flag.
 * @param BOOT_FLASH_DEBUG_WRITE_ERROR debug write flash error flag.
 * @param BOOT_FLASH_RDP_READ_ERROR read RDP error flag.
 * @param BOOT_FLASH_ALL_STATUS_MASK all status mask.
 */
#define BOOT_FLASH_WRITE_ERROR							0x00000002U
#define BOOT_FLASH_DISCHARGED							0x00000008U
#define BOOT_FLASH_WRITE_COMPLETED						0x00000020U
#define BOOT_FLASH_SEC_ERROR_CORRECTED					0x00000040U
#define BOOT_FLASH_DEC_ERROR_UNCORRECTED				0x00000080U
#define BOOT_FLASH_DEBUG_WRITE_ERROR					0x10000000U
#define BOOT_FLASH_RDP_READ_ERROR						0x20000000U
#define BOOT_FLASH_ALL_STATUS_MASK						(										\
														BOOT_FLASH_WRITE_ERROR				|	\
														BOOT_FLASH_DISCHARGED				|	\
														BOOT_FLASH_WRITE_COMPLETED			|	\
														BOOT_FLASH_SEC_ERROR_CORRECTED		|	\
														BOOT_FLASH_DEC_ERROR_UNCORRECTED	|	\
														BOOT_FLASH_DEBUG_WRITE_ERROR		|	\
														BOOT_FLASH_RDP_READ_ERROR				\
														)

/**
 * @brief FLASH Controller status codes.
 *
 * @param BOOT_FLASH_STAT_SUCCESS Successfully executed flash operation.
 * @param BOOT_FLASH_STAT_ERR Flash operation failed.
 */
typedef enum {
	BOOT_FLASH_STAT_SUCCESS				= 0U,
	BOOT_FLASH_STAT_ERR					= 1U,
} boot_flash_err_t;

/**
 * @brief FLASH Controller EFC Command.
 *
 * @param BOOT_FLASH_EFC_CMD_MAIN_MAS_ERASE Erase data in the main area except for the info area.
 * @param BOOT_FLASH_EFC_CMD_ERASE_PAGE Erase a page.
 * @param BOOT_FLASH_EFC_CMD_PROGRAM_START Start programming.
 * @param BOOT_FLASH_EFC_CMD_ERASE_REF_CELL
 * @param BOOT_FLASH_EFC_CMD_RECALL_TRIM_CODE
 * @param BOOT_FLASH_EFC_CMD_PROGRAM_END End programming.
 */
typedef enum {
	BOOT_FLASH_EFC_CMD_MAIN_MAS_ERASE	= 0x8U,
	BOOT_FLASH_EFC_CMD_ERASE_PAGE		= 0x10U,
	BOOT_FLASH_EFC_CMD_PROGRAM_START	= 0x20U,
	BOOT_FLASH_EFC_CMD_ERASE_REF_CELL	= 0x40U,
	BOOT_FLASH_EFC_CMD_RECALL_TRIM_CODE	= 0x80U,
	BOOT_FLASH_EFC_CMD_PROGRAM_END		= 0x100U,
} boot_flash_efc_cmd_t;

/**
 * @brief User Option bytes status.
 *
 * @param boot_pin_sel Boot pin selection.
 * @param boot_def Boot pin default value.
 * @param boot_sram_reset SRAM auto clear 0.
 * @param boot_ilm_dlm_shadow enable/disable ILM/DLM shadow.
 * @param boot_dual_bank_mode enable/disable dual bank mode.
 * @param boot_pin_cfg Boot pin lock configuration.
 * @param boot_pin_lock_sel Boot pin lock selection.
 * @param boot_test_pin_mult_0 Boot test pin multiplexing.
 * @param boot_test_pin_mult_1 Boot test pin multiplexing.
 */
typedef struct boot_flash_user_opt_info {
	uint32_t boot_pin_sel;

	union {
		struct {
			uint32_t boot_def				:	8;
			uint32_t boot_sram_reset		:	1;
			uint32_t boot_ilm_dlm_shadow	:	1;
			uint32_t boot_dual_bank_mode	:	1;
			uint32_t reserved_0				:	1;
			uint32_t boot_pin_lock_cfg		:	1;
			uint32_t boot_pin_lock_sel		:	1;
			uint32_t boot_test_pin_mult_0	:	1;
			uint32_t boot_test_pin_mult_1	:	1;
			uint32_t reserved_1				:	16;
		};
		uint32_t user_opt_1;
	};
} boot_flash_user_opt_info_t;

/**
 * @brief Obtain address information based on flash address.
 *
 * @param flash_efc_addr The address of flash efc.
 * @param flash_page Flash page.
 * @param bank_mode Flash bank mode(0 - single bank, 1 - dual bank).
 * @param address actual address of flash controller.
 */
typedef struct boot_flash_addr_info {
	uint32_t flash_efc_addr;
	uint32_t flash_page;
	uint32_t bank_mode;
	uint32_t address;
} boot_flash_addr_info_t;

/**
 * @brief Obtain flash status.
 *
 * @param efc_busy Flash efc busy status.
 * @param write_error Flash efc write error status.
 * @param disable_redund Disable redundancy.
 * @param discharge_status discharge status. 1: discharged 0: normal
 * @param brownout_status brownout status. 1: Brown out detected 0: Power normal
 * @param write_comp_status write complete status. 1: write complete 0: not complete
 * @param sec_detected Indicates if an error has been corrected.
 * @param ded_detected Indicates if an uncorrectable error has been detecteds.
 * @param smw_err_status status from smw, 1 to indicate smart-write failure. Debug usage.
 * @param smw_latest_status status from smw, indicate the value of {WIPGM[1:0],WMV[2:0],WHV[3:0]}
 * in the last loop of smw operation. Debug usage.
 * @param smw_loop_status status from smw, indicate the loop count to complete current smw
 * operation. Debug usage.
 * @param debug_wr_err Write operations that do not meet the protection level during the debug phase
 * @param rdp_rd_err rdp error, efc 2 invalid.
 * @param buffer_full_or_halt efc buffer full or halt status.
 * @param rww_status read while writing.
 */
typedef union boot_flash_efc_status {
	struct {
		uint32_t efc_busy				:	1;
		uint32_t write_error			:	1;
		uint32_t disable_redund			:	1;
		uint32_t discharge_status		:	1;
		uint32_t brownout_status		:	1;
		uint32_t write_comp_status		:	1;
		uint32_t sec_detected			:	1;
		uint32_t ded_detected			:	1;
		uint32_t smw_err_status			:	1;
		uint32_t smw_latest_status		:	9;
		uint32_t smw_loop_status		:	10;
		uint32_t debug_wr_err			:	1;
		uint32_t rdp_rd_err				:	1;
		uint32_t buffer_full_or_halt	:	1;
		uint32_t rww_status				:	1;
	};

	uint32_t efc_status;
} boot_flash_efc_status_t;

/**
 * @brief Set flash read wait timing.
 *
 * @param base Base address of flash efc.
 * @param read_cycle Read wait cycle.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_set_read_wait_timing(uintptr_t base, uint32_t read_cycle);

/**
 * @brief Set flash write wait timing.
 *
 * @param base Base address of flash efc.
 * @param write_cycle Write wait cycle.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_set_prog_cycles(uintptr_t base, uint32_t prog_cycle);

/**
 * @brief Set flash eight us cycles.
 *
 * @param base Base address of flash efc.
 * @param eight_us_cycles Eight us cycles.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_set_eight_us_cycles(uintptr_t base, uint32_t eight_us_cycles);


/**
 * @brief Set flash erase cycles.
 *
 * @param base Base address of flash efc.
 * @param erase_cycles Erase cycles.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_set_erc_erase_cycles(uintptr_t base, uint32_t erase_cycles);

/**
 * @brief Set flash two us cycles.
 *
 * @param base Base address of flash efc.
 * @param two_us_cycles Two us cycles.
 * @return if success, return 0, else return error code. 
 */
BOOT_FLASH_SECTION int boot_flash_set_two_us_cycles(uintptr_t base, uint32_t two_us_cycles);

/**
 * @brief Set flash test hold cycles.
 *
 * @param base Base address of flash efc.
 * @param test_hold_cycles test mode hold cycles.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_set_test_hold_cycles(uintptr_t base, uint32_t test_hold_cycles);

/**
 * @brief Set flash hundred ns cycles.
 *
 * @param base Base address of flash efc.
 * @param hundred_ns_cycles hundred ns cycles.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_set_hundred_ns_cycles(uintptr_t base, uint32_t hundred_ns_cycles);

/**
 * @brief Set flash ten ns cycles.
 *
 * @param base Base address of flash efc.
 * @param ten_ns_cycles hundred ns cycles.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_set_ten_ns_cycles(uintptr_t base, uint32_t ten_ns_cycles);

/**
 * @brief Enable flash sleep mode.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_enable_slp_mode(uintptr_t base);

/**
 * @brief Disable flash sleep mode.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_disable_slp_mode(uintptr_t base);

/**
 * @brief Enable flash ECC.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_enable_ecc(uintptr_t base);

/**
 * @brief Disable flash ECC.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_disable_ecc(uintptr_t base);

/**
 * @brief Enable flash fast program mode.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_enable_fast_program_mode(uintptr_t base);

/**
 * @brief Disable flash fast program mode.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_disable_fast_program_mode(uintptr_t base);

/**
 * @brief Enable flash error indicate mode.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_enable_err_indicate(uintptr_t base);

/**
 * @brief Disable flash error indicate mode.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_disable_err_indicate(uintptr_t base);

/**
 * @brief Enable flash program/erase complete interrupt.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_enable_program_erase_complete_interrupt(uintptr_t base);

/**
 * @brief Disable flash program/erase complete interrupt.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_disable_program_erase_complete_interrupt(uintptr_t base);

/**
 * @brief Enable flash enable ecc correction interrupt.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_enable_ecc_correction_interrupt(uintptr_t base);

/**
 * @brief Disable flash disable ecc correction interrupt.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_disable_ecc_correction_interrupt(uintptr_t base);

/**
 * @brief Enable flash ecc detected interrupt.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_enable_ecc_detected_interrupt(uintptr_t base);

/**
 * @brief Disable flash ecc detected interrupt.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_disable_ecc_detected_interrupt(uintptr_t base);

/**
 * @brief Enable flash register write protect.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_enable_register_write_protect(uintptr_t base);

/**
 * @brief Disable flash register write protect.
 *
 * @param base Base address of flash efc.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_disable_register_write_protect(uintptr_t base);

/**
 * @brief Get user option info.
 *
 * @param base Base address of flash efc.
 * @param info User option info.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_get_user_opt_infomation(uintptr_t base,
														boot_flash_user_opt_info_t *info);

/**
 * @brief Get address info.
 *
 * @param address The address of flash.
 * @param efc_base The base address of flash efc.
 * @param sec_num The number of flash sector.
 * @param dual_bank_mode The dual bank mode of flash.
 */
BOOT_FLASH_SECTION void boot_flash_get_addr_info(uint32_t address, boot_flash_addr_info_t *info);

/**
 * @brief Set flash efc command.
 *
 * @param base Base address of flash efc.
 * @param cmd EFC command.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_config_efc_cmd(uintptr_t base, boot_flash_efc_cmd_t cmd);

/**
 * @brief Erase flash page by flash address.
 *
 * @param address Flash address.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_erase_main_flash_sector(uint32_t address);

/**
 * @brief Write data to flash.
 * @note Write a minimum of 16 bytes each time.
 *
 * @param address address of flash, The incoming address must be
 * aligned according to 16 bytes.
 * @param buf data buffer.
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_write_main_flash(uint32_t address, uint8_t *buf);

/**
 * @brief Get the flash efc status.
 *
 * @param base Base address of flash efc.
 * @param status flash efc status( @note boot_flash_efc_status_t).
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_get_efc_status(uintptr_t base, boot_flash_efc_status_t *status);

/**
 * @brief clear flash efc status.
 *
 * @param base Base address of flash efc.
 * @param status flash efc status.
 * @e BOOT_FLASH_WRITE_ERROR write flash error flag.
 * @e BOOT_FLASH_DISCHARGED discharge flash flag.
 * @e BOOT_FLASH_WRITE_COMPLETED write or erase flash completed flag.
 * @e BOOT_FLASH_SEC_ERROR_CORRECTED single bit error flag.
 * @e BOOT_FLASH_DEC_ERROR_UNCORRECTED multi bit error flag.
 * @e BOOT_FLASH_DEBUG_WRITE_ERROR debug write flash error flag.
 * @e BOOT_FLASH_RDP_READ_ERROR read RDP error flag.
 * @e BOOT_FLASH_ALL_STATUS_MASK all status mask.
 *
 * @return if success, return 0, else return error code.
 */
BOOT_FLASH_SECTION int boot_flash_clear_efc_status(uintptr_t base, uint32_t status);

#endif	/* (GS32F00xx == 0x3000) */

#ifdef __cplusplus
}
#endif

#endif
