/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __BOOT_UART_V3_0_H__
#define __BOOT_UART_V3_0_H__

#ifdef __cplusplus
extern "C"{
#endif

#if (GS32F00xx == 0x3000)

#include "boot_hw_type.h"
#include "inc/hw_uart.h"

#define BOOT_UART_SEC					__attribute__((section(".boot_uart_section")))

/**
 * @brief UART function configuration parameters.
 *
 */
#define BOOT_UART_CONFIG_WLEN_MASK		0x0007U		/* Mask for extracting word length */
#define BOOT_UART_CONFIG_WLEN_8			0x0007U		/* 8 bit data */
#define BOOT_UART_CONFIG_WLEN_7			0x0006U		/* 7 bit data */
#define BOOT_UART_CONFIG_WLEN_6			0x0005U		/* 6 bit data */
#define BOOT_UART_CONFIG_WLEN_5			0x0004U		/* 5 bit data */
#define BOOT_UART_CONFIG_WLEN_4			0x0003U		/* 4 bit data */
#define BOOT_UART_CONFIG_WLEN_3			0x0002U		/* 3 bit data */
#define BOOT_UART_CONFIG_WLEN_2			0x0001U		/* 2 bit data */
#define BOOT_UART_CONFIG_WLEN_1			0x0000U		/* 1 bit data */
#define BOOT_UART_CONFIG_STOP_MASK		0x0080U		/* Mask for extracting stop bits */
#define BOOT_UART_CONFIG_STOP_ONE		0x0000U		/* One stop bit */
#define BOOT_UART_CONFIG_STOP_TWO		0x0080U		/* Two stop bits */
#define BOOT_UART_CONFIG_PAR_MASK		0x0060U		/* Parity Mask */
#define BOOT_UART_CONFIG_PAR_NONE		0x0000U		/* No parity */
#define BOOT_UART_CONFIG_PAR_EVEN		0x0060U		/* Even parity */
#define BOOT_UART_CONFIG_PAR_ODD		0x0020U		/* Odd parity */

/**
 * @brief UART interrupt flag bit.
 *
 */
#define BOOT_UART_INT_RXERR				0x01U		/* RXERR interrupt */
#define BOOT_UART_INT_RXRDY_BRKDT		0x02U		/* RXRDY interrupt */
#define BOOT_UART_INT_TXRDY				0x04U		/* TXRDY interrupt */
#define BOOT_UART_INT_TXFF				0x08U		/* TX FIFO level interrupt */
#define BOOT_UART_INT_RXFF				0x10U		/* RX FIFO level interrupt */
#define BOOT_UART_INT_FE				0x20U		/* Frame Error */
#define BOOT_UART_INT_OE				0x40U		/* Overrun Error */
#define BOOT_UART_INT_PE				0x80U		/* Parity Error */

typedef enum {
	BOOT_UART_FIFO_RX0			= 0x0000U,			/* Receive interrupt empty */
	BOOT_UART_FIFO_RX1			= 0x0001U,			/* Receive interrupt 1/16 full */
	BOOT_UART_FIFO_RX2			= 0x0002U,			/* Receive interrupt 2/16 full */
	BOOT_UART_FIFO_RX3			= 0x0003U,			/* Receive interrupt 3/16 full */
	BOOT_UART_FIFO_RX4			= 0x0004U,			/* Receive interrupt 4/16 full */
	BOOT_UART_FIFO_RX5			= 0x0005U,			/* Receive interrupt 5/16 full */
	BOOT_UART_FIFO_RX6			= 0x0006U,			/* Receive interrupt 6/16 full */
	BOOT_UART_FIFO_RX7			= 0x0007U,			/* Receive interrupt 7/16 full */
	BOOT_UART_FIFO_RX8			= 0x0008U,			/* Receive interrupt 8/16 full */
	BOOT_UART_FIFO_RX9			= 0x0009U,			/* Receive interrupt 9/16 full */
	BOOT_UART_FIFO_RX10			= 0x000AU,			/* Receive interrupt 10/16 full */
	BOOT_UART_FIFO_RX11			= 0x000BU,			/* Receive interrupt 11/16 full */
	BOOT_UART_FIFO_RX12			= 0x000CU,			/* Receive interrupt 12/16 full */
	BOOT_UART_FIFO_RX13			= 0x000DU,			/* Receive interrupt 13/16 full */
	BOOT_UART_FIFO_RX14			= 0x000EU,			/* Receive interrupt 14/16 full */
	BOOT_UART_FIFO_RX15			= 0x000FU,			/* Receive interrupt 15/16 full */
	BOOT_UART_FIFO_RX16			= 0x0010U			/* Receive interrupt full */
} BOOT_UART_RxFIFOLevel_t;

typedef enum {
	BOOT_UART_FIFO_TX0			= 0x0000U,			/* Transmit interrupt empty */
	BOOT_UART_FIFO_TX1			= 0x0001U,			/* Transmit interrupt 1/16 full */
	BOOT_UART_FIFO_TX2			= 0x0002U,			/* Transmit interrupt 2/16 full */
	BOOT_UART_FIFO_TX3			= 0x0003U,			/* Transmit interrupt 3/16 full */
	BOOT_UART_FIFO_TX4			= 0x0004U,			/* Transmit interrupt 4/16 full */
	BOOT_UART_FIFO_TX5			= 0x0005U,			/* Transmit interrupt 5/16 full */
	BOOT_UART_FIFO_TX6			= 0x0006U,			/* Transmit interrupt 6/16 full */
	BOOT_UART_FIFO_TX7			= 0x0007U,			/* Transmit interrupt 7/16 full */
	BOOT_UART_FIFO_TX8			= 0x0008U,			/* Transmit interrupt 8/16 full */
	BOOT_UART_FIFO_TX9			= 0x0009U,			/* Transmit interrupt 9/16 full */
	BOOT_UART_FIFO_TX10			= 0x000AU,			/* Transmit interrupt 10/16 full */
	BOOT_UART_FIFO_TX11			= 0x000BU,			/* Transmit interrupt 11/16 full */
	BOOT_UART_FIFO_TX12			= 0x000CU,			/* Transmit interrupt 12/16 full */
	BOOT_UART_FIFO_TX13			= 0x000DU,			/* Transmit interrupt 13/16 full */
	BOOT_UART_FIFO_TX14			= 0x000EU,			/* Transmit interrupt 14/16 full */
	BOOT_UART_FIFO_TX15			= 0x000FU,			/* Transmit interrupt 15/16 full */
	BOOT_UART_FIFO_TX16			= 0x0010U 			/* Transmit interrupt full */
} BOOT_UART_TxFIFOLevel_t;

BOOT_UART_SEC void boot_uart_reset(uint32_t base);

BOOT_UART_SEC void boot_uart_disable_module(uint32_t base);

BOOT_UART_SEC void boot_uart_enable_module(uint32_t base);

BOOT_UART_SEC void boot_uart_set_config(uint32_t base, uint32_t lspclkHz, uint32_t baud, uint32_t config);

BOOT_UART_SEC void boot_uart_reset_channel(uint32_t base);

BOOT_UART_SEC void boot_uart_enable_interrupt(uint32_t base, uint32_t intFlags);

BOOT_UART_SEC void boot_uart_disable_interrupt(uint32_t base, uint32_t intFlags);

BOOT_UART_SEC uint32_t boot_uart_get_interrupt_status(uint32_t base);

BOOT_UART_SEC void boot_uart_clear_interrupt_status(uint32_t base, uint32_t intFlgs);

BOOT_UART_SEC void boot_uart_enable_fifo_mode(uint32_t base);

BOOT_UART_SEC void boot_uart_disable_fifo_mode(uint32_t base);

BOOT_UART_SEC void boot_uart_set_fifo_level(uint32_t base, BOOT_UART_TxFIFOLevel_t txLevel, BOOT_UART_RxFIFOLevel_t rxLevel);

BOOT_UART_SEC BOOT_UART_RxFIFOLevel_t boot_uart_get_rxfifo_status(uint32_t base);

BOOT_UART_SEC BOOT_UART_TxFIFOLevel_t boot_uart_get_txfifo_status(uint32_t base);

BOOT_UART_SEC uint16_t boot_uart_get_tx_status(uint32_t base);

BOOT_UART_SEC uint16_t boot_uart_get_rx_status(uint32_t base);

BOOT_UART_SEC bool boot_uart_transmit_msg_fifo(uint32_t base, uint8_t msg);

BOOT_UART_SEC bool boot_uart_transmit_msg(uint32_t base, uint8_t msg);

BOOT_UART_SEC bool boot_uart_receive_msg_fifo(uint32_t base, uint8_t *msg);

BOOT_UART_SEC bool boot_uart_receive_msg(uint32_t base, uint8_t *msg);

#endif

#ifdef __cplusplus
}
#endif

#endif
