/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

OUTPUT_ARCH( "riscv" )
PROVIDE(__STACK_SIZE = 2K);
PROVIDE(__HEAP_SIZE = 1K);
PROVIDE(__SMP_CPU_CNT = 1);

ENTRY( _start )

MEMORY
{
	app_flash	(rxa!w) : ORIGIN = 0x08004000, LENGTH = 240K
	ilm			(wxa!r) : ORIGIN = 0x20100000, LENGTH = 128K
	dlm			(wxa!r) : ORIGIN = 0x20120000, LENGTH = 32K - __STACK_SIZE - __HEAP_SIZE
	dlm_heap	(wxa!r) : ORIGIN = 0x20120000 + 32K - __HEAP_SIZE - __STACK_SIZE, LENGTH = __HEAP_SIZE
	dlm_stk		(wxa!r) : ORIGIN = 0x20120000 + 32K - __STACK_SIZE, LENGTH = __STACK_SIZE
	gsram		(wxa!r) : ORIGIN = 0x20038000, LENGTH = 16K
}

REGION_ALIAS("APP_FLASH", app_flash)
REGION_ALIAS("ILM", ilm)
REGION_ALIAS("DLM", dlm)
REGION_ALIAS("GSRAM", gsram)
REGION_ALIAS("HEAP", dlm_heap)
REGION_ALIAS("STACK", dlm_stk)

SECTIONS
{
	/* To provide symbol __STACK_SIZE, __HEAP_SIZE and __SMP_CPU_CNT */
	/* PROVIDE(__STACK_SIZE = 2K); */
	__TOT_STACK_SIZE = __STACK_SIZE * __SMP_CPU_CNT;

	.rela.init : { *(.rela.init*) }
	.rela.sdata : { *(.rela.sdata*) }	/* add to avoid orphan sections warnings */
	.rela.data : { *(.rela.data*) }		/* add to avoid orphan sections warnings */
	.rela.text.__libc_init_array : { *(.rela.text.__libc_init_array*) }
	.rela.text.__libc_fini_array : { *(.rela.text.__libc_fini_array*) }
	.rela.text.__register_exitproc : { *(.rela.text.__register_exitproc*) }

	.bank_info :
	{
		KEEP (*(SORT_NONE(.boot_app_info_section)))
		. = ALIGN(16);
	} >APP_FLASH AT>APP_FLASH

	.init :
	{
		__app_start_address = .;
		*(.vtable)
		*(.vtable_s)
		KEEP (*(SORT_NONE(.init)))
		. = ALIGN(16);
	} >APP_FLASH AT>APP_FLASH

	.ilm_vector (NOLOAD) :
	{
		. = ALIGN(512);
		__vector_remap_start__ = .;
		. += 260*4;
		__vector_remap_end__ = .;
	} >ILM AT>ILM

	PROVIDE( __vector_remap_start = __vector_remap_start__ );
	PROVIDE( __vector_remap_end = __vector_remap_end__ );

	.text :
	{
		*(.text.unlikely .text.unlikely.*)
		*(.text.startup .text.startup.*)
		*(.text .text.*)
		*(.gnu.linkonce.t.*)
		*(.boot_text_section)
		*(.boot_fdp_rt_section)
		*(.boot_gpio_section)
		*(.boot_uart_section)
		*(.boot_flash_sec)
		. = ALIGN(16);
	} >ILM AT>APP_FLASH

	.fini :
	{
		KEEP (*(SORT_NONE(.fini)))
		. = ALIGN(16);
	} >APP_FLASH AT>APP_FLASH

	.preinit_array :
	{
		PROVIDE_HIDDEN (__preinit_array_start = .);
		KEEP (*(.preinit_array))
		PROVIDE_HIDDEN (__preinit_array_end = .);
		. = ALIGN(16);
	} >APP_FLASH AT>APP_FLASH

	.init_array :
	{
		PROVIDE_HIDDEN (__init_array_start = .);
		KEEP (*(SORT_BY_INIT_PRIORITY(.init_array.*) SORT_BY_INIT_PRIORITY(.ctors.*)))
		KEEP (*(.init_array EXCLUDE_FILE (*crtbegin.o *crtbegin?.o *crtend.o *crtend?.o ) .ctors))
		PROVIDE_HIDDEN (__init_array_end = .);
		. = ALIGN(16);
	} >APP_FLASH AT>APP_FLASH

	.fini_array :
	{
		PROVIDE_HIDDEN (__fini_array_start = .);
		KEEP (*(SORT_BY_INIT_PRIORITY(.fini_array.*) SORT_BY_INIT_PRIORITY(.dtors.*)))
		KEEP (*(.fini_array EXCLUDE_FILE (*crtbegin.o *crtbegin?.o *crtend.o *crtend?.o ) .dtors))
		PROVIDE_HIDDEN (__fini_array_end = .);
		. = ALIGN(16);
	} >APP_FLASH AT>APP_FLASH

	.const.table :
	{
		. = ALIGN(4);
		__cli_cmds_table_start__ = .;
		KEEP(*(.cli_cmds))
		__cli_cmds_table_end__ = .;

		. = ALIGN(4);
		KEEP(*(.rom_data))
		KEEP(*(.rom_data.*))
		. = ALIGN(16);
	}>APP_FLASH AT>APP_FLASH

	.copy.table :
	{
		. = ALIGN(4);
		__copy_table_start__ = .;

		LONG (LOADADDR(.data))
		LONG (ADDR(.data))
		LONG (SIZEOF(.data) / 4)

		LONG (LOADADDR(.ilm))
		LONG (ADDR(.ilm))
		LONG (SIZEOF(.ilm) / 4)

		LONG (LOADADDR(.text))
		LONG (ADDR(.text))
		LONG (SIZEOF(.text) / 4)

		__copy_table_end__ = .;

		. = ALIGN(16);
	} > APP_FLASH AT>APP_FLASH

	PROVIDE( __copy_table_start = __copy_table_start__ );
	PROVIDE( __copy_table_end = __copy_table_end__ );

	.zero.table :
	{
		. = ALIGN(4);
		__zero_table_start__ = .;

		LONG (ADDR(.bss))
		LONG (SIZEOF(.bss) / 4)

		LONG (ADDR(.ilm_bss))
		LONG (SIZEOF(.ilm_bss) / 4)

		__zero_table_end__ = .;

		. = ALIGN(16);
	} > APP_FLASH AT>APP_FLASH

	.RomFunc :ALIGN(4)
	{
		*(.RomFunc)
		*(.RomFunc.*)
		. = ALIGN(16);
	} > APP_FLASH AT>APP_FLASH

	PROVIDE( __zero_table_start = __zero_table_start__ );
	PROVIDE( __zero_table_end = __zero_table_end__ );

	PROVIDE( _text_lma = LOADADDR(.text) );
	PROVIDE( _text = ADDR(.text) );
	PROVIDE (_etext = .);
	PROVIDE (__etext = .);
	PROVIDE (etext = .);

	.earlyIlm :ALIGN(4)
	{
		*(.earlyIlmFunc)
		*(.earlyIlmFunc.*)
		. = ALIGN(16);
	} > ILM AT>APP_FLASH

	PROVIDE( __Early_ILM_Func_VMA = ADDR(.earlyIlm) );
	PROVIDE( __Early_ILM_Func_LMA = LOADADDR(.earlyIlm) );
	PROVIDE( __Early_ILM_Func_SIZE = SIZEOF(.earlyIlm) );

	.ilm :ALIGN(4)
	{
		*(.RamFunc)
		*(.RamFunc.*)
		*(.IlmFunc)
		*(.IlmFunc.*)
		*(.IlmData)
		*(.IlmData.*)
		. = ALIGN(16);
	} >ILM AT>APP_FLASH

	.ilm_bss (NOLOAD) : ALIGN(8)
	{
		*(.ilmbss)
		*(.ilmbss.*)
		. = ALIGN(4);
	} >ILM AT>ILM

	PROVIDE( _ilm_lma = LOADADDR(.ilm) );
	PROVIDE( _ilm = ADDR(.ilm) );
	PROVIDE( _eilm = . );
	PROVIDE( eilm = . );

	.data : ALIGN(8)
	{
		KEEP(*(.data.ctest*))
		*(.data .data.*)
		*(.gnu.linkonce.d.*)
		. = ALIGN(8);
		PROVIDE( __gp_rel16$ = . + 0x8000 );
		PROVIDE( __global_pointer$ = . + 0x800 );
		*(.sdata .sdata.* .sdata*)
		*(.gnu.linkonce.s.*)
		/* readonly data placed in RAM for access speed */
		. = ALIGN(8);
		*(.srodata.cst16)
		*(.srodata.cst8)
		*(.srodata.cst4)
		*(.srodata.cst2)
		*(.srodata .srodata.*)
		*(.rdata)
		*(.rodata .rodata.*)
		*(.gnu.linkonce.r.*)
		/* below sections are used for rt-thread */
		. = ALIGN(4);
		__rt_init_start = .;
		KEEP(*(SORT(.rti_fn*)))
		__rt_init_end = .;
		. = ALIGN(4);
		__fsymtab_start = .;
		KEEP(*(FSymTab))
		__fsymtab_end = .;
		. = ALIGN(4);
		__vsymtab_start = .;
		KEEP(*(VSymTab))
		__vsymtab_end = .;
		. = ALIGN(16);
	} >DLM AT>APP_FLASH

	.tdata : ALIGN(8)
	{
		PROVIDE( __tls_base = . );
		*(.tdata .tdata.* .gnu.linkonce.td.*)
		. = ALIGN(16);
	} >DLM AT>APP_FLASH

	.bss (NOLOAD) : ALIGN(8)
	{
		__bss_start__ = .;
		*(.sbss*)
		*(.gnu.linkonce.sb.*)
		*(.bss .bss.*)
		*(.gnu.linkonce.b.*)
		*(COMMON)
		*(.boot_bss_section)
		. = ALIGN(4);
		__bss_end__ = .;
	} >DLM AT>DLM

	/* Nuclei C Runtime Library requirements:
	* 1. heap need to be align at 16 bytes
	* 2. __heap_start and __heap_end symbol need to be defined
	* 3. reserved at least __HEAP_SIZE space for heap
	*/
	.heap (NOLOAD) : ALIGN(16)
	{
		. = ALIGN(16);
		PROVIDE( __heap_start = . );
		. += __HEAP_SIZE;
		. = ALIGN(16);
		PROVIDE( __heap_limit = . );
	} >HEAP AT>HEAP

	.stack ORIGIN(STACK) (NOLOAD) :
	{
		. = ALIGN(16);
		PROVIDE( _heap_end = . );
		PROVIDE( __heap_end = . );
		PROVIDE( __StackLimit = . );
		PROVIDE( __StackBottom = . );
		. += __TOT_STACK_SIZE;
		. = ALIGN(16);
		PROVIDE( __StackTop = . );
		PROVIDE( _sp = . );
	} >STACK AT>STACK

	.comment 0 : { *(.comment) }
	.debug_abbrev 0 : { *(.debug_abbrev) }
	.debug_aranges 0 : { *(.debug_aranges) }  
	.debug_frame 0 : { *(.debug_frame) }
	.debug_info 0 : { *(.debug_info) }
	.debug_line 0 : { *(.debug_line) }
	.debug_line_str 0 : { *(.debug_line_str) }
	.debug_loc 0 : { *(.debug_loc) }
	.debug_loclists 0 : { *(.debug_loclists) }
	.debug_macro 0 : { *(.debug_macro) }
	.debug_pubnames 0 : { *(.debug_pubnames) }
	.debug_pubtypes 0 : { *(.debug_pubtypes) }
	.debug_ranges 0 : { *(.debug_ranges) }
	.debug_rnglists 0 : { *(.debug_rnglists) }
	.debug_str 0 : { *(.debug_str) }
	.debug_str_offsets 0 : { *(.debug_str_offsets) }
	.riscv.attributes 0 : { *(.riscv.attributes) }
}
