/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

OUTPUT_ARCH( "riscv" )
PROVIDE(__STACK_SIZE = 2K);
PROVIDE(__HEAP_SIZE = 1K);
PROVIDE(__SMP_CPU_CNT = 1);
PROVIDE(__DATA_SEC_SIZE = 16K);

ENTRY( __boot_start )

MEMORY
{
	boot_flash (rxa!w)		: ORIGIN = 0x08000000, LENGTH = 8K
	flash_driver (rxa!w)	: ORIGIN = 0x08002000, LENGTH = 8K
	ilm (wxar)				: ORIGIN = 0x20100000, LENGTH = 64K
	dlm (wxar)				: ORIGIN = 0x20120000, LENGTH = 16K - __STACK_SIZE - __HEAP_SIZE
	dlm_heap (wxar)			: ORIGIN = 0x20120000 + __DATA_SEC_SIZE - __HEAP_SIZE - __STACK_SIZE, LENGTH = __HEAP_SIZE
	dlm_stk (wxar)			: ORIGIN = 0x20120000 + __DATA_SEC_SIZE - __STACK_SIZE, LENGTH = __STACK_SIZE
	gsram (wxar)			: ORIGIN = 0x20038000, LENGTH = 16K
}

REGION_ALIAS("BOOT_FLASH", boot_flash)
REGION_ALIAS("FLASH_DRIVER", flash_driver)
REGION_ALIAS("ILM", ilm)
REGION_ALIAS("DLM", dlm)
REGION_ALIAS("GSRAM", gsram)
REGION_ALIAS("HEAP", dlm_heap)
REGION_ALIAS("STACK", dlm_stk)

SECTIONS
{
	/* To provide symbol __STACK_SIZE, __HEAP_SIZE and __SMP_CPU_CNT */
	/* PROVIDE(__STACK_SIZE = 2K); */
	__TOT_STACK_SIZE = __STACK_SIZE * __SMP_CPU_CNT;

	.rela.boot_start_init : { *(.rela.boot_start_init*) }

	.boot_init :
	{
		*(.boot_start_init)
		KEEP (*(.boot_start_init))
		. = ALIGN(16);
	} >BOOT_FLASH AT>BOOT_FLASH

	.boot_text :
	{
		. = ALIGN(4);
		*(.boot_text_section)
		. = ALIGN(16);
	} >BOOT_FLASH AT>BOOT_FLASH

	.boot_vector (NOLOAD) :
	{
		. = ALIGN(4);
		__boot_vector_remap_start__ = .;
		. += 260*4;
		__boot_vector_remap_end__ = .;
	} >GSRAM AT>GSRAM

	.boot_data :
	{
		. = ALIGN(4);
		*(.boot_data_section)
		PROVIDE( __boot_tls_base = . );
		. = ALIGN(16);
	} >GSRAM AT>BOOT_FLASH

	.boot_bss (NOLOAD):
	{
		. = ALIGN(4);
		*(.boot_bss_section)
	} >GSRAM AT>GSRAM

	.boot_rom_to_ram :
	{
		. = ALIGN(4);
		*(.boot_rom_to_ram_section)
		. = ALIGN(16);
	} >GSRAM AT>BOOT_FLASH

	.boot_flash_driver :
	{
		KEEP (*(.boot_flash_driver_info_sec))
		KEEP (*(.boot_flash_sec .boot_flash_sec.*))
	} >FLASH_DRIVER AT>FLASH_DRIVER

	.boot_copy_table :
	{
		. = ALIGN(4);
		__boot_copy_table_start__ = .;

		LONG (LOADADDR(.boot_data))
		LONG (ADDR(.boot_data))
		LONG (SIZEOF(.boot_data) / 4)

		LONG (LOADADDR(.boot_rom_to_ram))
		LONG (ADDR(.boot_rom_to_ram))
		LONG (SIZEOF(.boot_rom_to_ram) / 4)

		__boot_copy_table_end__ = .;
		. = ALIGN(16);
	} > BOOT_FLASH AT>BOOT_FLASH

	PROVIDE( __boot_copy_table_start = __boot_copy_table_start__ );
	PROVIDE( __boot_copy_table_end = __boot_copy_table_end__ );

	.boot_zero_table :
	{
		. = ALIGN(4);
		__boot_zero_table_start__ = .;

		LONG (ADDR(.boot_bss))
		LONG (SIZEOF(.boot_bss) / 4)

		_boot_zero_table_end__ = .;

		. = ALIGN(16);
	} > BOOT_FLASH AT>BOOT_FLASH

	PROVIDE( __boot_zero_table_start = __boot_zero_table_start__ );
	PROVIDE( __boot_zero_table_end = _boot_zero_table_end__ );

	/* Nuclei C Runtime Library requirements:
	* 1. heap need to be align at 16 bytes
	* 2. __heap_start and __heap_end symbol need to be defined
	* 3. reserved at least __HEAP_SIZE space for heap
	*/
	.heap (NOLOAD) : ALIGN(16)
	{
		. = ALIGN(16);
		PROVIDE( __heap_start = . );
		. += __HEAP_SIZE;
		. = ALIGN(16);
		PROVIDE( __heap_limit = . );
	} >HEAP AT>HEAP

	.stack ORIGIN(STACK) (NOLOAD) :
	{
		. = ALIGN(16);
		PROVIDE( _heap_end = . );
		PROVIDE( __heap_end = . );
		PROVIDE( __StackLimit = . );
		PROVIDE( __StackBottom = . );
		. += __TOT_STACK_SIZE;
		. = ALIGN(16);
		PROVIDE( __StackTop = . );
		PROVIDE( _sp = . );
	} >STACK AT>STACK

	.comment 0 : { *(.comment) }
	.debug_abbrev 0 : { *(.debug_abbrev) }
	.debug_aranges 0 : { *(.debug_aranges) }  
	.debug_frame 0 : { *(.debug_frame) }
	.debug_info 0 : { *(.debug_info) }
	.debug_line 0 : { *(.debug_line) }
	.debug_line_str 0 : { *(.debug_line_str) }
	.debug_loc 0 : { *(.debug_loc) }
	.debug_loclists 0 : { *(.debug_loclists) }
	.debug_macro 0 : { *(.debug_macro) }
	.debug_pubnames 0 : { *(.debug_pubnames) }
	.debug_pubtypes 0 : { *(.debug_pubtypes) }
	.debug_ranges 0 : { *(.debug_ranges) }
	.debug_rnglists 0 : { *(.debug_rnglists) }
	.debug_str 0 : { *(.debug_str) }
	.debug_str_offsets 0 : { *(.debug_str_offsets) }
	.riscv.attributes 0 : { *(.riscv.attributes) }
}
