/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "boot_interrupt.h"

void boot_interrupt_init(void)
{
	Core_IRegion_Info_t iregion = boot_get_core_iregion_info();

	Eclic_Reg_t *eclic_base = (Eclic_Reg_t *)iregion.eclic_base;

	boot_eclic_set_machine_mode(eclic_base, 0);

	boot_eclic_set_cfg_nl_bits(eclic_base, boot_eclic_get_info_ctl_bits(eclic_base));

	/* init all irq type to be ECLIC_POSTIVE_EDGE_TRIGGER */
	for (uint32_t IRQn = BOOT_SOC_EXTERNAL_MAP_TO_ECLIC_IRQn_OFFSET; IRQn < BOOT_SOC_INT_MAX; IRQn += 1) {
		boot_eclic_set_trigger_irq(eclic_base, IRQn, BOOT_ECLIC_POSTIVE_EDGE_TRIGGER);
		boot_eclic_disable_irq(eclic_base, IRQn);
	}
}

void boot_interrupt_set_vector(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uintptr_t vector)
{
	volatile unsigned long vec_base;
	vec_base = ((unsigned long)__RV_CSR_READ(CSR_MTVT));
	vec_base += ((unsigned long)IRQn) * sizeof(unsigned long);
	(* (unsigned long *) vec_base) = vector;

	boot_machine_mode_flush_DCache_line((unsigned long)vec_base);

	boot_machine_mode_invalid_ICache_line((unsigned long)vec_base);
}

void boot_interrupt_register(BOOT_IRQn_t IRQn, BOOT_ECLIC_TRIGGER_t trigger_type, uint8_t level, uint8_t priority, boot_irq_handler_t handler)
{
	Core_IRegion_Info_t iregion = boot_get_core_iregion_info();

	Eclic_Reg_t *eclic_base = (Eclic_Reg_t *)iregion.eclic_base;

	/* set interrupt vector mode */
#if USING_VECTOR_INTERRUPT != 0
	boot_eclic_set_ShvIRQ(eclic_base, IRQn, BOOT_ECLIC_VECTOR_INTERRUPT);
#else
	boot_eclic_set_ShvIRQ(eclic_base, IRQn, BOOT_ECLIC_NON_VECTOR_INTERRUPT);
#endif

	/* set interrupt trigger mode and polarity */
	boot_eclic_set_trigger_irq(eclic_base, IRQn, trigger_type);

	/* set interrupt level */
	boot_eclic_set_level_irq(eclic_base, IRQn, level);

	/* set interrupt priority */
	boot_eclic_set_priority_irq(eclic_base, IRQn, priority);

	if (handler != NULL) {
		/* set interrupt handler entry to vector table */
		boot_interrupt_set_vector(eclic_base, IRQn, (uintptr_t)handler);
	}
}

void boot_interrupt_unregister(BOOT_IRQn_t IRQn)
 {
	Core_IRegion_Info_t iregion = boot_get_core_iregion_info();

	Eclic_Reg_t *eclic_base = (Eclic_Reg_t *)iregion.eclic_base;

	boot_eclic_disable_irq(eclic_base, IRQn);
	boot_interrupt_set_vector(eclic_base, IRQn, (uintptr_t)NULL);
 }

void boot_interrupt_enable(BOOT_IRQn_t IRQn)
{
	Core_IRegion_Info_t iregion = boot_get_core_iregion_info();

	Eclic_Reg_t *eclic_base = (Eclic_Reg_t *)iregion.eclic_base;

	boot_eclic_enable_irq(eclic_base, IRQn);
}

void boot_interrupt_disable(BOOT_IRQn_t IRQn)
{
	Core_IRegion_Info_t iregion = boot_get_core_iregion_info();

	Eclic_Reg_t *eclic_base = (Eclic_Reg_t *)iregion.eclic_base;

	boot_eclic_disable_irq(eclic_base, IRQn);
}

void boot_enable_irq(void)
{
	__RV_CSR_SET(CSR_MSTATUS, MSTATUS_MIE);
}

void boot_disable_irq(void)
{
	__RV_CSR_CLEAR(CSR_MSTATUS, MSTATUS_MIE);
}
